/* ***************************************************************************************
* FILE:          BlurShaderDynamic.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurShaderDynamic is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "BlurShaderDynamic.h"
#include <FeatStd/Util/Guid.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurShaderDynamic.cpp.trc.h"
#endif // ifdef VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace widget {
namespace blur {

using namespace Candera;


BlurShaderDynamic::BlurShaderDynamic(bool initHorizontal, bool initVertical) :
   m_uniformBlurDirectionDelta("u_BlurDirectionDelta"),
   m_uniformIncrementalGauss("u_IncrementalGaussian"),
   m_vertShaderSource(),
   m_fragShaderSource(),
   m_shader(),
   m_shaderParams(),
   m_horizontal(initHorizontal),
   m_vertical(initVertical)
{
   m_guidStringBase = "81D49BDC-052A-45CB-8C36-B148CD1B5954";
}


SharedPointerShaderType BlurShaderDynamic::GetShader()
{
   if (m_shader.PointsToNull() == true)
   {
      m_shader = Candera::Shader::Create();

      m_vertShaderSource = GenerateVertexShader();

      if (m_horizontal && m_vertical)
      {
         ETG_TRACE_ERR(("Dynamic Blur Shader: The dynamic shader does not support One-Pass Blur!"));
      }
      else
      {
         m_fragShaderSource = GenerateFragmentShader();
      }

      if (!m_shader->SetVertexShader(m_vertShaderSource.c_str(), 0))
      {
         ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Vertex Shader Source failed!"));
      }

      if (!m_shader->SetFragmentShader(m_fragShaderSource.c_str(), 0))
      {
         ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Fragment Shader Source failed!"));
      }
   }

   if ((m_shader->GetVertexShader() == 0) || (m_shader->GetFragmentShader() == 0))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Shader Source failed! Vertex NULL: %p, Fragment NULL: %p", m_shader->GetVertexShader(), m_shader->GetFragmentShader()));
   }

   // Set name and GUID to allow chaching of the compiled shader.
   m_shader->SetName("BlurShaderDynamic");
   UpdateGuidString();
   m_shader->SetGuid(MakeGuid());

   return m_shader;
}


SharedPointerParamSetterType BlurShaderDynamic::GetShaderParams()
{
   Candera::Float mypi = atan(1.0F) * 4.0F;

   // Avoid div by 0
   if (m_sigma != 0.0)
   {
      m_incrementalGaussian[0] = 1.0F / (sqrt(2.0F * mypi) * m_sigma);
   }
   else
   {
      m_incrementalGaussian[0] = 1.0F / (sqrt(2.0F * mypi) * 0.000001F);
      ETG_TRACE_ERR(("Dynamic Blur Shader: Radius to sigma must not be zero. Setting it hard to 0.000001"));
   }

   m_incrementalGaussian[1] = exp(-0.5F / (m_sigma * m_sigma));
   m_incrementalGaussian[2] = m_incrementalGaussian[1] * m_incrementalGaussian[1];

   if (m_horizontal && m_vertical)
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: The dynamic shader does not support One-Pass Blur!"));
   }
   else
   {
      if (m_horizontal)
      {
         if (m_textureWidth != 0)
         {
            m_blurDirectionDelta[0] = 1.0F / static_cast<Candera::Float>(m_textureWidth);
         }
         else
         {
            m_blurDirectionDelta[0] = 1.0F / static_cast<Candera::Float>(0.000001);
            ETG_TRACE_ERR(("Dynamic Blur Shader: Texture Width must not be zero. Setting it hard to 0.000001"));
         }

         m_blurDirectionDelta[1] = 0.0;
      }
      else
      {
         m_blurDirectionDelta[0] = 0.0;

         if (m_textureHeight != 0)
         {
            m_blurDirectionDelta[1] = 1.0F / static_cast<Candera::Float>(m_textureHeight);
         }
         else
         {
            m_blurDirectionDelta[1] = 1.0F / static_cast<Candera::Float>(0.000001);
            ETG_TRACE_ERR(("Dynamic Blur Shader: Texture Height must not be zero. Setting it hard to 0.000001"));
         }
      }
   }

   if (m_shaderParams.PointsToNull() == true)
   {
      m_shaderParams = Candera::GenericShaderParamSetter::Create();
   }

   if (!m_shaderParams->SetUniform(m_uniformBitmapMaskPosition.c_str(), Shader::FloatVec4, &m_bitmapMaskPosition[0], 1))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Bitmap Mask Position uniform failed!"));
   }

   if (!m_shaderParams->SetUniform(m_uniformDarken.c_str(), Shader::FloatVec4, &m_darken[0], 1))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Darken uniform failed!"));
   }

   if (!m_shaderParams->SetUniform(m_uniformBlurRadius.c_str(), Shader::/*UniformType::*/Integer, static_cast<Int32*>(&m_radius), 1))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Blur Radius uniform failed!"));
   }

   if (!m_shaderParams->SetUniform(m_uniformBlurDirectionDelta.c_str(), Shader::/*UniformType::*/FloatVec2, m_blurDirectionDelta, 1))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Blur Direction Delta uniform failed!"));
   }

   if (!m_shaderParams->SetUniform(m_uniformIncrementalGauss.c_str(), Shader::/*UniformType::*/FloatVec3, m_incrementalGaussian, 1))
   {
      ETG_TRACE_ERR(("Dynamic Blur Shader: Setting the Incremental Gauss uniform failed!"));
   }

   Candera::GenericShaderParamSetter* gsps = static_cast<Candera::GenericShaderParamSetter*>(m_shaderParams.GetPointerToSharedInstance());
   if (NULL != gsps)
   {
      gsps->SetTextureActivationEnabled(true);
   }

   return m_shaderParams;
}


bool BlurShaderDynamic::HasValidShader()
{
   return false;
}


bool BlurShaderDynamic::HasValidParams()
{
   return false;
}


std::string BlurShaderDynamic::GenerateVertexShader() const
{
   std::string result = "uniform mat4 u_MVPMatrix;\n";
   result += "attribute vec4 a_Position;\n";
   result += "attribute vec2 a_TextureCoordinate;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "void main(void)\n";
   result += "{\n";
   result += "gl_Position = u_MVPMatrix * a_Position;\n";
   result += m_varyingTexCoord + " = a_TextureCoordinate;\n";
   result += "}\n";

   return result;
}


std::string BlurShaderDynamic::GenerateFragmentShader() const
{
   std::string result = "precision mediump int;\n";
   result += "precision mediump float;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "uniform sampler2D " + m_uniformTexSampler + ";\n";
   result += "uniform int " + m_uniformBlurRadius + "; \n";
   result += "uniform vec2 " + m_uniformBlurDirectionDelta + "; \n";
   result += "uniform vec4 " + m_uniformDarken + ";\n";
   result += "uniform vec3 " + m_uniformIncrementalGauss + "; \n";
   result += "uniform highp vec4 " + m_uniformBitmapMaskPosition + "; \n";

   /* Add texture units in case a mask exist.*/
   result += GenerateMaskUnformCode();

   result += "void main ()\n";
   result += "{\n";

   /* Add code in case a mask exist.*/
   result += GenerateMaskDecisionCode();

   result += "lowp vec4 " + m_blurAccumulator + "; \n";
   result += "vec3 incGauss;\n";
   result += "float coefficientSum;\n";
   result += m_blurAccumulator + " = (texture2D (" + m_uniformTexSampler + ", " + m_varyingTexCoord + ") * " + m_uniformIncrementalGauss + ".x); \n";
   result += "coefficientSum = " + m_uniformIncrementalGauss + ".x; \n";
   result += "incGauss.xy = (" + m_uniformIncrementalGauss + ".xy * " + m_uniformIncrementalGauss + ".yz); \n";
   result += "incGauss.z = " + m_uniformIncrementalGauss + ".z; \n";
   result += "for (int radius = 1; radius <= " + m_uniformBlurRadius + "; radius++) {\n";
   result += "vec2 tempDeltaTexCoord;\n";
   result += "tempDeltaTexCoord = (float(radius) * " + m_uniformBlurDirectionDelta + "); \n";
   result += "mediump vec2 texCoordLeft;\n";
   result += "texCoordLeft = (" + m_varyingTexCoord + " - tempDeltaTexCoord); \n";
   result += "mediump vec2 texCoordRight;\n";
   result += "texCoordRight = (" + m_varyingTexCoord + " + tempDeltaTexCoord); \n";
   result += m_blurAccumulator + " = ((" + m_blurAccumulator + " + (texture2D(" + m_uniformTexSampler + ", texCoordLeft) * incGauss.x)) + (texture2D(" + m_uniformTexSampler + ", texCoordRight) * incGauss.x)); \n";
   result += "coefficientSum = (coefficientSum + (2.0 * incGauss.x));\n";
   result += "incGauss.xy = (incGauss.xy * incGauss.yz);\n";
   result += "}\n";
   result += m_blurAccumulator + " = ((" + m_blurAccumulator + " / coefficientSum) * " + m_uniformDarken + "); \n";

   std::string fragColorCode = GenerateMaskFragColorCode();

   if (fragColorCode.empty())
   {
      result += "gl_FragColor = " + m_blurAccumulator + "; \n";
   }
   else
   {
      result += fragColorCode;
   }
   result += "}\n";
   return result;
}


void BlurShaderDynamic::UpdateGuidString()
{
   BlurShaderBase::UpdateGuidString();

   std::stringstream target;
   //make a string out of all the members
   target << m_guidString << "/BlurShaderDynamic:" << m_uniformBlurDirectionDelta << ":" << m_uniformIncrementalGauss;
   m_guidString += target.str();
}


}
}


}   /* namespace */
