/* ***************************************************************************************
* FILE:          BlurShaderBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurShaderBase is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "BlurShaderBase.h"

#if defined(FEATSTD_PLATFORM_OS_Posix)
#include <openssl/md5.h>
#elif defined(FEATSTD_PLATFORM_OS_Win32)
#include <windows.h>
#include <Wincrypt.h>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurShaderBase.cpp.trc.h"
#endif // ifdef VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace widget {
namespace blur {

using namespace Candera;


BlurShaderBase::BlurShaderBase()
   : m_textureWidth(-1)
   , m_textureHeight(-1)
   , m_sigma(0.0)
   , m_radius(0)
   , m_varyingTexCoord("v_TexCoord")
   , m_uniformTexSampler("u_Texture")
   , m_uniformDarken("u_Darken")
   , m_uniformBlurRadius("u_BlurRadius")
   , m_uniformBitmapMaskPosition("u_BitmapMaskPosition")
   , m_blurAccumulator("accum")
   , m_maskTextureUnit()
   , m_sourceTextureUnit()
   , m_combineWithMask(false),
     m_keepFrameBuffer(true),
     m_guidStringBase("00000000-0000-0000-0000-000000000000")
{
   m_darken[0] = 1.0;
   m_darken[1] = 1.0;
   m_darken[2] = 1.0;
   m_darken[3] = 1.0;

   m_bitmapMaskPosition[0] = 0.0;
   m_bitmapMaskPosition[1] = 0.0;
   m_bitmapMaskPosition[2] = 1.0;
   m_bitmapMaskPosition[3] = 1.0;
}


void BlurShaderBase::SetDarken(Candera::Float inDarken[4])
{
   if (NULL != inDarken)
   {
      m_darken[0] = inDarken[0];
      m_darken[1] = inDarken[1];
      m_darken[2] = inDarken[2];
      m_darken[3] = inDarken[3];
   }
}


void BlurShaderBase::SetBlurMaskRectangle(Candera::Rectangle inRect, bool inKeepFrameBuffer, bool inMaskEnabled)
{
   if (inMaskEnabled == true)
   {
      m_keepFrameBuffer = inKeepFrameBuffer;
      if ((inRect.GetHeight() > 0) &&
            (inRect.GetWidth() > 0) &&
            /*(inRect.GetLeft() >= 0) &&
            (inRect.GetTop() >= 0) &&*/
            (m_textureHeight > 0) &&
            (m_textureWidth > 0))
      {
         m_bitmapMaskPosition[0] = inRect.GetLeft() / m_textureWidth;
         m_bitmapMaskPosition[1] = 1.0F - ((inRect.GetTop() + inRect.GetHeight()) / m_textureHeight);
         m_bitmapMaskPosition[2] = 1.0F / (inRect.GetWidth() / m_textureWidth);
         m_bitmapMaskPosition[3] = 1.0F / (inRect.GetHeight() / m_textureHeight);
      } // if either height or width is not setup, default to fullscreen blur
      else if ((inRect.GetHeight() <= 0) ||
               (inRect.GetWidth() <= 0))
      {
         m_bitmapMaskPosition[0] = 0.0;
         m_bitmapMaskPosition[1] = 0.0;
         m_bitmapMaskPosition[2] = 1.0;
         m_bitmapMaskPosition[3] = 1.0;

         ETG_TRACE_ERR(("Blur Shader Base: Width or Height of Bitmap Mask equals 0. Defaulting to fullscreen mask."));
      }
   }
   else
   {
      // Mask disabled -> setting sensible values
      m_bitmapMaskPosition[0] = 0.0;
      m_bitmapMaskPosition[1] = 0.0;
      m_bitmapMaskPosition[2] = 1.0;
      m_bitmapMaskPosition[3] = 1.0;
   }
}


/* Set to negative value to remove */
void BlurShaderBase::SetMaskTextureUnit(Candera::Int32 maskUnit, bool inCombineWithMask)
{
   m_combineWithMask = inCombineWithMask;

   if ((maskUnit > 0) && (maskUnit < 8))
   {
      Candera::Char buffer[2];

      if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%d", maskUnit))
      {
         ETG_TRACE_ERR(("Blur Shader Base: Writing the mask texture unit index into char array failed!"));
      }
      else
      {
         m_maskTextureUnit = m_uniformTexSampler + std::string(buffer);
      }
   }
   else
   {
      m_maskTextureUnit = "";
   }
}


/* Set to negative value to remove */
void BlurShaderBase::SetSourceTextureUnit(Candera::Int32 sourceUnit)
{
   if ((sourceUnit > 0) && (sourceUnit < 8))
   {
      Candera::Char buffer[2];

      if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%d", sourceUnit))
      {
         ETG_TRACE_ERR(("Blur Shader Base: Writing the source texture unit index into char array failed!"));
      }
      else
      {
         m_sourceTextureUnit = m_uniformTexSampler + std::string(buffer);
      }
   }
   else
   {
      m_sourceTextureUnit = "";
   }
}


std::string BlurShaderBase::GenerateMaskUnformCode() const
{
   std::string result = "";

   if (m_maskTextureUnit.empty() || m_sourceTextureUnit.empty())
   {
      return result;
   }
   result = "uniform sampler2D " + m_maskTextureUnit + ";//mask texture\n";
   result += "uniform sampler2D " + m_sourceTextureUnit + ";//source texture\n";
   return result;
}


std::string BlurShaderBase::GenerateMaskDecisionCode() const
{
   std::string result = "";

   if (m_maskTextureUnit.empty() || m_sourceTextureUnit.empty())
   {
      return result;
   }

   result = "highp vec2 maskTexCoord = vec2((" + m_varyingTexCoord + ".x - " + m_uniformBitmapMaskPosition + ".x) * " + m_uniformBitmapMaskPosition + ".z, " +
            "(" + m_varyingTexCoord + ".y - " + m_uniformBitmapMaskPosition + ".y) * " + m_uniformBitmapMaskPosition + ".w);\n";
   result += "vec4 maskColor;\n";
   result += "if ((maskTexCoord.x >= 0.0) && (maskTexCoord.x <= 1.0) && ";
   result += "(maskTexCoord.y >= 0.0) && (maskTexCoord.y <= 1.0)){ \n";
   result += "maskColor = texture2D(" + m_maskTextureUnit + ", maskTexCoord); \n";
   result += "}else{\n";
   if (m_keepFrameBuffer == false)
   {
      result += "maskColor = vec4(0.0);\n";
   }
   else
   {
      result += "maskColor = vec4(1.0);\n";
   }

   result += "}\n";

   if (m_combineWithMask)
   {
      result += "if (maskColor.a == 0.0)\n";
      result += "{\n";
      result += "gl_FragColor = texture2D(" + m_sourceTextureUnit + ", " + m_varyingTexCoord + "); \n";
      result += "} else if (maskColor.a >= 1.0){\n";
      result += "gl_FragColor = maskColor;\n";
      result += "}else{\n";
   }
   else if (m_keepFrameBuffer)
   {
      result += "if (maskColor.a >= 1.0)\n";
      result += "{\n";
      result += "gl_FragColor = texture2D(" + m_sourceTextureUnit + ", " + m_varyingTexCoord + "); \n";
      result += "}\n";
      result += "else if (maskColor.a == 0.0)\n";
      result += "{\n";
      result += "discard;\n";
      result += "}else{\n";
   }
   else
   {
      result += "if ((maskColor.a == 0.0) || (maskColor.a >= 1.0))\n";
      result += "{\n";
      result += "gl_FragColor = texture2D(" + m_sourceTextureUnit + ", " + m_varyingTexCoord + "); \n";
      result += "}else{\n";
   }
   return result;
}


std::string BlurShaderBase::GenerateMaskFragColorCode() const
{
   std::string result = "";

   if (m_maskTextureUnit.empty() || m_sourceTextureUnit.empty())
   {
      return result;
   }
   if (m_combineWithMask)
   {
      result = "gl_FragColor = mix(" + m_blurAccumulator + ", maskColor, maskColor.a); \n";
   }
   else
   {
      result = "gl_FragColor = " + m_blurAccumulator + ";\n";
   }
   result += "}\n";
   return result;
}


void BlurShaderBase::UpdateGuidString()
{
   std::stringstream target;
   //make a string out of all the members
   target << "BlurShaderBase:" << ":" << m_sigma << ":" << m_radius <<
          ":" << m_varyingTexCoord << ":" << m_uniformTexSampler << ":" << m_uniformDarken <<
          ":" << m_uniformBlurRadius << ":" << m_uniformBitmapMaskPosition <<
          ":" << m_blurAccumulator << ":" << m_maskTextureUnit << ":" << m_sourceTextureUnit <<
          ":" << m_keepFrameBuffer << ":" << m_combineWithMask;

   for (int i = 0; i < 4; ++i)
   {
      target << ":" << m_darken[i];
   }

   m_guidString = target.str();
}


static FeatStd::Char ValueToCharacter(FeatStd::UInt8 tmp)
{
   if (tmp > 9)
   {
      tmp += 'A' - 10;
   }
   else
   {
      tmp += '0';
   }
   return tmp;
}


static void MakeHumanReadable(const FeatStd::UInt8* src, FeatStd::SizeType srcSize, FeatStd::Char* dest, FeatStd::SizeType destSize)
{
   FeatStd::SizeType j = 0;
   FeatStd::SizeType i = 0;
   while (j < srcSize && i < destSize)
   {
      switch (i)
      {
         case 8:
            dest[i] = '-';
            ++i;
            break;
         case 13:
            dest[i] = '-';
            ++i;
            break;
         case 18:
            dest[i] = '-';
            ++i;
            break;
         case 23:
            dest[i] = '-';
            ++i;
            break;
         default:
            dest[i] = ValueToCharacter((src[j] & 0xF0) >> 4);
            ++i;
            dest[i] = ValueToCharacter(src[j] & 0x0F);
            ++i;
            ++j;
            break;
      }
   }
   dest[i] = 0;
}


#ifdef FEATSTD_PLATFORM_OS_Win32
FeatStd::Internal::Guid BlurShaderBase::MakeGuid()
{
   unsigned char store[16] = { 0 };

   std::string shaderText = m_guidStringBase + m_guidString;
   DWORD dwStatus = 0;

   HCRYPTPROV hProv = 0;
   HCRYPTHASH hHash = 0;

   DWORD cbHash = 16;

   if (!CryptAcquireContext(&hProv,
                            NULL,
                            NULL,
                            PROV_RSA_FULL,
                            CRYPT_VERIFYCONTEXT))
   {
      dwStatus = GetLastError();
      printf("CryptAcquireContext failed: %d\n", dwStatus);
      return FeatStd::Internal::Guid();
   }

   if (!CryptCreateHash(hProv, CALG_MD5, 0, 0, &hHash))
   {
      dwStatus = GetLastError();
      printf("CryptAcquireContext failed: %d\n", dwStatus);

      CryptReleaseContext(hProv, 0);
      return FeatStd::Internal::Guid();
   }

   if (!CryptHashData(hHash, reinterpret_cast<const BYTE*>(shaderText.c_str()), shaderText.length(), 0))
   {
      dwStatus = GetLastError();
      printf("CryptHashData failed: %d\n", dwStatus);
      CryptReleaseContext(hProv, 0);
      CryptDestroyHash(hHash);
      return FeatStd::Internal::Guid();
   }

   if (!CryptGetHashParam(hHash, HP_HASHVAL, reinterpret_cast<BYTE*>(store), &cbHash, 0))
   {
      dwStatus = GetLastError();
      printf("CryptGetHashParam failed: %d\n", dwStatus);
      CryptDestroyHash(hHash);
      CryptReleaseContext(hProv, 0);
      return FeatStd::Internal::Guid();
   }

   CryptDestroyHash(hHash);
   CryptReleaseContext(hProv, 0);

   //Make a valid version 3 (GUID)
   //xxxxxxxx-xxxx-3xxx-yxxx-xxxxxxxxxxxx, where the 3 indicates Type 3 and y is masked to 10xx.
   store[8] = store[8] & 0x3F;
   store[8] = store[8] | (0x1 << 7);
   char dest[37];
   MakeHumanReadable(store, 16, dest, 37);
   dest[14] = '3';

   return FeatStd::Internal::Guid(dest);
}


#endif

#ifdef FEATSTD_PLATFORM_OS_Posix
FeatStd::Internal::Guid BlurShaderBase::MakeGuid()
{
   unsigned char store[16] = {0};
   std::string shaderText = m_guidStringBase + m_guidString;
   if ((MD5_DIGEST_LENGTH <= 16) && (MD5((unsigned char*)shaderText.c_str(), shaderText.length(), store)))
   {
      //Make a valid version 3 (GUID)
      //xxxxxxxx-xxxx-3xxx-yxxx-xxxxxxxxxxxx, where the 3 indicates Type 3 and y is masked to 10xx.
      store[8] = store[8] & 0x3F;
      store[8] = store[8] | (0x1 << 7);
      char dest[37];
      MakeHumanReadable(store, 16, dest, 37);
      dest[14] = '3';
      return FeatStd::Internal::Guid(dest);
   }
   return FeatStd::Internal::Guid();
}


#endif
}


}
}   /* namespace */
