/* ***************************************************************************************
* FILE:          NodeSelector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  NodeSelector is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "NodeSelector.h"

using namespace Candera;

NodeSelector::SelectionMode NodeSelector::m_selectionMode = NodeSelector::GeometryLineTest;

#ifdef CANDERA_3D_ENABLED
Node* NodeSelector::SelectNode(Node* node, Camera* camera, Int16 x, Int16 y, ScopeMask sm, bool includeChildNodes)
{
   Node* nearestNode = 0;
   Float minDistance = Math::MaxFloat();

   if (camera != 0)
   {
      SelectNodeRecursive(camera, node, &nearestNode, &minDistance, x, y, sm, includeChildNodes);
   }

   return nearestNode;
}


void NodeSelector::SelectNodeRecursive(Camera* camera, Node* node, Node** nearestNode, Float* minDistance, Int16 x, Int16 y, ScopeMask sm, bool includeChildNodes)
{
   if (node->IsIntersectionTestEnabled())
   {
      Line line;
      bool result = Math3D::CalculatePickLine(*camera, static_cast<Float>(x), static_cast<Float>(y), line);
      FEATSTD_DEBUG_ASSERT(result);
      CANDERA_RELEASE_UNUSED_PARAMETER(result);
      if (includeChildNodes)
      {
         //Recursion for child nodes
         for (Node* currentChild = node->GetFirstChild(); currentChild != 0; currentChild = currentChild->GetNextSibling())
         {
            SelectNodeRecursive(camera, currentChild, nearestNode, minDistance, x, y, sm, includeChildNodes);
         }
      }

      //Handle self
      Float temp_distance = 0.0F;
      bool isLineIntersecting = false;

      switch (m_selectionMode)
      {
         case BoundingSphereLineTest:
            isLineIntersecting = node->IsLineIntersectingBoundingSphere(line, temp_distance);
            break;
         case GeometryLineTest:
            isLineIntersecting = node->IsLineIntersectingGeometry(line, temp_distance);
            break;
         case GeometryPickTest:
            isLineIntersecting = node->IsPickIntersectingGeometry(*camera, x, y, temp_distance);
            break;
         default:
            break;
      }

      //is nearest node -> set new minDistance and nearest node
      if (isLineIntersecting)
      {
         if (temp_distance < *minDistance)
         {
            *minDistance = temp_distance;
            *nearestNode = node;
            return;
         }
      }
   }
   // not nearest node
}


#endif // CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
Node2D* NodeSelector::SelectNode2D(Node2D* node, Camera2D* camera, Int16 x, Int16 y, ScopeMask sm, bool includeChildNodes)
{
   Node2D* nearestNode = 0;
   Float minDistance = Math::MaxFloat();

   if (camera != 0)
   {
      SelectNodeRecursive2D(camera, node, &nearestNode, &minDistance, x, y, sm, includeChildNodes);
   }

   return nearestNode;
}


void NodeSelector::SelectNodeRecursive2D(Camera2D* camera, Node2D* node, Node2D** nearestNode, Float* minDistance, Int16 x, Int16 y, ScopeMask sm, bool includeChildNodes)
{
   if (includeChildNodes)
   {
      //Recursion for child nodes
      for (Node2D* currentChild = node->GetFirstChild(); currentChild != 0; currentChild = currentChild->GetNextSibling())
      {
         SelectNodeRecursive2D(camera, currentChild, nearestNode, minDistance, x, y, sm, includeChildNodes);
      }
   }

   Candera::Rectangle boundingRec;
   node->GetComputedBoundingRectangle(boundingRec);
   if ((static_cast<Float>(x) >= node->GetWorldPosition().GetX()) && (static_cast<Float>(x) <= node->GetWorldPosition().GetX() + boundingRec.GetWidth()) &&
         (static_cast<Float>(y) >= node->GetWorldPosition().GetY()) && (static_cast<Float>(y) <= node->GetWorldPosition().GetY() + boundingRec.GetHeight()))
   {
      *nearestNode = node;
      return;
   }
   //Handle self
   /*
   Float temp_distance = 0;
   Vector2 point = Vector2(x, y);
   bool isLineIntersecting = false; // function node->IsPickIntersecting... is not implemented in this version of candera

   //is nearest node -> set new minDistance and nearest node
   if (isLineIntersecting){
       if (temp_distance < *minDistance){
           *minDistance = temp_distance;
           *nearestNode = node;
           return;
       }
   }
   */
   // not nearest node
   return;
}


#endif // CANDERA_2D_ENABLED
