/* ***************************************************************************************
* FILE:          LineListWidget3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LineListWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

/* ***************************************************************************************
This is a 3D widget which will draw lines in a 3D view. The inputs to this widget will
be a set of points representing the X and Y coordinates.
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaWidget/Widget.h"
#include "Courier/Version.h"

#include "Candera/Engine3D/Core/Appearance.h"
#include "Candera/Engine3D/Core/Mesh.h"
#include "Candera/Engine3D/Core/VertexBuffer.h"
#include "FeatStd/MemoryManagement/SharedPointer.h"
#include "Courier/Messaging/MessagingMsgs.h"

#include "LineListWidget3D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_GRAPHWIDGETS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/LineListWidget3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(LineListWidget3D)

using namespace Candera;
using namespace Courier;

const char* LineListWidget3D::SMOOTH_VERT_SHADER = "\
    #ifndef GL_ES\n\
        #define lowp \n\
        #define mediump \n\
        #define highp \n\
        #define precision \n\
    #endif \n\
    uniform mat4 u_MVPMatrix; \n\
    attribute vec4 a_Position; \n\
    attribute vec4 a_Color; \n\
    attribute vec2 a_TextureCoordinate; \n\
    varying mediump vec4 v_Color;    \n\
    varying mediump vec2 v_TexCoord; \n\
    void main(void) \n\
    { \n\
        gl_Position = u_MVPMatrix * a_Position; \n\
        v_TexCoord = a_TextureCoordinate; \n\
        v_Color = a_Color;  \n\
    }";

const char* LineListWidget3D::SMOOTH_FRAG_SHADER = "\
    #ifndef GL_ES \n\
        #define lowp  \n\
        #define mediump  \n\
        #define highp  \n\
        #define precision \n\
    #endif \n\
    varying mediump vec4 v_Color;    \n\
    varying mediump vec2 v_TexCoord; \n\
    \n\
    void main(void) { \n\
        mediump vec4 color = v_Color; \n\
        if (v_TexCoord[1] < 0.1) { \n\
            color[3] = v_TexCoord[1] * 10.0; \n\
        } else if (v_TexCoord[1] > 0.9) { \n\
            color[3] = (1.0 - v_TexCoord[1]) * 10.0; \n\
        } \n\
        gl_FragColor = color; \n\
    }";

/****************************************************************************
CONSTRUCTOR
****************************************************************************/
LineListWidget3D::LineListWidget3D() :
   _assetProvider(0),
   _builder(),
   _modified(true),
   _checkRenderTargetActivated(true),
   _lineColor(1.f, 1.f, 1.f, 1.f),
   _lineWidth(1.f),
   _group(NULL),
   _lineListIndex(0),
   _lineLists(),
   _appearance(NULL),
   _vertexBuffer(NULL),
   _previous_X(0),
   _previous_Y(0),
   _incUpdateCounter(0)
{
   _builder.SetBufferType(VertexGeometry::ArrayBuffer);
   _builder.SetBufferUsage(VertexGeometry::StaticWrite);
   _builder.SetMemoryPool(VertexGeometry::VideoMemory);
}


LineListWidget3D::~LineListWidget3D()
{
   _checkRenderTargetActivated = true;
   renderTargetActivate();

   _assetProvider = NULL;
   _group        = NULL;

   for (size_t i = 0; i < _lineLists.Size(); i++)
   {
      if (_lineLists[i])
      {
         _lineLists[i]->Unload();
      }
   }
   _lineLists.Clear();
}


void LineListWidget3D::Init(Candera::AssetProvider* assetProvider)
{
   _assetProvider = assetProvider;
   Node* node = Widget::GetNode();

   if (node != 0 && node->IsTypeOf(Group::GetTypeId()))
   {
      _group =  Dynamic_Cast<Group*>(node);
   }
}


/****************************************************************************
*     Function    : Update
*     Description : If there is any changes in Vertex Goemetry, then draw the lines.
****************************************************************************/
void LineListWidget3D::Update()
{
   // Update the VertexGeometry when a new vertex is added. VertexGeometry contains
   // the information of all the points to be connected by a linelist.
   if (_modified || _incUpdateCounter > 0)
   {
      if (_incUpdateCounter == 0)
      {
         renderTargetActivate();
      }

      if (_lineLists.Size() > 1)
      {
         Candera::Mesh* mesh = _lineLists[_lineListIndex - 1];  //lint !e713
         if (mesh)
         {
            if (mesh->GetVertexBuffer()->IsUploaded())
            {
               mesh->GetVertexBuffer()->Unload();
            }
            VertexGeometry* geom = _builder.GetVertexGeometry();
            mesh->GetVertexBuffer()->SetVertexGeometry(geom, VertexBuffer::VertexGeometryDisposer::Dispose);
            mesh->GetVertexBuffer()->Upload();
         }
      }
      Invalidate();
      if (_incUpdateCounter > 0)
      {
         _incUpdateCounter--;
      }
      _modified = false;
   }
}


/****************************************************************************
*     Function    : CreateLineList
*     Description : Creates a linelist to connect two points(pixels)
*     Parameters  :
*     Return      : Returns the pointer to the created LineList object
****************************************************************************/
Mesh* LineListWidget3D::createLineList()
{
   Mesh* lineList = Mesh::Create();
   if (lineList == 0)
   {
      return 0;
   }

   lineList->SetName("LineListMesh");
   _previous_X = 0;
   _previous_Y = 0;
   //End 4933
   _vertexBuffer = VertexBuffer::Create();

   if (_vertexBuffer != NULL)
   {
      _builder.SetVertexCursor();
      _builder.SetIndexCursor();
      lineList->SetVertexBuffer(_vertexBuffer);
   }

   _appearance = createAppearance();

   if (_appearance != NULL)
   {
      lineList->SetAppearance(_appearance);
   }

   return lineList;
}


/****************************************************************************
*     Function    : CreateAppearance
*     Description : Creates an Appearance to display the linelist
*     Parameters  :
*     Return      : appearance
****************************************************************************/
SharedPointer<Appearance> LineListWidget3D::createAppearance()
{
   SharedPointer<Appearance> appearance = Appearance::Create();
   if (!appearance.PointsToNull())
   {
      SharedPointer<Shader> shader = getShader();
      appearance->SetShader(shader);
      SharedPointer<RenderMode> rm = RenderMode::Create();
      if (!rm.PointsToNull())
      {
         rm->SetBlendingEnabled(true);
         rm->SetCulling(RenderMode::NoCulling);
         appearance->SetRenderMode(rm);
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D : create render node failed."));
      }
      return appearance;
   }

   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D : Appearance::Create failed."));
   return SharedPointer<Appearance>(0);
}


/****************************************************************************
*     Function    : GetShader
*     Description : Get the shader from the Asset file
*     Parameters  :
*     Return      : returns the shader that is present in the asset
****************************************************************************/
SharedPointer<Shader> LineListWidget3D::getShader()
{
   if (_shader.PointsToNull())
   {
      _shader = Shader::Create(); // GetAssetProvider()->GetShader(DEFAULT_SHADER_FOR_LINELIST);
      if (_shader.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D : shader not found in asset"));
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D: GetShader: shader loaded"));
         bool rc = _shader->SetVertexShader(SMOOTH_VERT_SHADER, 0);
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D: GetShader: vertex shader applied: [%s]", (rc ? "yes" : "no")));
         rc = _shader->SetFragmentShader(SMOOTH_FRAG_SHADER, 0);
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D: GetShader: fragment shader applied: [%s]", (rc ? "yes" : "no")));
         _shader->SetDisposedAfterUpload(false);

         FeatStd::Internal::Guid guid("C9813215-BAD0-4EC3-9BFC-23A438287E43");
         _shader->SetGuid(guid);
      }
   }
   return _shader;
}


void LineListWidget3D::addVertex(Candera::Float x, Candera::Float y, Candera::Float z)
{
#undef THIS_CODE_WITH_DEBUG_FEATURE
#ifdef THIS_CODE_WITH_DEBUG_FEATURE
   static bool dbg = false;
   static Candera::Float threshold = (dbg ? 100.0f * 100.0f : 1.0f * 1.0f);
#else
   Candera::Float threshold = (1.0f * 1.0f);
#endif
   if (_builder.GetVertexCount() == 0)
   {
      _builder.SetVertexElement(VertexGeometry::Position, 0, x, y, z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0.0f, 0.5f);
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      _builder.IncrementVertexCursor();
      _previous_X = x;
      _previous_Y = y;
      _modified = true;
   }
   else if (((_previous_X - x) * (_previous_X - x) + (_previous_Y - y) * (_previous_Y - y)) > threshold)
   {
      Candera::Float a = x - _previous_X;
      Candera::Float b = y - _previous_Y;
      Candera::Float dir = (a < 0 ? -1.0f : 1.0f);
      Candera::Float alpha = Candera::Math::ATangent(b / a);
#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      Candera::Float dh = (dbg ? 50.0f  : (_lineWidth) / 2.0f * (1.2f));
#else
      Candera::Float dh = (((_lineWidth) / 2.0f) * (1.2f));
#endif
      Candera::Float dx = Candera::Math::Sine(alpha) * dh * dir;
      Candera::Float dy = Candera::Math::Cosine(alpha) * dh * dir;
      //First line
      _builder.SetVertexElement(VertexGeometry::Position, 0, _previous_X - dx, _previous_Y + dy , z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0.0f, 1.0f);

#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      if (dbg)
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, 1.0, 0.0, 0.0, getLineColor().GetAlpha());
      }
      else
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      }
#else
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
#endif
      _builder.IncrementVertexCursor();
      //Second line
      _builder.SetVertexElement(VertexGeometry::Position, 0, _previous_X + dx, _previous_Y - dy , z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0.0f, 0.0f);

#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      if (dbg)
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, 0.0, 1.0, 0.0, getLineColor().GetAlpha());
      }
      else
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      }
#else
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
#endif
      _builder.IncrementVertexCursor();
      //Third line
      _builder.SetVertexElement(VertexGeometry::Position, 0, x - dx, y + dy , z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0.0f, 1.0f);

#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      if (dbg)
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, 0.0, 0.0, 1.0, getLineColor().GetAlpha());
      }
      else
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      }
#else
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
#endif
      _builder.IncrementVertexCursor();
      //fourth line
      _builder.SetVertexElement(VertexGeometry::Position, 0, x + dx, y - dy , z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0, 0);

#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      if (dbg)
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, 1.0, 1.0, 0.0, getLineColor().GetAlpha());
      }
      else
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      }
#else
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
#endif

      _builder.IncrementVertexCursor();
      //fifth line
      _builder.SetVertexElement(VertexGeometry::Position, 0, x, y , z);
      _builder.SetVertexElement(VertexGeometry::TextureCoordinate, 0, 0.0f, 0.5f);

#ifdef THIS_CODE_WITH_DEBUG_FEATURE
      if (dbg)
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, 1.0, 0.0, 1.0, getLineColor().GetAlpha());
      }
      else
      {
         _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
      }
#else
      _builder.SetVertexElement(VertexGeometry::Color, 0, getLineColor().GetRed(), getLineColor().GetGreen(), getLineColor().GetBlue(), getLineColor().GetAlpha());
#endif

      _builder.IncrementVertexCursor();
      const Candera::UInt32 indexCnt = 12;
      Candera::UInt32 indexOffset = _builder.GetIndexCount();
      Candera::UInt16 vc = static_cast<Candera::UInt16>(_builder.GetVertexCount() - 6);

      if (vc >= 5)
      {
         const Candera::UInt32 indexCntItm = 6;
         Candera::UInt16 indexBufferItm[indexCntItm] = { vc, static_cast<Candera::UInt16>(vc + 1), static_cast<Candera::UInt16>(vc - 2), vc, static_cast<Candera::UInt16>(vc + 2), static_cast<Candera::UInt16>(vc - 1) };
         _builder.SetIndexElementRange(indexOffset, indexCntItm, indexBufferItm);
         indexOffset = _builder.GetIndexCount();
      }

      Candera::UInt16 indexBuffer[indexCnt] = { vc, static_cast<Candera::UInt16>(vc + 3), static_cast<Candera::UInt16>(vc + 1), static_cast<Candera::UInt16>(vc + 3), vc, static_cast<Candera::UInt16>(vc + 5), static_cast<Candera::UInt16>(vc + 5), vc, static_cast<Candera::UInt16>(vc + 2), static_cast<Candera::UInt16>(vc + 2), static_cast<Candera::UInt16>(vc + 4), static_cast<Candera::UInt16>(vc + 5) };
      _builder.SetIndexElementRange(indexOffset, indexCnt, indexBuffer);
      _previous_X = x;
      _previous_Y = y;
      _modified = true;
   }
   if (_modified)
   {
      //triggerUpdate(); extern trigger via triggerUpdateFlag should be used !!
   }
}


/****************************************************************************
*     Function    : createNewLineList
*     Description : This will create a new linelist in the widget.This is called
                    each time a seperate line is to be drawn.
*     Parameters  :
*     Return      :
****************************************************************************/
void LineListWidget3D::createNewLineList()
{
   _checkRenderTargetActivated = true;
   renderTargetActivate();

   _builder.Clear();
   Mesh* lineList = createLineList();

   if (lineList != 0 && _group != 0)
   {
      if (lineList->Upload())
      {
         _lineLists.Add(lineList);
         _group->AddChild(_lineLists[_lineListIndex]);                                          //lint !e713
         _lineListIndex++;
         //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"LineListWidget3D : LineList Uploaded"));
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D : LineList Upload Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineList creation Failed"));
   }
}


/**
* Set the bool to indicate the widget to clear all the lines drawn.Removes all the Linelist
* created by the widget for drawing the lines.. Unloads the LineList from the Video Memory (VRAM).
*/
bool LineListWidget3D::clearLines()
{
   bool bLinesClearStatus = true;
   _lineListIndex = 0;

   _builder.Clear();

   _checkRenderTargetActivated = true;
   renderTargetActivate();

   for (size_t i = 0; i < _lineLists.Size(); i++)
   {
      Candera::Mesh* lineList = _lineLists[i];

      if (_group != NULL)
      {
         _group->RemoveChild(lineList);
      }

      if (lineList && lineList->GetVertexBuffer()->Unload())
      {
         bLinesClearStatus = true;
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineList Unload Failed"));
         bLinesClearStatus = false;
      }

      _modified = true;
   }

   _lineLists.Clear();
   if (_modified)
   {
      //triggerUpdate(); extern trigger via triggerUpdateFlag should be used !!
   }
   return bLinesClearStatus;
}


void LineListWidget3D::renderTargetActivate()
{
   if (_checkRenderTargetActivated)
   {
      Node* node = Widget::GetNode();
      if (node)
      {
         Scene* scene = node->GetScene();
         if (scene)
         {
            Camera* camera = dynamic_cast<Camera*>(scene->GetDescendant("Camera"));
            if (camera && camera->GetRenderTarget())
            {
               bool rc = camera->GetRenderTarget()->Activate();
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LineListWidget3D: GetRenderTarget activated successfully? [%s]", (rc ? "yes" : "no")));
               _checkRenderTargetActivated = false;
            }
         }
      }
   }
}


void LineListWidget3D::triggerUpdate() const
{
   Courier::Message* msg = COURIER_MESSAGE_NEW(TriggerComponentExecuteMsg)(static_cast<ComponentId>(ComponentType::View));
   if (msg)
   {
      msg->Post();
   }
}
