/* ***************************************************************************************
* FILE:          DirectTextureConsumer3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DirectTextureConsumer3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "DirectTextureConsumer3D.h"
#include "CanderaAssetLoader/AssetLoaderBase/AssetProvider.h"
#include "Widgets/generated/WidgetsGenericMessages.h"

using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_DIRECTTEXTURE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DirectTextureConsumer3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(DirectTextureConsumer3D)

// Widget to be reworked
#define VARIANT_S_FTR_WIDGET_DEACTIVATED

DirectTextureConsumer3D::DirectTextureConsumer3D() :
   m_directTextureImage(0),
   m_invalidated(false),
   m_camera(0),
#if !defined(CGI_GPU_SIMULATION)
   m_drmBuffer(0),
#endif
   m_drmId(0)
{
}


DirectTextureConsumer3D::~DirectTextureConsumer3D()
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Start DirectTextureConsumer3D Destructor."));
   m_camera = 0;
#if !defined(CGI_GPU_SIMULATION)
   m_drmBuffer = 0;
#endif
}


void DirectTextureConsumer3D::Init(Candera::AssetProvider* /*assetProvider*/)
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   m_invalidated = true;
   Base::Init(assetProvider);
#endif
}


void DirectTextureConsumer3D::Update()
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   if (m_invalidated)
   {
      m_invalidated = !Upload();
   }
#endif
}


bool DirectTextureConsumer3D::Upload()
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
#if !defined(CGI_GPU_SIMULATION)
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Start DirectTextureConsumer3D::Upload()..."));
   if (GetNode() == 0)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No Node set on DirectTextureConsumer3D."));
      return false;
   }

   if (m_drmId == 0)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer3D: No drm id available"));
      return false;
   }

   if (m_camera != 0 && m_camera->IsTypeOf(Candera::Camera::GetTypeId()))
   {
      Candera::Camera* camera = static_cast<Candera::Camera*>(m_camera);
      Candera::RenderTarget3D* rt = camera->GetRenderTarget();
      if (rt != 0)
      {
         //ensure correct context is loaded
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Activating render target of camera \"%s\"...", camera->GetName()));
         rt->Activate();
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No render target defined for given camera."));
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No display camera defined for direct texture widget."));
   }

   Candera::MemoryManagement::SharedPointer<Texture> texture;
   if (m_directTextureImage == 0)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Creating DirectTextureImage..."));
      m_directTextureImage = DirectTextureImage::Create();
      texture = Texture::Create();
      GetNode()->GetAppearance()->SetTexture(texture);
   }

   hmibase::gadget::DrmAccessor accessor;
   m_drmBuffer = accessor.attach(static_cast<int>(m_drmId));

   if (m_drmBuffer == 0)
   {
      ETG_TRACE_ERR(("DirectTexture Consumer: no buffer for id %d", m_drmId));
      return false;
   }
   m_directTextureImage->SetWidth(m_drmBuffer->width);
   m_directTextureImage->SetHeight(m_drmBuffer->height);
   m_directTextureImage->SetFormat(DirectTextureImage::FormatRGBA);

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Drm buffer: FB: %u, Handle: %X, Data*: %X, Width %u, Height %u, Size: %u", m_drmBuffer->fb, m_drmBuffer->handle, m_drmBuffer->data, m_drmBuffer->width, m_drmBuffer->height, m_drmBuffer->size));

   //Set logical address of allocated memory for direct texture image.
   void* logical[] = { m_drmBuffer->data, NULL, NULL, NULL };
   Candera::UInt physical[4] = { ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U) };
   m_directTextureImage->SetLogicalAddress(logical);
   m_directTextureImage->SetPhysicalAddress(physical);
   texture->SetTextureImage(m_directTextureImage);

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Starting Upload of direct texture..."));
   if ((texture.PointsToNull() == false) && (!texture->Upload()))
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Failed to upload direct texture."));
      return false;
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTexture upload successful."));
#endif
#endif
   return true;
}


void DirectTextureConsumer3D::SetDRMId(unsigned int /*id*/)
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   if (id != m_drmId)
   {
      m_drmId = id;
      m_invalidated = true;
      if (GetNode() != 0)
      {
         GetNode()->SetRenderingEnabled(id != 0);
      }
      Invalidate();
   }
#endif
}
