/* ***************************************************************************************
* FILE:          SimpleButtonWidget3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SimpleButtonWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include <Candera/Engine3D/Engine3D.h>
#include "SimpleButtonWidget3D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SimpleButtonWidget3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(SimpleButtonWidget3D);


using namespace Candera;
using namespace Candera::MemoryManagement;


SimpleButtonWidget3D::SimpleButtonWidget3D() : _invalid(true)
{
}


SimpleButtonWidget3D::~SimpleButtonWidget3D()
{
}


void SimpleButtonWidget3D::Init(AssetProvider* assetProvider)
{
   Base::Init(assetProvider);
}


void SimpleButtonWidget3D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _invalid = true;
}


void SimpleButtonWidget3D::OnNodeChanged()
{
   if (GetNode() != 0)
   {
      _invalid = true;
   }
}


void SimpleButtonWidget3D::onTouchPress()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SimpleButtonWidget3D: onTouchPress for [%s]", GetLegacyName()));
   SetPressed(true);
   triggerUpdate();
   postReactionMsg(enPress);
}


void SimpleButtonWidget3D::onTouchRelease()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SimpleButtonWidget3D: onTouchRelease for [%s]", GetLegacyName()));
   SetPressed(false);
   triggerUpdate();
   postReactionMsg(enRelease);
}


void SimpleButtonWidget3D::Update()
{
   if (_invalid)
   {
      _invalid = false;

      if (GetNode() == 0)
      {
         return;
      }

      SharedPointer<Appearance> apperance = GetNode()->GetAppearance();
      if (apperance.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SimpleButtonWidget3D: in [%s], GetAppearance should not be NULL ", GetLegacyName()));
         return;
      }

      SharedPointer<Texture> texture = apperance->GetTexture();
      if (texture.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SimpleButtonWidget3D: in [%s], Texture should not be NULL", GetLegacyName()));
         return;
      }

      if (texture->GetTextureImage() != 0)
      {
         BitmapTextureImage* textureImage = Dynamic_Cast<BitmapTextureImage*, TextureImage*>(texture->GetTextureImage().GetPointerToSharedInstance());
         if (textureImage != NULL)
         {
            textureImage->Unload();

            if (!IsEnabled())
            {
               textureImage->SetBitmap(GetDisabledBitmap());
            }
            else
            {
               if (IsPressed())
               {
                  textureImage->SetBitmap(GetPressedBitmap());
               }
               else
               {
                  textureImage->SetBitmap(GetNormalBitmap());
               }
            }

            textureImage->Upload();
         }
      }

      Invalidate();
   }
}


void SimpleButtonWidget3D::postReactionMsg(enReaction reaction)
{
   Courier::View* parentView = GetParentView();
   if (parentView != NULL)
   {
      ButtonReactionMsg* msg = COURIER_MESSAGE_NEW(ButtonReactionMsg)(parentView->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), reaction, getCurrentSessionTouchInfo().mSourceId);
      if (msg != NULL)
      {
         bool result = msg->Post();
         if (result)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ButtonWidget3D: [%19s] in view [%19s] successfully posted ButtonReactionMsg with Button Reaction - [%d] from Surface - [%u]",
                                GetLegacyName(), parentView->GetId().CStr(), reaction, getCurrentSessionTouchInfo().mSourceId));
         }
      }
   }
}
