/* ***************************************************************************************
* FILE:          BlurWidget3D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef BlurWidgetSpace_BlurWidget3D_h__
#define BlurWidgetSpace_BlurWidget3D_h__

#include <Widgets/3D/Blur/generated/BlurWidget3DBase.h>

#include "BlurConfig3D.h"
namespace hmibase {
namespace widget {
namespace blur {


struct RenderTargetSetup3D
{
   Candera::GraphicDeviceUnit* passOneGdu;
   Candera::GraphicDeviceUnit* passTwoGdu;
   IBlurShader* passOneShader;
   IBlurShader* passTwoShader;
   Candera::Scene* passOneScene;
   Candera::Scene* passTwoScene;
   OriginalRenderTarget3D* originalRenderTarget;
   WindowRect backupViewportRect;
   WindowRect backupScissorRect;
   Candera::String passOneCameraName;
   Candera::String passTwoCameraName;
   Candera::ClearMode originalClearMode;
   Candera::Int16 backupSequenceNumber;
};


typedef std::map<OriginalCamera3D*, RenderTargetSetup3D> BlurCamera3DMapType;
typedef std::vector<Candera::Camera*> Camera3DVectorType;
typedef std::vector<OriginalCamera3D*> OriginalCamera3DVectorType;

/**
 * @Brief Class for storing common BlurWidget3D properties.
 *
 */
class BlurWidget3D : public BlurWidget3DBase
{
   public:
      CGI_WIDGET_RTTI_DECLARATION(BlurWidget3D, BlurWidget3DBase);

      BlurWidget3D();
      virtual ~BlurWidget3D();

      virtual void Update();
      virtual void OnChanged(FeatStd::UInt32 propertyId);

      virtual void Finalize();

      bool IsCameraPartOfBlur(Candera::Camera* camera);
      bool IsRenderTargetPartOfBlur(Candera::RenderTarget* renderTarget);

#ifdef COURIER_ENHANCED_ENABLED
      void AddBlurPassGdus();
      /* Needs to be public to enable the view to remove the GDUs at Finalize */
      void RemoveBlurPassGdus();
      Camera3DVectorType GetBlurCameras(OriginalCamera3D* camera, Candera::Int passIndex);
      OriginalCamera3DVectorType GetOriginalCameras();
      WindowRect GetOriginalCameraViewport(OriginalCamera3D* camera);
      void SetupCourier(bool inBlurEnabled);

      // Reimplementations of base functions
      virtual void OnParentViewActivate(bool activate);
      virtual void OnParentViewRenderingEnabled(bool enable);

      bool GetUpdateAfterInactive()
      {
         bool result = m_updateAfterInactive;
         m_updateAfterInactive = false;
         return result;
      };
#endif  // #ifdef COURIER_ENHANCED_ENABLED

   protected:

      /* Begin Blur Setup Methods */
      void SetupRenderTarget(bool blurEnabled);
      void SetupCameraNode(OriginalNode3D* node, bool blurEnabled);
      void SetupRenderTargetForCamera(OriginalNode3D* child, bool blurEnabled);
      Candera::GraphicDeviceUnit* CreateOffscreenRendertarget(OriginalCamera3D const* camera);
      void SetMetaInfoInt(char* item, Candera::Int newValue, Candera::GraphicDeviceUnit* blurringGdu);
      void SetupBlurScene(bool blurEnabled);
      Candera::Scene* SetupBlurPass(Candera::RenderTarget3D* dstRenderTarget, const Candera::RenderTarget3D* srcRenderTarget, Candera::ImageSource3D* mskRenderTarget,
                                    Candera::Int width, Candera::Int height, Candera::ImageSource3D* imageSource, Candera::String cameraName, Candera::Int sequenceNumber,
                                    IBlurShader* passShaderGenerator, bool darken);
      void CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap);
      void RemoveRendertargetsFromBillboard(Candera::Node* node);
      /* End Blur Setup Methods */

      /* Begin Validation Methods */
      void ValidateBlurParameters();
      bool ValidateBlurShaders();
      bool ValidateShaderType();
      void UpdateBlurBillboardTransform();
      void InvalidateBlurWidget();
      void ActivateContext();
      /* End Validation Methods */

      bool SetupBlurShader(IBlurShader* shaderGenerator, Candera::Int newWidth, Candera::Int newHeight, bool darken, bool mask);

      /* Begin Camera Configuration */
      void SetOriginalCamera(OriginalCamera3D* camera, OriginalRenderTarget3D* renderTarget, RenderTargetSetup3D& renderTargetSetup);
      void ResetOriginalCamera(OriginalCamera3D* camera);
      void SetFinalPassCamera(OriginalCamera3D* originalCamera, Candera::Camera* finalCamera);
      void GetBlurSceneSize(OriginalRenderTarget3D const* renderTarget, OriginalCamera3D const* camera, Candera::Int& width, Candera::Int& height);
      /* End Camera Configuration */

      bool IsCameraAlreadyBlurred(OriginalCamera3D const* camera) const;
      bool IsRunningInSceneComposer() const;
      bool CheckForOriginalCameraModifications();
      Candera::RenderTarget3D* RetrieveOwnerRenderTarget(Candera::GraphicDeviceUnit* gdu) const;

   private:
      void OnDarkenPropertyChanged();
      void OnBlurMaskVariantPropertyChanged();
      void OnBillboardMaskPropertyChanged();
      void OnBlurShaderVariantPropertyChanged();
      void OnBitmapMaskPropertyChanged();
      void OnBlendOutputPropertyChanged();
      void OnMultipassPropertyChanged();

      /* Begin Widget Properties */
      bool m_originalNodeRenderingEnabled;
      Candera::Float m_darkenUniform[4];
      Candera::Float m_noDarkenUniform[4];
      Candera::Rectangle m_bitmapMaskRectangle;
      Candera::Rectangle m_billboardViewport;
      FeatStd::MemoryManagement::SharedPointer<Candera::BitmapTextureImage> m_bitmapMaskTexture;
      FeatStd::MemoryManagement::SharedPointer<Candera::RenderMode> m_outputRenderModeBlending;
      FeatStd::MemoryManagement::SharedPointer<Candera::RenderMode> m_outputRenderModeOverwrite;
      Candera::Node* m_oldBillboardMask;
      /* End Widget Properties */

      /* Begin validation properties */
      bool m_validateBlurParameters;
      bool m_validateShaderType;
      /* End validation properties */

      /* Begin members used to handle Scene Composer specific behavior */
      // This flag is used to delay the setup of the blur in scene composer by one update cycle.
      bool m_skipSetup;
      bool m_isRunningInSceneComposer;
      bool m_isOriginalCameraModifiedBySceneComposer;
      /* End members used to handle Scene Composer specific behavior */

      bool m_blurEnabledInternal;
      static const Candera::Int32 m_minRadius;

      BlurCamera3DMapType m_blurCameraMap;

      // This vector stores all blured cameras. Its purpose is to avoid
      // that a camera is blurred multiple times.
      static OriginalCamera3DVectorType s_originalCameraVector;

      const Candera::Int32 m_maskTextureUnit;
      const Candera::Int32 m_sourceTextureUnit;

#ifdef COURIER_ENHANCED_ENABLED
      // Variables for disabling the widget, when Courier has the viewscene deactivated or rendering disabled
      bool m_viewSceneActive;
      bool m_viewSceneRenderingEnabled;
      bool m_updateAfterInactive;
#endif

      CdaWidgetDef(BlurWidget3D, BlurWidget3DBase)
      CdaDescription("BlurWidget3D widget")
      CdaReadableName("Blur Widget 3D")
      CdaCategory("Image")
      CdaProperties()
      CdaPropertiesEnd()
      CdaWidgetDefEnd()
};


}   /* hmibase */
}   /* widget */
}   /* blur */

FEATSTD_LINT_NEXT_EXPRESSION(1960, "Rule 7-3-1 (global namespace) disabled for deprecated declaration.")
typedef  CANDERA_DEPRECATED_3_0_0("Use namespace hmibase::widget::blur for Blur Widget.", hmibase::widget::blur::BlurWidget3D BlurWidget3D);
#endif  // BlurWidgetSpace_BlurWidget3D_h__
