/* ***************************************************************************************
* FILE:          BaseWidget3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BaseWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "BaseWidget3D.h"
#include "Courier/Messaging/MessagingMsgs.h"


#include "Widgets/widget_etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BASEWIDGET
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BaseWidget3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(BaseWidget3D);

// Global declaration warnings, will be fixed with ExampleWidgets namespace in CGI-Studio V3.0.0
//lint -esym(1960, 7-3-1)

using namespace Candera;

BaseWidget3D::BaseWidget3D() :
   m_assetProvider(0),
   _cachedName(0),
   _cachedStringIdentifier(0)
{
}


BaseWidget3D::~BaseWidget3D()
{
   ClearController(*this);

   CleanupCachedStringIdentifier();
   m_assetProvider = 0;
}


void BaseWidget3D::Init(AssetProvider* assetFactory)
{
   Base::Init(assetFactory);
   m_assetProvider = assetFactory;
}


static void widget3DAddToLegacyName(std::string& legacyName, const StringIdentifier& stringId)
{
   //process parent first
   if (stringId.GetOwner() != NULL)
   {
      widget3DAddToLegacyName(legacyName, *(stringId.GetOwner()));
   }

   const Char* id = stringId.GetId();
   if ((id != NULL) && (strnlen(id, 2) != 0))
   {
      std::string toParse(id);
      //look for backslash from right
      size_t pos = toParse.rfind('/');
      if (pos != std::string::npos)
      {
         if (!legacyName.empty())
         {
            legacyName.append("/");
         }

         //append the name contained in the specified StringIdentifier
         legacyName.append(toParse.substr(pos + 1));
      }
   }
}


void BaseWidget3D::InitLegacyName()
{
   _legacyName.clear();

   const StringIdentifier* stringId = GetStringId();

   //if name of this widget is set => use it
   if (GetName() != NULL)
   {
#if ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3))
      //skip direct StringIdentifier because it is covered by GetName() (since CGIStudio 3.3)
      if (stringId != NULL)
      {
         stringId = stringId->GetOwner();
      }
#endif

      //append the ancestors
      if (stringId != NULL)
      {
         widget3DAddToLegacyName(_legacyName, *stringId);
      }

      if (!_legacyName.empty())
      {
         _legacyName.append("/");
      }

      //append the set name
      _legacyName.append(GetName());
   }
   //else use only the StringIdentifier
   else if (stringId != NULL)
   {
      widget3DAddToLegacyName(_legacyName, *stringId);
   }
   else
   {
      //nothing to do
   }
}


void BaseWidget3D::InitCachedStringIdentifier()
{
   CleanupCachedStringIdentifier();
   FEATSTD_DEBUG_ASSERT(NULL == _cachedStringIdentifier);

   const StringIdentifier* stringId = GetStringId();
   if (NULL != stringId)
   {
      // clone owner chain
      const StringIdentifier* owner = GetStringId();
      StringIdentifier* childClone = NULL;
      while (NULL != owner)
      {
         StringIdentifier* clone = FEATSTD_NEW(Candera::StringIdentifier);
         if (NULL == clone)
         {
            ETG_TRACE_ERR(("Memory of string identifier clone could not be allocated. Owner chain will be incomplete."));
            return;
         }

         if (NULL == _cachedStringIdentifier)
         {
            // first element
            _cachedStringIdentifier = clone;
         }

         clone->SetId(owner->GetId(), NULL); //only need the pointer

         if (NULL != childClone)
         {
            childClone->SetOwner(clone);
         }
         childClone = clone;

         owner = owner->GetOwner();
      }
   }
}


void BaseWidget3D::CleanupCachedStringIdentifier()
{
   //free cached identifier memory
   const StringIdentifier* container = _cachedStringIdentifier;
   while (NULL != container)
   {
      const StringIdentifier* owner = container->GetOwner();
      FEATSTD_DELETE(container);
      container = owner;
   }
   _cachedStringIdentifier = NULL;
}


const FeatStd::Char* BaseWidget3D::GetLegacyName()
{
   const StringIdentifier* container = GetStringId();
   const StringIdentifier* cachedStringIdentifier = _cachedStringIdentifier;
   bool isNameCacheInvalid = (_cachedName != GetName()) || (cachedStringIdentifier == 0);
   while ((!isNameCacheInvalid) && (container != NULL))
   {
      isNameCacheInvalid = (NULL == cachedStringIdentifier) || (container->GetId() != cachedStringIdentifier->GetId());

      container = container->GetOwner();
      if (cachedStringIdentifier != 0)
      {
         cachedStringIdentifier = cachedStringIdentifier->GetOwner();
      }
   }

   if (isNameCacheInvalid)
   {
      _cachedName = GetName();
      InitCachedStringIdentifier();
      InitLegacyName();
   }

   return _legacyName.c_str();
}


bool BaseWidget3D::IsEffectiveVisible() const
{
   return (GetNode() != NULL) && GetNode()->IsEffectiveRenderingEnabled();
}
