/* ***************************************************************************************
* FILE:          AnimationWidget3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "AnimationWidget3D.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/ClearMode.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/EngineBase/Animation/ElasticEaseFunction.h>
#include <FeatStd/Platform/PerfCounter.h>

using namespace Candera;
using namespace Candera::Animation;

CGI_WIDGET_RTTI_DEFINITION(AnimationWidget3D);

AnimationWidget3D::AnimationWidget3D() :
   m_animationValue(),
   m_startTime(FeatStd::Internal::PerfCounter::Now()),
   m_lastTime(FeatStd::Internal::PerfCounter::Now()),
   m_sequence(0),
   m_values(0)
{
   m_currentAnimationTime = GetDuration();
   m_interpolationStrategy = Candera::Animation::EaseInterpolationStrategy::Create();
   Animation::ElasticEaseFunction::SharedPointer backFunction = Animation::ElasticEaseFunction::Create();
   if (!backFunction.PointsToNull())
   {
      backFunction->SetExponent(2.0F);
      if (!m_interpolationStrategy.PointsToNull())
      {
         m_interpolationStrategy->SetEaseDirection(EaseInterpolationStrategy::EaseOut);
         m_interpolationStrategy->SetEaseFunction(backFunction);
      }
   }

   m_kfs = Animation::AnimationKeyframeSequence::Create();
   if (!m_kfs.PointsToNull())
   {
      m_sequence = CANDERA_NEW_ARRAY(Animation::SequenceTimeType, 2);
      m_values = CANDERA_NEW_ARRAY(Float, 8);
      m_kfs->SetKeyframes(4, 2, m_sequence, Animation::KeyframeSequence::TimeTypeDisposer::Dispose, m_values, Animation::KeyframeSequence::ValuesDisposer::Dispose);
   }
}


AnimationWidget3D::~AnimationWidget3D()
{
   m_sequence = 0;//memory is deleted by the keyframe sequence (member m_kfs)
   m_values = 0;//memory is deleted by the keyframe sequence (member m_kfs)
}


void AnimationWidget3D::Init(Candera::AssetProvider* assetFactory)
{
   Base::Init(assetFactory);
   if (GetDuration() == 0)
   {
      SetDuration(1);
   }
   Node* node = GetNode();
   if (0 != node)
   {
      switch (GetPropertyAnimation())
      {
         case enAnimationProperty3D::AnimatePosition:
         {
            m_animationValue = Vector4(node->GetPosition(), 0.0f);
            SetTargetValue(node->GetPosition());
            break;
         }
         case enAnimationProperty3D::AnimateRotation:
         {
            m_animationValue = Vector4(node->GetRotation(), 0.0f);
            SetTargetValue(node->GetRotation());
            break;
         }
         case enAnimationProperty3D::AnimateClearColor:
         {
            Camera* camera = Candera::Dynamic_Cast<Candera::Camera*>(node);
            if (0 != camera)
            {
               Candera::ClearMode clearMode = camera->GetClearMode();
               m_animationValue = Vector4(clearMode.GetClearColor().GetRed(), clearMode.GetClearColor().GetGreen(), clearMode.GetClearColor().GetBlue(), clearMode.GetClearColor().GetAlpha());
               SetTargetColor(clearMode.GetClearColor());
            }
            break;
         }
         case enAnimationProperty3D::AnimateDiffuseColor:
         {
            Appearance::SharedPointer appearance = node->GetAppearance();
            if (!appearance.PointsToNull())
            {
               Material::SharedPointer material = appearance->GetMaterial();
               if (!material.PointsToNull())
               {
                  m_animationValue = Vector4(material->GetDiffuse().GetRed(), material->GetDiffuse().GetGreen(), material->GetDiffuse().GetBlue(), material->GetDiffuse().GetAlpha());
                  SetTargetColor(material->GetDiffuse());
               }
            }
            break;
         }
         default:
            break;
      }
   }
}


void AnimationWidget3D::Update()
{
   Base::Update();
   if (!IsTargetReached())
   {
      m_lastTime = FeatStd::Internal::PerfCounter::Now();
      m_currentAnimationTime = FeatStd::Int32((m_lastTime - m_startTime) * FEATSTD_PERFCOUNTER_RESOLUTION) / 1000;

      Float m_result[8];
      if (Math::Absolute(m_currentAnimationTime) > GetDuration())
      {
         m_currentAnimationTime = GetDuration();
      }
      Animation::SequenceTimeType currentTime(m_currentAnimationTime);
      if (!m_interpolationStrategy.PointsToNull())
      {
         m_interpolationStrategy->Interpolate(m_kfs.GetPointerToSharedInstance(), currentTime, m_result);
         m_animationValue = Vector4(m_result[0], m_result[1], m_result[2], m_result[3]);
         Candera::Node* node = GetNode();
         if (0 != node)
         {
            switch (GetPropertyAnimation())
            {
               case enAnimationProperty3D::AnimatePosition:
               {
                  node->SetPosition(Vector3(m_animationValue.GetX(), m_animationValue.GetY(), m_animationValue.GetZ()));
                  break;
               }
               case enAnimationProperty3D::AnimateRotation:
               {
                  node->SetRotation(Vector3(m_animationValue.GetX(), m_animationValue.GetY(), m_animationValue.GetZ()));
                  break;
               }
               case enAnimationProperty3D::AnimateClearColor:
               {
                  Camera* camera = Candera::Dynamic_Cast<Candera::Camera*>(node);
                  if (0 != camera)
                  {
                     Candera::ClearMode clearMode = camera->GetClearMode();
                     Candera::Color color(m_animationValue.GetX(), m_animationValue.GetY(), m_animationValue.GetZ(), m_animationValue.GetW());
                     clearMode.SetClearColor(color);
                     camera->SetClearMode(clearMode);
                  }
                  break;
               }
               case enAnimationProperty3D::AnimateDiffuseColor:
               {
                  Appearance::SharedPointer appearance = node->GetAppearance();
                  if (!appearance.PointsToNull())
                  {
                     Material::SharedPointer material = appearance->GetMaterial();
                     if (!material.PointsToNull())
                     {
                        Candera::Color color(m_animationValue.GetX(), m_animationValue.GetY(), m_animationValue.GetZ(), m_animationValue.GetW());
                        material->SetDiffuse(color);
                     }
                  }
                  break;
               }
               default:
                  break;
            }
         }
         Invalidate(); // TODO: Probably Invalidate(bool forceInvalidate) has to be implemented in BaseWidget3D to force Invalidation
         WakeUpAllRenderComponents();
      }
   }
}


void AnimationWidget3D::OnChanged(::FeatStd::UInt32 propertyId)
{
   switch (propertyId)
   {
      case TargetValuePropertyId:
      {
         m_startTime = FeatStd::Internal::PerfCounter::Now();
         if (0 != m_sequence)
         {
            m_sequence[0] = 0;
            m_sequence[1] = GetDuration();
         }
         if (0 != m_values)
         {
            m_values[0] = m_animationValue.GetX();
            m_values[1] = m_animationValue.GetY();
            m_values[2] = m_animationValue.GetZ();
            m_values[3] = 0.0f;

            m_values[4] = GetTargetValue().GetX();
            m_values[5] = GetTargetValue().GetY();
            m_values[6] = GetTargetValue().GetZ();
            m_values[7] = 0.0f;
         }
         break;
      }
      case TargetColorPropertyId:
      {
         m_startTime = FeatStd::Internal::PerfCounter::Now();
         if (0 != m_sequence)
         {
            m_sequence[0] = 0;
            m_sequence[1] = GetDuration();
         }
         if (0 != m_values)
         {
            m_values[0] = m_animationValue.GetX();
            m_values[1] = m_animationValue.GetY();
            m_values[2] = m_animationValue.GetZ();
            m_values[3] = m_animationValue.GetW();

            m_values[4] = GetTargetColor().GetRed();
            m_values[5] = GetTargetColor().GetGreen();
            m_values[6] = GetTargetColor().GetBlue();
            m_values[7] = GetTargetColor().GetAlpha();
         }
         break;
      }
      default:
         break;
   }
   Base::OnChanged(propertyId);
}


bool AnimationWidget3D::IsTargetReached()
{
   if (IsVector3Selected())
   {
      if (GetTargetValue().GetX() != m_animationValue.GetX() ||
            GetTargetValue().GetY() != m_animationValue.GetY() ||
            GetTargetValue().GetZ() != m_animationValue.GetZ())
      {
         return false;
      }
   }

   if (IsColorSelected())
   {
      if (GetTargetColor().GetRed() != m_animationValue.GetX() ||
            GetTargetColor().GetGreen() != m_animationValue.GetY() ||
            GetTargetColor().GetBlue() != m_animationValue.GetZ() ||
            GetTargetColor().GetAlpha() != m_animationValue.GetW())
      {
         return false;
      }
   }

   return true;
}
