/**
* @FILE:          HandWriting_Util_Widget2D.cpp
* @author:        Sandesh Rao (sra9kor), RBEI/ECG2
* @SW-COMPONENT:  HMI-BASE
* @DESCRIPTION:   HandWriting_Util_Widget2D is part of HMI-Base Widget Library
* @COPYRIGHT:     (c) 2015 Robert Bosch Car Multimedia GmbH
*
*  This software is part of the Bosch Widget Library and property of Robert Bosch GmbH.
*  Unauthorized duplication and disclosure to third parties is prohibited.
*  All rights reserved.
*/
#include "widget2D_std_if.h"
#include "HandWriting_Util_Widget2D.h"
#include "Widgets/2Dext/HandWritingRecognition/generated/HandWriting_Util_Widget2D_Messages.h"
#include "hmibase/util/Ticker.h"

using namespace Candera;
using namespace Courier;
using namespace std;
using Util::Timer;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_HANDWRITING
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HandWriting_Util_Widget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(HandWriting_Util_Widget2D)

//using namespace std;
//using Util::Timer;

// Added for last six character feature
#define MAX_HINT_UTF8CHAR 6UL

Courier::Char HandWriting_Util_Widget2D::_defaultChar[5] = {0};
//---------------------------------------------------------------------------
HandWriting_Util_Widget2D::HandWriting_Util_Widget2D() :
   _lineListWidgetPtr(NULL),
   _hanWangInterface(NULL),
   _pointData(NULL),
   _yPointData(0),
   _charList(NULL),
   _nodePosition_X(0),
   _nodePosition_Y(0),
   _index(0),
   _worldPosition(),
   _currentLanguage(CHINESE_LANGUAGE),
   _languageRange(GB_ALC_GBK),
   _strokeEnded(true),
   _syncFlag(false),
   _processTouchPoints(true),
   _drawingAreaWidth(0.0f),
   _drawingAreaHeight(0.0f),
   _numericRange(0),
   _symbolRange(0),
   _lowerCaseLatinLetterRange(0),
   _upperCaseLatinLetterRange(0),
   _gestureRecognitionRange(0),
   _previous_X(0),
   _previous_Y(0),
   _sourcePointer(NULL),
   _accessMode(READ),
   _update(true),
   _temp2(0),
   _curStrLength(0),
   _maxTextLength(0)
{
   _pointData                    = FEATSTD_NEW_ARRAY(short, MAX_TOUCH_POINTS);
   _charList                     = FEATSTD_NEW_ARRAY(Char, MAX_CHAR_LIMITS);
   _charecterRecDictionaryPath   = CHAR_DICTIONARY_PATH_CHINESE;
   _wordRecDictionaryPath        = WORD_DICTIONARY_PATH_CHINESE;
   SetTouchable(true);
   memset(_utf8CharBuffer, 0, sizeof(_utf8CharBuffer));
   //Initialize the Array which stores the Points
   memset(_pointData, 0, MAX_TOUCH_POINTS * sizeof(short));
   _characterRecogniton_Timer.setName("HWR", "RecognitonTimer");
}


void HandWriting_Util_Widget2D::setSourcePointer(
   const Candera::MemoryManagement::SharedPointer<HWRBackendInterface>& pSourcePointer
)
{
   _syncFlag = false;
   if ((false == pSourcePointer.PointsToNull()))
   {
      _sourcePointer = pSourcePointer;
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "_sourcePointer [%d]", (false == (pSourcePointer.PointsToNull()))));
      getPredictedText();
      if (0 != GetParentView())
      {
         PropertyModified("PredictedWordsSourcePointer");
      }
   }
   _syncFlag = true;
}


const Candera::MemoryManagement::SharedPointer<HWRBackendInterface>& HandWriting_Util_Widget2D::getSourcePointer() const
{
   return _sourcePointer;
}


//---------------------------------------------------------------------------
HandWriting_Util_Widget2D::~HandWriting_Util_Widget2D()
{
   if (_pointData != 0)
   {
      FEATSTD_SAFE_DELETE_ARRAY(_pointData);
      _pointData = 0;
   }
   if (_charList != NULL)
   {
      FEATSTD_SAFE_DELETE_ARRAY(_charList);
      _charList = 0;
   }
   if (_hanWangInterface != NULL)
   {
      FEATSTD_SAFE_DELETE(_hanWangInterface);
   }
   if (!_sourcePointer.PointsToNull())
   {
      _sourcePointer.Release();
   }
   _lineListWidgetPtr         = NULL;
   _nodePosition_X            = 0;
   _nodePosition_Y            = 0;
   _strokeEnded               = false;
   _hanWangInterface          = NULL;
   _drawingAreaWidth          = 0.0f;
   _drawingAreaHeight         = 0.0f;
   _numericRange              = 0;
   _symbolRange               = 0;
   _lowerCaseLatinLetterRange = 0;
   _upperCaseLatinLetterRange = 0;
   _gestureRecognitionRange   = 0;
   _previous_X                = 0;
   _previous_Y                = 0;
   _processTouchPoints        = true;
}


void HandWriting_Util_Widget2D::setCharactersToPredictedWord(const FeatStd::String& characterstopredictedword)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "setCharactersToPredictedWord"));
   if ((utfLength(characterstopredictedword.GetCString()) <= 6))
   {
      _charactersToPredictedWord = characterstopredictedword;
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "characterstopredictedword.GetCharCount() <= 6"));
      //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"characterstopredictedword = [%x]", _charactersToPredictedWord.GetCString()));
      setSourcePointer(HWRBackendInterface::pCreate());
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "characterstopredictedword.GetCharCount() <= 6 is false"));
      //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"characterstopredictedword = [%x]", _charactersToPredictedWord.GetCString()));
      getlastHintUTF8Chars(characterstopredictedword);
      setSourcePointer(HWRBackendInterface::pCreate());
   }
}


void HandWriting_Util_Widget2D::setAccessMode(Candera::ACCESS_MODE AccessMode)
{
   _accessMode = AccessMode;
   if (true == _update)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Flag changed even before previous update completion!!!"));
   }
   _update = true;
}


Candera::ACCESS_MODE HandWriting_Util_Widget2D::getAccessMode() const
{
   return _accessMode;
}


void HandWriting_Util_Widget2D::Init(Candera::AssetProvider* assetProvider)
{
   Base::Init(assetProvider);

   _lineListWidgetPtr = Dynamic_Cast<LineListWidget3D*>(GetAddLineListWidget());
   if (_lineListWidgetPtr != NULL)
   {
      if (NULL != _lineListWidgetPtr->GetNode())
      {
         _worldPosition = _lineListWidgetPtr->GetNode()->GetWorldPosition();
         _lineListWidgetPtr->setLineWidth(GetLineWidth());
         _lineListWidgetPtr->setLineColor(GetLineColor());
      }
   }

   assignLanguage();
   _characterRecogniton_Timer.setTimeout(0, GetCharacterCompletionTime(), NULL);
   _hanWangInterface = FEATSTD_NEW(HanWangInterface)(_currentLanguage, _languageRange,
                       _charecterRecDictionaryPath.GetCString(), _wordRecDictionaryPath.GetCString(), static_cast<int>(GetPriorityMode()));
   if (_hanWangInterface != NULL)
   {
#ifdef WIN32
      _hanWangInterface->setDictionaryRootPath(hmibase::util::File::getBinFolder().c_str());
#endif
      _hanWangInterface->initialize();
   }
}


void HandWriting_Util_Widget2D::Update()
{
}


bool HandWriting_Util_Widget2D::OnMessage(const Courier::Message& msg)
{
   bool rc = Base::OnMessage(msg);
   if (!rc)
   {
      switch (msg.GetId())
      {
         case TouchMsg::ID:
         {
            const Courier::TouchMsg* touchMsg = Courier::message_cast<const Courier::TouchMsg*>(&msg);

            if (touchMsg != NULL && touchMsg->GetPointerId() == 0)
            {
               Int32 x = static_cast<Int32>(touchMsg->GetXPos() - _nodePosition_X);
               Int32 y = (static_cast<Int32>(_worldPosition.GetY() * -2) - static_cast<Int32>(touchMsg->GetYPos())) + static_cast<Int32>((_nodePosition_Y));
               _yPointData = touchMsg->GetYPos();

               if ((((x > 0) && (x < static_cast<Int32>(_drawingAreaWidth))) &&
                     ((y > 0) && (y < static_cast<Int32>(_drawingAreaHeight)))))
               {
                  _characterRecogniton_Timer.stop();
                  //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Character Recognition Timer KILLED"));

                  switch (touchMsg->GetState())
                  {
                     case Courier::TouchMsgState::Down:
                     {
                        _processTouchPoints = true;
                        handleTouchDown(static_cast<UInt32>(x), static_cast<UInt32>(y));
                        rc = true;
                     }
                     break;

                     case Courier::TouchMsgState::Move:
                     {
                        if (!_processTouchPoints)
                        {
                           clearHWRTextWidget();
                           _processTouchPoints = true;
                        }

                        handleTouchMove(static_cast<UInt32>(x), static_cast<UInt32>(y));
                        rc = true;
                     }
                     break;

                     case Courier::TouchMsgState::Up:
                     {
                        handleTouchUp();
                        _strokeEnded = true;
                        rc = true;
                     }
                     break;

                     default:
                        break;
                  }
               }
               else
               {
                  if (_processTouchPoints)
                  {
                     handleTouchUp();
                     _strokeEnded = true;
                     //ETG_TRACE_ERR((APP_TRACECLASS_ID(),"Touched Outside the HandWriting Area - <X,Y>(%d,%d)",x,y));
                     _processTouchPoints = false;
                  }
               }
            }
         }
         break;

         case TimerExpiredMsg::ID:
         {
            const TimerExpiredMsg* timerMsg = message_cast<const TimerExpiredMsg*>(&msg);
            if (timerMsg != 0)
            {
               if (
                  (timerMsg->GetTimer() == &_characterRecogniton_Timer) &&
                  (_characterRecogniton_Timer.getStatus() != Timer::Stopped)
               )
               {
                  processRecognizedCharacters();
                  // Marking the return value as consumed only when the timer id is same as reuired by Handwritting
                  rc = true;
               }
            }
         }
         break;

         default:
            break;
      } // ![switch case]
   }

   return rc;
}


void HandWriting_Util_Widget2D::handleTouchDown(UInt32 x, UInt32 y)
{
   clearHWRTextWidget();

   if (_lineListWidgetPtr != NULL)
   {
      _previous_X = x;
      _previous_Y = y;
      _lineListWidgetPtr->createNewLineList();
      _lineListWidgetPtr->addVertex((Candera::Float)x, (Candera::Float)y);
      _lineListWidgetPtr->addVertex((Candera::Float)x, (Candera::Float)y); // To draw a line to represent a point

      // Store the points
      if ((_pointData != NULL) && (_index + 1 < MAX_TOUCH_POINTS))
      {
         _pointData[_index++] = (short)x;
         _pointData[_index++] = static_cast<short>(_yPointData - _nodePosition_Y);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Points : (%d,%d)", _pointData[_index - 2], _pointData[_index - 1]));
      }
      _lineListWidgetPtr->triggerUpdateFlag();
   }
}


void HandWriting_Util_Widget2D::handleTouchMove(UInt32 x, UInt32 y)
{
   if (_lineListWidgetPtr != NULL)
   {
      if (_strokeEnded)
      {
         _lineListWidgetPtr->createNewLineList();
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "New stroke created due to touch going out of the HandWriting Area"));
      }
      _strokeEnded = false;
      //Added new variable z to store variables of X&Y also avoid square of variables for optimization
      int xdiff = static_cast<int>(_previous_X - x);
      int ydiff = static_cast<int>(_previous_Y - y);
      // vectDistance = sqrt(xdiff*xdiff + ydiff*ydiff);
      unsigned vectDistance = static_cast<unsigned int>(xdiff * xdiff + ydiff * ydiff); // Note: to prevent sqrt we take the calculates value 25, not 5 pixel, abs not necessary
      if (vectDistance >= 20)
      {
         _lineListWidgetPtr->addVertex((Candera::Float)x, (Candera::Float)y);
         // Store the points
         if ((_pointData != NULL) && (_index + 1 < MAX_TOUCH_POINTS))
         {
            _pointData[_index++] = static_cast<short>(x);
            _pointData[_index++] = static_cast<short>(_yPointData - _nodePosition_Y);
            ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Points : (%d,%d)", _pointData[_index - 2],
                                 _pointData[_index - 1]));
         }
         _previous_X = x;
         _previous_Y = y;
      }
      _lineListWidgetPtr->triggerUpdateFlag();
   }
}


void HandWriting_Util_Widget2D::handleTouchUp()
{
   // Store the points to represent end of stroke
   if ((_pointData != NULL) && (_index + 1 < MAX_TOUCH_POINTS))
   {
      _pointData[_index++] = STROKE_END_MARK;
      _pointData[_index++] = 0;
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Points : (%d,%d) for UP", _pointData[_index - 2],
                           _pointData[_index - 1]));
   }
   if (_lineListWidgetPtr != NULL)
   {
      _lineListWidgetPtr->triggerUpdateFlag();
   }
   // Start a Timer after the touch release
   _characterRecogniton_Timer.start();
}


bool HandWriting_Util_Widget2D::processRecognizedCharacters()
{
   if ((_pointData != NULL) && (_index + 1 < MAX_TOUCH_POINTS))
   {
      _pointData[_index++] = STROKE_END_MARK;
      _pointData[_index] = STROKE_END_MARK;
      // ETG_TRACE_USR1(("Points : (%d,%d) End of Character Stroke",_pointData[_index-1], _pointData[_index]));
   }
   UInt16 u16AsciiResult[22]; // The recognized characters will occupy max 40 bytes
   u16AsciiResult[21] = 0xE35A;
   memset(u16AsciiResult, 0, sizeof(u16AsciiResult));

   if (_hanWangInterface != NULL)
   {
      if (_hanWangInterface->recognizeCharacters(_pointData, u16AsciiResult) <= 0)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), " No characters recognized by Engine"));
         clearDrawingArea();
         return false;
      }
      if (u16AsciiResult[21] != 0xE35A)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), " HWR: memory corruption in processRecognizedCharacters"));
      }
      if (_charList != NULL)
      {
         memset(_charList, '\0' , MAX_CHAR_LIMITS);
      }
      Candera::UInt8 u8RowIndex = 0;
      Candera::UInt8 nCharCount = 0;

      while ((u8RowIndex < 20) && (u16AsciiResult[u8RowIndex] != 0))
      {
         memset(_utf8CharBuffer, 0, sizeof(_utf8CharBuffer));
         UInt nIndex = static_cast<UInt>(nCharCount * sizeof(_utf8CharBuffer));
         UInt16 uCurrentCharacterInUTF16 = u16AsciiResult[u8RowIndex++];
         if ((u8RowIndex < 20) && (u16AsciiResult[u8RowIndex] != 0)) // Get the score for each character from the result
         {
            u8RowIndex++;
         }
         if (0 == convertUTF16toUTF8(_utf8CharBuffer, uCurrentCharacterInUTF16))
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "HANDWRITING ENCODING FAILED"));
            clearDrawingArea();
            return false;
         }
         else
         {
            for (UInt i = 0; i < sizeof(_utf8CharBuffer); i++)
            {
               if ((_charList != NULL) && (nIndex < MAX_CHAR_LIMITS))
               {
                  _charList[nIndex + i] = _utf8CharBuffer[i];
               }
            }
            // ETG_TRACE_USR1(("Recognized Characters : char : %u", uCurrentCharacterInUTF16));
            // ETG_TRACE_USR1(("Score : = %u", uCurrentScore));
            nCharCount++;
         }
      } // while loop
      postCandidateUpdateMsg();
      postHWRTextWidgetUpdateMsg();
      clearDrawingArea();
   }
   return true;
}


bool HandWriting_Util_Widget2D::getPredictedText()
{
   int predictionResult = 0;
   unsigned short wCodel[8] = {0 };//��;
   unsigned short psResult[2048] = {0};

   // Check added to solve the reset during scene creation
   if ((_hanWangInterface != NULL) && (false == _charactersToPredictedWord.IsEmpty()))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "_charactersToPredictedWord.IsEmpty() is not empty"));
      //u8Text = (UInt8 *)(_charactersToPredictedWord.GetCString()); //lint
      // for lint fix
      FeatStd::UInt8* u8Text = static_cast<FeatStd::UInt8*>(static_cast <void*>(const_cast<TChar*>(
                                  _charactersToPredictedWord.GetCString()))
                                                           );
      memset(wCodel, 0, sizeof(wCodel));
      if (u8Text != 0)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "u8Text != 0 is ture"));
         int j = 0;
         int i = 0;
         int bytenum = convertUTF8toUTF16(u8Text, &(wCodel[0]));
         do
         {
            j++;
            i = i + bytenum;
            bytenum = convertUTF8toUTF16(u8Text + i, &(wCodel[j]));
         }
         while ((j < 7) && (wCodel[j] != 0x0));

         predictionResult = _hanWangInterface->predictWords(wCodel, psResult);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "predictionResult = [%d]", predictionResult));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Predicted Words = [%p]", psResult));
      }
      //Calling the backend interface here
      extractStrings(psResult, 2048, predictionResult);
      clearDrawingArea();
      triggerUpdate();
   }
   return false;
}


void HandWriting_Util_Widget2D::clearDrawingArea()
{
   if (isRecognitionTimerRunning())
   {
      _characterRecogniton_Timer.stop();
   }

   if (_lineListWidgetPtr != NULL)
   {
      if (_lineListWidgetPtr->clearLines())
      {
         _lineListWidgetPtr->triggerUpdateFlag(2);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Drawing Area Cleared"));
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Drawing Area NOT Cleared"));
      }
   }

   if (_pointData != NULL)
   {
      memset(_pointData, 0, MAX_TOUCH_POINTS * sizeof(short));
   }
   _index = 0; // Reset the index
}


int HandWriting_Util_Widget2D::convertUTF16toUTF8(Char* pcString, int nCharacter)
{
   int nResult = 0;
   // check if character and string are valid
   if ((nCharacter > 0) && pcString != 0)
   {
      UInt32   nChar = static_cast<UInt32>(nCharacter);
      if (nChar < 0x80)
      {
         // one-byte character (0x01 .. 0x7F)
         pcString[0] = (Char) nChar;
         nResult = 1;
      }
      else if (nChar < 0x800)
      {
         // two-byte character (0x80 .. 0x7FF)
         pcString[0] = (Char)(((nChar >>  6)) | 0xC0);
         pcString[1] = (Char)(((nChar) & 0x3F) | 0x80);
         nResult = 2;
      }
      else if (nCharacter < 0x10000)
      {
         // three-byte character (0x800 .. 0xFFFF)
         pcString[0] = (Char)(((nChar >> 12)) | 0xE0);
         pcString[1] = (Char)(((nChar >>  6) & 0x3F) | 0x80);
         pcString[2] = (Char)(((nChar) & 0x3F) | 0x80);
         nResult = 3;
      }
      else if (nCharacter < 0x110000)
      {
         // four-byte character (0x10000 .. 0x10FFFF)
         pcString[0] = (Char)(((nChar >> 18)) | 0xF0);
         pcString[1] = (Char)(((nChar >> 12) & 0x3F) | 0x80);
         pcString[2] = (Char)(((nChar >>  6) & 0x3F) | 0x80);
         pcString[3] = (Char)(((nChar) & 0x3F) | 0x80);
         nResult = 4;
      }
   }
   return nResult;
}


int HandWriting_Util_Widget2D::convertUTF8toUTF16(const UInt8* pInput, UInt16* unic)
{
   int nResult = 0;
   UInt8 u8Bytes[6];
   UInt8 u8First = pInput[0];
   *unic = 0x0;
   UInt8* pOutput = (UInt8*) unic;
   if (u8First < 0x80)
   {
      *pOutput = *pInput;
      nResult = 1;
   }
   else if (u8First < 0xc0)
   {
   }
   else if (u8First < 0xe0)
   {
      u8Bytes[0] = pInput[0];
      u8Bytes[1] = pInput[1];
      if ((u8Bytes[1] & (0xE0)) != (0x80))
      {
         return 0;
      }
      *pOutput = static_cast<UInt8>((u8Bytes[0] << 6) + (u8Bytes[1] & 0x3f));
      *(pOutput + 1) = ((u8Bytes[0] >> 2) & (0x07));
      nResult = 2;
   }
   else if (u8First < 0xf0)
   {
      u8Bytes[0] = *pInput;
      u8Bytes[1] = *(pInput + 1);
      u8Bytes[2] = *(pInput + 2);
      if (((u8Bytes[1] & (0xC0)) != 0x80) || ((u8Bytes[2] & (0xC0)) != (0x80)))
      {
         return 0;
      }
      *pOutput       = static_cast<UInt8>((u8Bytes[1] << 6) + (u8Bytes[2] & (0x3F)));
      *(pOutput + 1) = static_cast<UInt8>((u8Bytes[0] << 4) + ((u8Bytes[1] >> 2) & (0x0F)));
      nResult = 3;
   }
   else if (u8First < 0xf8)
   {
      u8Bytes[0] = *pInput;
      u8Bytes[1] = *(pInput + 1);
      u8Bytes[2] = *(pInput + 2);
      u8Bytes[3] = *(pInput + 3);
      if (((u8Bytes[1] & 0xC0) != 0x80) || ((u8Bytes[2] & 0xC0) != 0x80)
            || ((u8Bytes[3] & 0xC0) != 0x80))
      {
         return 0;
      }
      *pOutput       = static_cast<UInt8>((u8Bytes[2] << 6) + (u8Bytes[3] & 0x3F));
      *(pOutput + 1) = static_cast<UInt8>((u8Bytes[1] << 4) + ((u8Bytes[2] >> 2) & (0x0F)));
      *(pOutput + 2) = static_cast<UInt8>(((u8Bytes[0] << 2) & (0x1C)) + ((u8Bytes[1] >> 4) & (0x03)));
      nResult = 4;
   }
   else if (u8First < 0xfc)
   {
      u8Bytes[0] = *pInput;
      u8Bytes[1] = *(pInput + 1);
      u8Bytes[2] = *(pInput + 2);
      u8Bytes[3] = *(pInput + 3);
      u8Bytes[4] = *(pInput + 4);
      if (((u8Bytes[1] & 0xC0) != 0x80) || ((u8Bytes[2] & 0xC0) != 0x80)
            || ((u8Bytes[3] & 0xC0) != 0x80) || ((u8Bytes[4] & 0xC0) != 0x80))
      {
         return 0;
      }
      *pOutput       = static_cast<UInt8>((u8Bytes[3] << 6) + (u8Bytes[4] & (0x3F)));
      *(pOutput + 1) = static_cast<UInt8>((u8Bytes[2] << 4) + ((u8Bytes[3] >> 2) & (0x0F)));
      *(pOutput + 2) = static_cast<UInt8>((u8Bytes[1] << 2) + ((u8Bytes[2] >> 4) & (0x03)));
      *(pOutput + 3) = static_cast<UInt8>(u8Bytes[0] << 6);
      nResult = 5;
   }
   else
   {
      u8Bytes[0] = *pInput;
      u8Bytes[1] = *(pInput + 1);
      u8Bytes[2] = *(pInput + 2);
      u8Bytes[3] = *(pInput + 3);
      u8Bytes[4] = *(pInput + 4);
      u8Bytes[5] = *(pInput + 5);
      if (((u8Bytes[1] & 0xC0) != 0x80) || ((u8Bytes[2] & 0xC0) != 0x80)
            || ((u8Bytes[3] & 0xC0) != 0x80) || ((u8Bytes[4] & 0xC0) != 0x80)
            || ((u8Bytes[5] & 0xC0) != 0x80))
      {
         return 0;
      }
      *pOutput       = static_cast<UInt8>((u8Bytes[4] << 6) + (u8Bytes[5] & (0x3F)));
      *(pOutput + 1) = static_cast<UInt8>((u8Bytes[4] << 4) + ((u8Bytes[5] >> 2) & (0x0F)));
      *(pOutput + 2) = static_cast<UInt8>((u8Bytes[2] << 2) + ((u8Bytes[3] >> 4) & (0x03)));
      *(pOutput + 3) = static_cast<UInt8>(((u8Bytes[0] << 6) & (0x40)) + ((u8Bytes[1] & (0x3F))));
      nResult = 6;
   }
   return nResult;
}


void HandWriting_Util_Widget2D::assignLanguage()
{
   switch (GetLanguage())
   {
      case CHINESE:
      {
         _currentLanguage = CHINESE_LANGUAGE;
         if (GetIncludeNumbers())
         {
            _numericRange = GB_ALC_NUMERIC;
         }
         if (GetIncludeSymbols())
         {
            _symbolRange = GB_ALC_SYMBOL;
         }
         if (GetIncludeLatinLowerCaseLetters())
         {
            _lowerCaseLatinLetterRange = GB_ALC_LCALPHA;
         }
         if (GetIncludeLatinUpperCaseLetters())
         {
            _upperCaseLatinLetterRange = GB_ALC_UCALPHA;
         }
         if (GetIncludeGestureRecognition())
         {
            _gestureRecognitionRange = GB_ALC_GESTURE;
         }

         _languageRange = GB_ALC_GBK | _numericRange | _symbolRange | _lowerCaseLatinLetterRange | \
                          _upperCaseLatinLetterRange | _gestureRecognitionRange;

         _charecterRecDictionaryPath = CHAR_DICTIONARY_PATH_CHINESE;
         _wordRecDictionaryPath       = WORD_DICTIONARY_PATH_CHINESE;
      }
      break;

      case ENGLISH:
      {
         _currentLanguage = HWLANG_English;
         if (GetIncludeNumbers())
         {
            _numericRange = EUR_ALC_NUMERIC;
         }
         if (GetIncludeSymbols())
         {
            _symbolRange = EUR_ALC_SYMBOL;
         }
         if (GetIncludeLatinLowerCaseLetters())
         {
            _lowerCaseLatinLetterRange = EUR_ALC_LATIN_LOWERCASE;
         }
         if (GetIncludeLatinUpperCaseLetters())
         {
            _upperCaseLatinLetterRange = EUR_ALC_LATIN_UPPERCASE;
         }
         if (GetIncludeGestureRecognition())
         {
            _gestureRecognitionRange = EUR_ALC_GESTURE;
         }
         _languageRange = _lowerCaseLatinLetterRange | _upperCaseLatinLetterRange | _numericRange | _symbolRange | _gestureRecognitionRange;
         _charecterRecDictionaryPath = DICTIONARY_PATH_EUROPE;
      }
      break;

      default:
      {
         _currentLanguage = CHINESE_LANGUAGE;
         _languageRange = GB_ALC_GBK;
         _charecterRecDictionaryPath = CHAR_DICTIONARY_PATH_CHINESE;
         _wordRecDictionaryPath       = WORD_DICTIONARY_PATH_CHINESE;
      }
      break;
   }
}


void HandWriting_Util_Widget2D::OnParentViewRenderingEnabled(bool enable)
{
   if (enable == false)
   {
      _characterRecogniton_Timer.stop();
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "View INACTIVE, so HWR Timer KILLED"));
      if (_charList != 0)
      {
         memset(_charList, '\0' , MAX_CHAR_LIMITS);
      }
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HandWriting Widget View ACTIVE"));
      clearDrawingArea();
      getNodePosition();
      getNodeDimension();
   }
}


void HandWriting_Util_Widget2D::postCandidateUpdateMsg()
{
   if (_charList != NULL)
   {
      Message* msg = COURIER_MESSAGE_NEW(HWR_CandidateUpdate)(_charList);
      if (msg != 0)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HWR_CandidateUpdate Message posted\n"));
         msg->Post();
      }
   }
}


void HandWriting_Util_Widget2D::postHWRTextWidgetUpdateMsg()
{
   if (_charList != NULL)
   {
      Message* msg = COURIER_MESSAGE_NEW(HWR_TextWidgetUpdate)(&_charList[0]);
      if (msg != 0)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HWR_TextWidgetUpdate Message posted\n"));
         msg->Post();
      }
   }
}


bool HandWriting_Util_Widget2D::isRecognitionTimerRunning()
{
   if (_characterRecogniton_Timer.getStatus() == Timer::Running)
   {
      return true;
   }
   else
   {
      return false;
   }
}


void HandWriting_Util_Widget2D::clearHWRTextWidget()
{
   if (_charList != 0)
   {
      memset(_charList, '\0' , MAX_CHAR_LIMITS);
      postHWRTextWidgetUpdateMsg();
   }
}


void HandWriting_Util_Widget2D::extractStrings(
   const Candera::UInt16* f_array, Candera::UInt32 f_size, FeatStd::Int32 predictedResultCount
)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HandWriting_Util_Widget2D [%p]:extractStrings()" , this));
   if ((f_array) && (false == (getSourcePointer().PointsToNull())))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),
                          "((f_array) && (false == (getSourcePointer().PointsToNull()))) is ture"));
      //Allocate memory
      Candera::UInt32 count = 0;
      Candera::TChar l_String[5];
      std::string l_utf8Word;
      for (Candera::UInt8 i = 0; ((count < f_size) && (i < predictedResultCount)); i++)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "f_array[count] = [0x%x]", f_array[count]));
         l_utf8Word.clear();

         //Convert from UTF-16 to UTF-8 and store the word
         for (; ((count < f_size) && (f_array[count])) ; count++)
         {
            memset(l_String, 0, 5 * sizeof(TChar));
            (void)convertUTF16toUTF8(l_String, f_array[count]);

            //Read only one UTF-8 Character
            //if the string is empty we read one UTF-8 character after that the if condition is always
            //False
            if (l_utf8Word.empty())
            {
               l_utf8Word = l_utf8Word + std::string(l_String); //need to change here  for prediction
            }
         }
         _temp = l_utf8Word; //change for last characater
         //_temp = std::string(l_String);
         //ETG_TRACE_USR1(("_temp[%d] = [%s]",i, l_utf8Word.c_str()));
         {
            getSourcePointer()->setString(_temp); //for char prediction
         }
         count++;
      }
   }
}


/**start of String CRQ */
Candera::UInt8 HandWriting_Util_Widget2D::utfLength(Candera::String sText) const
{
   FeatStd::Internal::Utf8TextIterator iter_Text(sText.GetCString());
   UInt8 u8Length = 0;
   while (*iter_Text)
   {
      u8Length++;
      iter_Text.Advance();
   }
   return u8Length;
}


Candera::UInt8 HandWriting_Util_Widget2D::calculateUTF8Bytes(
   Candera::UInt8 iStartIndex, Candera::UInt8 uCharCount, Candera::String string
) const
{
   Candera::UInt8 nBytes = 0;
   // Validation
   if ((uCharCount > 0) && (uCharCount < MAX_TEXT_LENGTH) && (iStartIndex < MAX_TEXT_LENGTH))
   {
      Candera::UInt8 uStartIndex = iStartIndex;
      Candera::UInt8 iCharCount = uCharCount;
      FeatStd::Internal::Utf8TextIterator iter_Text(string.GetCString());

      // Calculate the number of bytes for a UTF character at index uStartIndex
      if (uStartIndex > 0)
      {
         while (uStartIndex > 0)
         {
            iter_Text.Advance(); // Move by one character till the start index is reached
            uStartIndex--;
         }
      }

      while (iCharCount > 0)
      {
         nBytes = static_cast<UInt8>(nBytes + iter_Text.Advance());
         iCharCount--;
      }
   }
   return nBytes;
}


bool HandWriting_Util_Widget2D::getlastHintUTF8Chars(Candera::String string)
{
   bool rc = false;
   //Get the max character
   Candera::UInt8 l_u8MaxLength = utfLength(string);
   FeatStd::Internal::Utf8TextIterator iter_Text(string.GetCString());
   Candera::UInt8 l_startIndex = 0;

   //will work for 255 codepoints only
   Candera::UInt32 l_startPos = 0;
   Candera::UInt8 l_index = 0;

   if (l_u8MaxLength > MAX_HINT_UTF8CHAR)
   {
      Candera::TChar* pDestStr = CANDERA_NEW_ARRAY(Char, (MAX_HINT_UTF8CHAR * MAX_BYTE_FOR_A_CHARACTER + 1));
      memset(pDestStr, 0, sizeof(Char) * (MAX_HINT_UTF8CHAR * MAX_BYTE_FOR_A_CHARACTER + 1));
      const Candera::TChar* pSource = string.GetCString();

      l_startIndex = static_cast<UInt8>(l_u8MaxLength - MAX_HINT_UTF8CHAR);

      //skip the number of characters indicated by l_startIndex from the beginning,
      //this makes sure we are reading the last MAX_HINT_UTF8CHAR characters
      while (*iter_Text && (l_index < l_startIndex))
      {
         l_index++;
         l_startPos += iter_Text.Advance(); //Counting number of Bytes from the starting address to be skipped
      }
      //Copy Last 6 characters
      memcpy(pDestStr, pSource + l_startPos, (string.GetCharCount() - l_startPos));

      //Hardcoded right now. _charactersToPredictedWord is updated for now
      //In future can be made generic
      _charactersToPredictedWord = pDestStr;
      FEATSTD_SAFE_DELETE_ARRAY(pDestStr);
      rc = true;
   }
   return rc;
}


void HandWriting_Util_Widget2D::getNodePosition()
{
   if (GetNode() != NULL)
   {
      _nodePosition_X = (Courier::UInt32)GetNode()->GetPosition().GetX();
      _nodePosition_Y = (Courier::UInt32)GetNode()->GetPosition().GetY();
   }
}


void HandWriting_Util_Widget2D::getNodeDimension()
{
   if (GetNode() != NULL)
   {
      Candera::Rectangle nodeBoundingRect;
      GetNode()->GetEffectiveBoundingRectangle(nodeBoundingRect);
      _drawingAreaWidth = nodeBoundingRect.GetWidth();
      _drawingAreaHeight = nodeBoundingRect.GetHeight();
   }
}


/**End of String CRQ */
