/* ***************************************************************************************
* FILE:          HanWang_Interface.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HanWang_Interface is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "HanWang_Interface.h"
#include <FeatStd/Platform/File.h>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "Widgets/widget_etg_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_HANDWRITING
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HanWang_Interface.cpp.trc.h"
#endif


HanWangInterface::HanWangInterface(
   FeatStd::Int32 dwCurrentLanguage,
   FeatStd::UInt32 dwRange,
   const FeatStd::Char* pcCharRecDictionaryPath,
   const FeatStd::Char* pcWordRecDictionaryPath,
   int modePriority
) :
   _characterRecDicRom(0),
   _wordRecDicRom(0),
   _attribute(0),
   _dwRange(dwRange),
   _dwLanguage(dwCurrentLanguage),
   _charRecDictionaryPath(pcCharRecDictionaryPath),
   _wordRecDictionaryPath(pcWordRecDictionaryPath),
   _priorityMode(modePriority)
{
   memset(_ram, 0, sizeof(_ram));
}


HanWangInterface::~HanWangInterface()
{
   if (_characterRecDicRom != 0)
   {
      FEATSTD_DELETE_ARRAY(_characterRecDicRom);
      _characterRecDicRom = 0;
   }
   if (_wordRecDicRom != 0)
   {
      FEATSTD_DELETE_ARRAY(_wordRecDicRom);
      _wordRecDicRom = 0;
   }
   if (_attribute != 0)
   {
      FEATSTD_DELETE(_attribute);
      _attribute = 0;
   }
   _charRecDictionaryPath = 0;
   _wordRecDictionaryPath = 0;
}


bool HanWangInterface::initialize()
{
   _attribute = FEATSTD_NEW(THWAttribute);

   if (_attribute != 0)
   {
      _characterRecDicRom = loadDictionaryPointer(_charRecDictionaryPath);
      _wordRecDicRom = loadDictionaryPointer(_wordRecDictionaryPath);

      _attribute->pRom = _characterRecDicRom;
      _attribute->pRam = (FeatStd::UInt8*)_ram;
      _attribute->dwLanguage = _dwLanguage;
      _attribute->iCandidateNum = 10;
      _attribute->dwRange = _dwRange;
      _attribute->dwSlantRange = 0;
      _attribute->wMode = (unsigned short)_priorityMode;
      _attribute->pFrame = 0;
      return true;
   }
   return false;
}


int HanWangInterface::recognizeCharacters(short* points, FeatStd::UInt16* result) const
{
   if (points != 0 && result != 0)
   {
      void* temp = (void*)points; // copy the pointer to remove LINT warning
      int size = HWRecognize((THWPoint*)temp, _attribute, result, 0);
      return size;
   }
   return 0;
}


FeatStd::UInt8* HanWangInterface::loadDictionaryPointer(const FeatStd::Char* dicPath)
{
   std::string dicFile = _dictionaryRootPath + dicPath;
   FeatStd::FileHandle currentFile = FeatStd::File::Open(dicFile.c_str());
   if (currentFile == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "FATAL: Couldn't open the handwriting dictionary %s", dicFile.c_str()));
      return 0;
   }

   // obtain file size:
   FeatStd::File::Seek(currentFile, 0, FeatStd::File::End);
   FeatStd::OffsetType lSize = FeatStd::File::Tell(currentFile);
   FeatStd::File::Seek(currentFile, 0, FeatStd::File::Begin);

   // allocate memory to contain the whole file:
   FeatStd::UInt8* pcDictionaryROM = (FeatStd::UInt8*)FEATSTD_NEW_ARRAY(FeatStd::Char, lSize);
   if (pcDictionaryROM == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "FATAL: No memory for handwriting dictionary %s", dicPath));
      return 0;
   }

   if (0 <= lSize)
   {
      // copy the file into internal memory
      FeatStd::SizeType readlength = FeatStd::File::Read(currentFile, pcDictionaryROM, 1, lSize);
      if (readlength != static_cast<FeatStd::SizeType>(lSize))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "FATAL: Could not read complete handwriting dictionary %s", dicPath));
         FEATSTD_DELETE_ARRAY(pcDictionaryROM);
         return 0;
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "FATAL: Invalid current file pointer position"));
      FEATSTD_DELETE_ARRAY(pcDictionaryROM);
      return 0;
   }

   if ((currentFile != NULL) && FeatStd::File::IsFileOpened(currentFile))
   {
      FeatStd::File::Close(currentFile);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "File Closed"));
   }
   return pcDictionaryROM;
}


//Added for the CRQ interface
int HanWangInterface::predictWords(const unsigned short* predictHint, FeatStd::UInt16* result)
{
   int icodeNum = 0;
   // Check added to solve the reset during scene creation
   if ((NULL != predictHint) && (NULL != result) && (NULL != _wordRecDicRom))
   {
      // need for Tradinal language
      //icodeNum=HWIM_PredictText(predictHint,m_pcRom,Result,2048,HWLANG_Trad_Chinese);
      // need for simple chinese
      icodeNum = HWIM_PredictText(predictHint, _wordRecDicRom, result, 2048, HWLANG_Simp_Chinese);
   }
   return icodeNum;
}
