/* ***************************************************************************************
* FILE:          WidgetFinder2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetFinder2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef WIDGET2D_FINDER_H
#define WIDGET2D_FINDER_H

#include "Widgets/2D/BaseWidget2D.h"

/* Utility class for searching widgets based on their relationship with other widgets and nodes.
* We consider that a widget is an ancestor of another widget if its node is an ancestor of the other's widget node.
* A particular situation is when 2 widgets are linked to the same node. They will be considered each other's ancestor depending on the situation.
*
* For example assume the following node structure:
* - RootNode
* -- Button1Group
* --- Text1RenderNode
* -- Button2Group
* --- Bitmap2RenderNode
* and widgets:
* - Button1Widget linked to Button1Group
* - Text1Widget linked to Text1RenderNode
* - ButtonTextColor1Widget linked also to Text1RenderNode
* - Button2Widget linked to Button2Group
* - ButtonImage2Widget linked to Bitmap2RenderNode
*
* Button1Widget is the ancestor of Text1Widget and ButtonTextColor1Widget.
* Button2Widget is the ancestor of ButtonImage2Widget.
*/
class WidgetFinder
{
   public:
      /* Searches for a widget of type TAncestorWidget which is linked to a node being an ancestor of the specified widget's node.
      * We consider that a widget is an ancestor of another widget if its node is an ancestor of the other's widget node or if both widgets are linked to the same node (this being a particular situation).
      * If the specified node is not null the search will be done only for ancestor nodes of it, otherwise the node of the widget will be used.
      */
      template<class TAncestorWidget>
      static TAncestorWidget* FindAncestorWidget(BaseWidget2D* widget, bool (*func)(TAncestorWidget&) = NULL)
      {
         if ((widget == NULL) || (widget->GetNode() == NULL))
         {
            return NULL;
         }

         return FindAncestorWidget<TAncestorWidget>(GetSceneContext(widget), widget->GetNode(), func);
      }

      /* Searches for a widget of type TAncestorWidget which is linked to a node being an ancestor of the specified node.
      * We consider that a widget is an ancestor of another widget if its node is an ancestor of the other's widget node or if both widgets are linked to the same node (this being a particular situation).
      */
      template<class TAncestorWidget>
      static TAncestorWidget* FindAncestorWidget(Candera::Scene2DContext* sceneContext, Candera::Node2D* node, bool (*func)(TAncestorWidget&) = NULL)
      {
         if ((sceneContext == NULL) || (node == NULL))
         {
            return NULL;
         }

         AncestorWidgetChecker<TAncestorWidget> widgetChecker(node, func);
         PerformAncestorSearch(sceneContext, node, &widgetChecker);
         return widgetChecker.GetAncestorWidget();
      }

      /* Searches for a widget of type TDescendantWidget which is linked to a node being an descendant of the specified node.
      * We consider that a widget is a descendant of another widget if its node is an descendant of the other's widget node or if both widgets are linked to the same node (this being a particular situation).
      */
      template<class TDescendantWidget>
      static TDescendantWidget* FindDescendantWidget(Candera::Scene2DContext* sceneContext, Candera::Node2D* node, bool visibleWidgetsOnly = false)
      {
         if ((sceneContext == NULL) || (node == NULL))
         {
            return NULL;
         }

         DescendantWidgetChecker<TDescendantWidget> widgetChecker(node, visibleWidgetsOnly);
         PerformSearch(sceneContext, &widgetChecker);
         //PerformDescendantSearch(sceneContext, node, &widgetChecker);//not working correctly yet
         return widgetChecker.GetDescendantWidget();
      }

      /* Searches and returns the descendant widget for the AncestorWidget.
      *	 The Node attached to the AncestorWidget is received as an input.
      * Makes use of the DescendentWidget2DFinder Callback
      */
      static FeatStd::Internal::Vector<Candera::WidgetBase*> FindDescendantWidget2D(Candera::Scene2DContext* sceneContext, Candera::Node2D* parentWidgetNodePtr, std::string descendentWidgetName, bool visibleWidgetsOnly = false)
      {
         if ((sceneContext == NULL) || (parentWidgetNodePtr == NULL))
         {
            //return NULL;
            return FeatStd::Internal::Vector<Candera::WidgetBase*>();
         }

         DescendentWidget2DFinder widgetChecker(parentWidgetNodePtr, descendentWidgetName, visibleWidgetsOnly);
         PerformSearch(sceneContext, &widgetChecker);
         return widgetChecker.GetChildWidgets();
      }

      class WidgetChecker : public WidgetCheckCallback
      {
         public:
            virtual bool IsWidgetFound() const = 0;
            virtual Candera::Node2D* GetNode() const = 0;
      };

      static Candera::Scene2DContext* GetSceneContext(BaseWidget2D* widget);
      static void PerformSearch(Candera::Scene2DContext* sceneContext, WidgetChecker* widgetChecker);
      static void PerformAncestorSearch(Candera::Scene2DContext* sceneContext, const Candera::Node2D* node, WidgetChecker* widgetChecker);
      static void PerformDescendantSearch(Candera::Scene2DContext* sceneContext, const Candera::Node2D* node, WidgetChecker* widgetChecker);
      static bool IsAncestorOf(const Candera::Node2D* ancestor, const Candera::Node2D* node);
      static bool IsDescendentOf(const Candera::Node2D* ancestor, const Candera::Node2D* node);

   private:
      WidgetFinder();
      WidgetFinder(const WidgetFinder&);
      WidgetFinder& operator=(const WidgetFinder&);

      template <typename NodeType, typename CompositeType>
      class MatchWidgetTraverser : public Candera::TreeTraverserBase<NodeType>
      {
         public:
            MatchWidgetTraverser(WidgetChecker* checker) : _checker(checker) { }

         protected:
            typedef ::Candera::TreeTraverserBase<NodeType> Base;
            typedef typename Base::TraverserAction MatchWidgetTraverserAction;

            virtual MatchWidgetTraverserAction ProcessNode(NodeType& node)
            {
               WidgetCheckReqMsg msg(_checker);
               Candera::CompositeGroup2D* compositeGroup = Candera::Dynamic_Cast<Candera::CompositeGroup2D*>(&node);
               if (compositeGroup != 0)
               {
                  // found composite
                  for (Candera::CompositeGroup2D::WidgetIterator it = compositeGroup->GetWidgetIterator(); it.IsValid(); ++it)
                  {
                     bool exactMatch = (*it)->OnMessage(msg);
                     if (_checker->IsWidgetFound())
                     {
                        //on scHost there is a problem if we iterate through all the widgets, so we will not go for the full search as we do in non SCHost environment
                        bool isSCHost = ((*it)->GetParentView() == NULL);
                        if (isSCHost || exactMatch)
                        {
                           return Base::StopTraversing;
                        }
                     }
                  }
               }
               return Base::ProceedTraversing;
            }

         private:
            WidgetChecker* _checker;
      };

      template<class TAncestorWidget>
      class AncestorWidgetChecker : public WidgetChecker
      {
         public:
            typedef bool (*TFunc)(TAncestorWidget&);

            AncestorWidgetChecker(Candera::Node2D* node, TFunc func = NULL)
               : _node(node),
                 _ancestorWidget(NULL),
                 _ancestorWidgetNode(NULL),
                 _func(func)
            {
               FEATSTD_DEBUG_ASSERT(_node != NULL);
            }

            TAncestorWidget* GetAncestorWidget() const
            {
               return _ancestorWidget;
            }

            virtual Candera::Node2D* GetNode() const
            {
               return _node;
            }

            virtual bool IsWidgetFound() const
            {
               return _ancestorWidget != NULL;
            }

            virtual bool CheckWidget(Candera::Widget2D* baseWidget)
            {
               TAncestorWidget* widget = Candera::Dynamic_Cast<TAncestorWidget*>(baseWidget);
               //todo: further optimizations can be done to check the chain of parent nodes only once instead of twice
               if ((widget != NULL) && (baseWidget != NULL) && WidgetFinder::IsAncestorOf(baseWidget->GetNode(), _node)
                     && ((_ancestorWidget == NULL) || WidgetFinder::IsAncestorOf(_ancestorWidgetNode, baseWidget->GetNode()))
                     && (_func == NULL || _func(*widget)))
               {
                  _ancestorWidget = widget;
                  _ancestorWidgetNode = baseWidget->GetNode();
                  //stop the search only if the widget is linked to exactly the same node and not an acenstor
                  //this way we'll get a chance to find widget which are "closer" to our node in the chain of nodes to the root
                  if (_node == _ancestorWidgetNode)
                  {
                     return true;
                  }
               }

               return false;
            }

         private:
            Candera::Node2D* _node;
            TAncestorWidget* _ancestorWidget;
            Candera::Node2D* _ancestorWidgetNode;
            TFunc _func;
      };

      template<class TDescendantWidget>
      class DescendantWidgetChecker : public WidgetChecker
      {
         public:
            DescendantWidgetChecker(Candera::Node2D* node, bool visibleWidgetsOnly)
               : _node(node),
                 _descendantWidget(NULL),
                 _descendantWidgetNode(NULL),
                 _visibleWidgetsOnly(visibleWidgetsOnly)
            {
               FEATSTD_DEBUG_ASSERT(_node != NULL);
            }

            TDescendantWidget* GetDescendantWidget() const
            {
               return _descendantWidget;
            }

            virtual Candera::Node2D* GetNode() const
            {
               return _node;
            }

            virtual bool IsWidgetFound() const
            {
               return _descendantWidget != NULL;
            }

            virtual bool CheckWidget(Candera::Widget2D* baseWidget)
            {
               TDescendantWidget* widget = Candera::Dynamic_Cast<TDescendantWidget*>(baseWidget);
               //todo: further optimizations can be done to check the chain of parent nodes only once instead of twice
               if ((widget != NULL) && (baseWidget != NULL) && (baseWidget->GetNode() != NULL)
                     && (!_visibleWidgetsOnly || baseWidget->GetNode()->IsEffectiveRenderingEnabled())
                     && WidgetFinder::IsAncestorOf(_node, baseWidget->GetNode())
                     && ((_descendantWidget == NULL) || WidgetFinder::IsAncestorOf(baseWidget->GetNode(), _descendantWidgetNode)))
               {
                  _descendantWidget = widget;
                  _descendantWidgetNode = baseWidget->GetNode();
                  //stop the search only if the widget is linked to exactly the same node and not an descendant
                  //this way we'll get a chance to find widget which are "closer" to our node in the chain of nodes to the root
                  if (_node == _descendantWidgetNode)
                  {
                     return true;
                  }
               }

               return false;
            }

         private:
            Candera::Node2D* _node;
            TDescendantWidget* _descendantWidget;
            Candera::Node2D* _descendantWidgetNode;
            bool _visibleWidgetsOnly;
      };

      /* CallBack function used to find if any descendent widgets are present for a given AncestorWidgetNode.
      *	Makes use of WidgetFinder::IsDescendentOf(const Candera::Node2D* ancestor, const Candera::Node2D* node) interface
      */
      class DescendentWidget2DFinder : public WidgetChecker
      {
         public:
            DescendentWidget2DFinder(Candera::Node2D* parentWidgetNodePtr, std::string descendentWidgetName, bool visibleWidgetsOnly)
               : _parentWidgetNodePtr(parentWidgetNodePtr),
                 _descendentWidgetName(descendentWidgetName),
                 _visibleWidgetsOnly(visibleWidgetsOnly),
                 _descendantBaseWidget2D(0),
                 _descendantWidgetNode(0)
            {
               FEATSTD_DEBUG_ASSERT(_parentWidgetNodePtr != NULL);
            }

            FeatStd::Internal::Vector<Candera::WidgetBase*> GetChildWidgets()
            {
               return _vecOfChildWidgets;
            }

            virtual Candera::Node2D* GetNode() const
            {
               return _parentWidgetNodePtr;
            }

            virtual bool IsWidgetFound() const
            {
               return _vecOfChildWidgets.Size() != 0;
            }

            virtual bool CheckWidget(Candera::Widget2D* widget2D)
            {
               //Candera::WidgetBase* widget = dynamic_cast<Candera::WidgetBase*>(baseWidget);
               //todo: further optimizations can be done to check the chain of parent nodes only once instead of twice
               if (widget2D != NULL && (widget2D->GetNode() != NULL)
                     && (!_visibleWidgetsOnly || widget2D->GetNode()->IsEffectiveRenderingEnabled())
                     && WidgetFinder::IsDescendentOf(_parentWidgetNodePtr, widget2D->GetNode())
                  )
               {
                  BaseWidget2D* baseWidget = Candera::Dynamic_Cast<BaseWidget2D*>(widget2D);

                  const char* name = baseWidget ? baseWidget->GetLegacyName() : widget2D->GetName();
                  if (!_descendentWidgetName.empty())
                  {
                     if (!strcmp(_descendentWidgetName.c_str(), name))
                     {
                        _descendantBaseWidget2D = widget2D;
                        _descendantWidgetNode = widget2D->GetNode();

                        // add the found child widget to the vector
                        _vecOfChildWidgets.Add(widget2D);
                     }
                  }
                  else
                  {
                     // child widget name not specified, so add all the found childs to the vector
                     _vecOfChildWidgets.Add(widget2D);
                  }
                  //stop the search only if the widget is linked to exactly the same node and not an descendant
                  //this way we'll get a chance to find widget which are "closer" to our node in the chain of nodes to the root
                  if (_parentWidgetNodePtr == _descendantWidgetNode)
                  {
                     return true;
                  }
               }
               return false;
            }

         private:
            Candera::Node2D* _parentWidgetNodePtr;
            std::string _descendentWidgetName;
            bool _visibleWidgetsOnly;
            Candera::WidgetBase* _descendantBaseWidget2D;
            Candera::Node2D* _descendantWidgetNode;
            Candera::Internal::Vector<Candera::WidgetBase*> _vecOfChildWidgets;
      };
};


#endif
