/* ***************************************************************************************
* FILE:          WidgetFinder2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetFinder2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/WidgetFinder2D.h"
#include <Widgets/utils/WidgetTraverser.h>
#include <View/CGI/CgiExtensions/SCHostUtils.h>
#include "CanderaAssetLoader/AssetLoaderBase/AssetProvider.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "CanderaAssetLoader/AssetLoader2D/Scene2DContext.h"
#ifdef WIN32
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#endif
#endif

Candera::Scene2DContext* WidgetFinder::GetSceneContext(BaseWidget2D* widget)
{
   Candera::Scene2DContext* context = NULL;
   if ((widget->GetParentView() != NULL) && (widget->GetParentView()->Is2D()))
   {
      context = widget->GetParentView()->ToViewScene2D()->GetScene2DContext();
   }

#ifdef WIN32
   //in Scene Composer the view is NULL so we use the scene context to iterate through the widgets
   if ((context == NULL) && (widget->GetNode() != NULL) && (widget->GetAssetProvider() != NULL))
   {
      Candera::Scene2D* scene = widget->GetNode()->GetScene();
      if (scene != NULL)
      {
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
         context = widget->GetAssetProvider()->GetScene2DById(Candera::Internal::AssetProviderFunctions::GetIdByName(widget->GetAssetProvider(), Candera::Scene2DLib, scene->GetName()));
#else
         context = widget->GetAssetProvider()->GetScene2D(scene->GetName());
#endif
         //if a normal scene was not found try getting the currently edited composite from SceneComposer
         if (context == NULL)
         {
            //this is required in order to have extension widgets working while editing the composites in SceneComposer, without it you will not see the button images
            context = hmibase::view::util::SCHostUtils::GetComposite2DContext(*(widget->GetAssetProvider()), scene->GetName());
         }
      }
   }
#endif

   return context;
}


void WidgetFinder::PerformSearch(Candera::Scene2DContext* sceneContext, WidgetChecker* widgetChecker)
{
   if ((sceneContext != NULL) && (sceneContext->GetScene() != NULL))
   {
      PerformDescendantSearch(sceneContext, sceneContext->GetScene(), widgetChecker);
   }
}


void WidgetFinder::PerformAncestorSearch(Candera::Scene2DContext* sceneContext, const Candera::Node2D* node, WidgetChecker* widgetChecker)
{
   if ((sceneContext != NULL) && (node != NULL) && (widgetChecker != NULL))
   {
      WidgetCheckReqMsg msg(widgetChecker);

      for (Candera::SceneContextBase::WidgetIterator it = sceneContext->GetWidgetIterator(); it.IsValid(); ++it)
      {
         Candera::WidgetBase* widget = *it;

         bool exactMatch = widget->OnMessage(msg);
         if (widgetChecker->IsWidgetFound())
         {
            //on scHost there is a problem if we iterate through all the widgets, so we will not go for the full search as we do in non SCHost environment
            bool isSCHost = (widget->GetParentView() == NULL);
            if (isSCHost || exactMatch)
            {
               return;
            }
         }
      }

      while (node != NULL)
      {
         const Candera::CompositeGroup2D* compositeGroup = Candera::Dynamic_Cast<const Candera::CompositeGroup2D*>(node);
         if (compositeGroup != 0)
         {
            // found composite
            for (Candera::CompositeGroup2D::WidgetIterator it = compositeGroup->GetWidgetIterator(); it.IsValid(); ++it)
            {
               bool exactMatch = (*it)->OnMessage(msg);
               if (widgetChecker->IsWidgetFound())
               {
                  //on scHost there is a problem if we iterate through all the widgets, so we will not go for the full search as we do in non SCHost environment
                  bool isSCHost = ((*it)->GetParentView() == NULL);
                  if (isSCHost || exactMatch)
                  {
                     return;
                  }
               }
            }
         }

         //this is required because widgets cloned by the list are not added to their composite group, instead they are kept into the control template data of the node
         if (hmibase::widget::utils::MessageUtils::distributeToControlTemplate(*node, msg))
         {
            return;
         }

         node = node->GetParent();
      }
   }
}


void WidgetFinder::PerformDescendantSearch(Candera::Scene2DContext* sceneContext, const Candera::Node2D* node, WidgetChecker* widgetChecker)
{
   if ((sceneContext != NULL) && (node != NULL) && (widgetChecker != NULL))
   {
      WidgetCheckReqMsg msg(widgetChecker);

      for (Candera::SceneContextBase::WidgetIterator it = sceneContext->GetWidgetIterator(); it.IsValid(); ++it)
      {
         Candera::WidgetBase* widget = *it;

         bool exactMatch = widget->OnMessage(msg);
         if (widgetChecker->IsWidgetFound())
         {
            //on scHost there is a problem if we iterate through all the widgets, so we will not go for the full search as we do in non SCHost environment
            bool isSCHost = (widget->GetParentView() == NULL);
            if (isSCHost || exactMatch)
            {
               return;
            }
         }
      }
      MatchWidgetTraverser<Candera::Node2D, Candera::CompositeGroup2D> matchWidgetTraverer(widgetChecker);
      matchWidgetTraverer.Traverse(const_cast<Candera::Node2D&>(*node));
   }
}


bool WidgetFinder::IsAncestorOf(const Candera::Node2D* ancestor, const Candera::Node2D* node)
{
   if (ancestor == NULL)
   {
      return false;
   }

   while ((node != NULL) && (node != ancestor))
   {
      node = node->GetParent();
   }
   return (node == ancestor);
}


bool WidgetFinder::IsDescendentOf(const Candera::Node2D* ancestor, const Candera::Node2D* node)
{
   if (ancestor == NULL || ancestor == node)
   {
      return false;
   }

   while ((node != NULL) && (node != ancestor))
   {
      node = node->GetParent();
   }
   return (node == ancestor);
}
