/* ***************************************************************************************
* FILE:          TimerWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "Widgets/2D/TimerWidget/TimerWidget2D.h"

using namespace Courier;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TIMER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TimerWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TimerWidget2D);


TimerWidget2D::TimerWidget2D()
{
   _timer.setName("TimerWidget2D", GetName()); // TODO instance name should be better
}


TimerWidget2D::~TimerWidget2D()
{
}


/****************************************************************************
*  Framework call when the rendering state of the scene changes
*  Overridden methods from BaseWidget
*  Parameters  : bool
****************************************************************************/
void TimerWidget2D::OnParentViewRenderingEnabled(bool bIsEnabled)
{
   if (bIsEnabled && GetAutoStartEnabled())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s]. Autostart the timer Duration=%ld", GetName(), GetTimerDuration()));
      startTimer();
   }
}


bool TimerWidget2D::startTimer()
{
   if (!_timer.running())
   {
      if (setupTimer())
      {
         _timer.start();
         return true;
      }
   }
   return false;
}


bool TimerWidget2D::setupTimer()
{
   if (GetTimerDuration() == 0)
   {
      return false;
   }

   bool running = _timer.running();

   if (GetAutoRestartEnabled())
   {
      // If this property is enabled, the timer is automatically triggered when the scene rendering is activated.
      _timer.setTimeoutWithRepeat(GetTimerDuration(), GetTimerDuration());
   }
   else
   {
      _timer.setTimeoutWithRepeat(GetTimerDuration(), 0);
   }

   if (running)
   {
      _timer.start();
   }

   return true;
}


/****************************************************************************
*  Framework call when a message is to be received by the widget
*  Overridden methods from BaseWidget
*  ->> see Timer::Expire()
****************************************************************************/
bool TimerWidget2D::OnMessage(const Message& oMsg)
{
   if (TimerExpiredMsg::ID == oMsg.GetId())
   {
      const TimerExpiredMsg* p = message_cast<const TimerExpiredMsg*>(&oMsg);

      if (p->GetTimer() == &_timer)
      {
         bool stopped = GetAutoRestartEnabled() ? false : true;

         Courier::Message* widgetMsg = COURIER_MESSAGE_NEW(TimerWidgetExpiryMsg)(Courier::ViewId(GetParentView()->GetId().CStr()), Courier::Identifier(GetName()), GetUserData(),
                                       _timer.getElapsedTimeSinceStart(), stopped);

         if (widgetMsg)
         {
            widgetMsg->Post();
         }

         if (!GetAutoRestartEnabled())
         {
            _timer.stop();
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s].TimerWidget timer Expired.", GetName()));
         return true;
      }
   }

   return false;
}


void TimerWidget2D::Update()
{
   // nothing to do, no render tasks for this widget
}


/****************************************************************************
* Framework Property changed message
*  Overridden methods from BaseWidget
****************************************************************************/
void TimerWidget2D::OnChanged(UInt32 propertyId)
{
   switch (propertyId)
   {
      case VisiblePropertyId:
      case EnabledPropertyId:
         Base::OnChanged(propertyId);
         break;

      case AutoRestartEnabledPropertyId: //  If this property is enabled, the timer is automatically triggered when the scene rendering is activated.
      case TimerDurationPropertyId:
         setupTimer();
         break;

      case StartStopTriggerPropertyId:
         handleStartStopTrigger(GetStartStopTrigger());
         break;

      case PauseTriggerPropertyId:
         handlePauseTrigger(GetPauseTrigger());
         break;

      default:
         break;
   }
   // triggerUpdate(); nothing to do, no render tasks for this widget
}


/****************************************************************************
* A explicit trigger to start the timer. If start event is received
* while timeout is running or stoped its time is reset to run over full duration.
* If start event is received while timeout is paused its time is continued to
* run over the remaining duration.
****************************************************************************/
void TimerWidget2D::handleStartStopTrigger(bool bStart)
{
   if (bStart)
   {
      switch (_timer.getStatus())
      {
         case ::Util::Timer::Running:
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s].Restart a running Timer", GetName()));
            _timer.stop();
            startTimer();
            return;

         case ::Util::Timer::Stopped:
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s]. Start a stopped Timer", GetName()));
            FEATSTD_DEBUG_ASSERT(GetTimerDuration() != 0);
            startTimer();
            return;

         case ::Util::Timer::Paused:
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s], Restarted after pause.", GetName()));
            _timer.resume();
            return;

         default:
            break;
      }
      return;
   }
   else
   {
      // Stop
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TimerWidget2D: [%19s]. Stop Timer", GetName()));
      _timer.stop();
   }
}


/****************************************************************************
* Trigger to pause the timer
*  bPause: true continues the timer not elapsed ( remaining time > 0 )
****************************************************************************/
void TimerWidget2D::handlePauseTrigger(bool bPause)
{
   if (bPause && _timer.getStatus() == ::Util::Timer::Running)
   {
      _timer.pause();
   }

   if (!bPause && _timer.getStatus() == ::Util::Timer::Paused)
   {
      _timer.resume();
   }
}
