/* ***************************************************************************************
* FILE:          TextWaitAnimationWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextWaitAnimationWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/TextWaitAnimation/TextWaitAnimationWidget2D.h"
#include "Widgets/2D/Text/TextWidget2D.h"
#include "Widgets/2D/StepAnimation/StepAnimationWidget2D.h"
#include "Widgets/2D/WidgetFinder2D.h"
#include "View/CGI/CgiExtensions/CourierMessageMapper.h"
#include "Candera/Engine2D/Core/TextNode2D.h"

using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextWaitAnimationWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TextWaitAnimationWidget2D);


/******************************************************************************
*  Constructor
******************************************************************************/
TextWaitAnimationWidget2D::TextWaitAnimationWidget2D() :
   _text("")
{
}


/***************************************************************************
*     destructor
****************************************************************************/
TextWaitAnimationWidget2D::~TextWaitAnimationWidget2D()
{
}


/*****************************************************************************/
void TextWaitAnimationWidget2D::InitWidget()
{
   Base::InitWidget();
}


/*****************************************************************************/
void TextWaitAnimationWidget2D::Update()
{
   if (NULL != GetTextWidget() && _text != dynamic_cast<TextWidget2D*>(GetTextWidget())->GetText())
   {
      calAnimationPosition();
   }
}


/*****************************************************************************/
void TextWaitAnimationWidget2D::OnParentViewActivate(bool enable)
{
   if (enable)
   {
      calAnimationPosition();
   }
}


/*****************************************************************************/
bool TextWaitAnimationWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget,
      ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const TextWaitAnimationWidget2D* textWaitAnimationWidget = CLONEABLE_WIDGET_CAST<const TextWaitAnimationWidget2D*>(originalWidget);
      if (textWaitAnimationWidget == NULL)
      {
         return false;
      }

      SetAnimationText(textWaitAnimationWidget->GetAnimationText());
      SetTextWidget(controlTemplateMap.ResolveWidgetClone(textWaitAnimationWidget->GetTextWidget()));
      cloned = true;
   }
   return cloned;
}


/*****************************************************************************/
void TextWaitAnimationWidget2D::calAnimationPosition()
{
   TextWidget2D* textWidget = NULL;
   StepAnimationWidget2D* stepAnimationWidget = NULL;
   if (NULL != GetTextWidget())
   {
      textWidget = dynamic_cast<TextWidget2D*>(GetTextWidget());
   }
   stepAnimationWidget = WidgetFinder::FindAncestorWidget<StepAnimationWidget2D>(this);
   if (NULL != textWidget && NULL != stepAnimationWidget && NULL != GetNode() && !textWidget->GetText().IsEmpty())
   {
      //calculate the textwidget string width
      Candera::Int16 textWidth = calculateTextWidth(textWidget->GetText());
      //append the animation text to end of the text
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(textWidget->GetText())
      std::string str = textWidget->GetText().GetCString();
      str.append(GetAnimationText().GetCString());
      textWidget->SetText(str.c_str());
      SECURE_FEATSTD_STRING_ACCESS_END()

      Candera::Int32 animationTextWidth = 0;
      Candera::Int32 animationTextLength = 0;
      Candera::Int32 singleDotWidth = 0;

      if (!GetAnimationText().IsEmpty())
      {
         //calculate the width of the animation text
         animationTextWidth = calculateTextWidth(GetAnimationText());
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetAnimationText())
         animationTextLength = static_cast<Candera::Int32>(StringPlatform::Length(GetAnimationText().GetCString()));
         SECURE_FEATSTD_STRING_ACCESS_END()
         if (animationTextLength != 0)
         {
            singleDotWidth = static_cast<Candera::Int32>(animationTextWidth / animationTextLength);
         }
      }

      Candera::Vector2 textNodeEndPos = Candera::Vector2(0.0f, 0.0f);
      if (NULL != textWidget->GetNode() && NULL != GetNode()->GetParent())
      {
         textNodeEndPos.SetX((textWidget->GetNode()->GetPosition().GetX() + (textWidth)));
         textNodeEndPos.SetY(textWidget->GetNode()->GetPosition().GetY());
      }
      //set the mask node position to the end of the text
      GetNode()->SetPosition((textNodeEndPos.GetX()), (textNodeEndPos.GetY()));

      //calculate keyframes
      stepAnimationWidget->setKeyframeValues(calKeyFrames(textNodeEndPos, animationTextLength, singleDotWidth));
      _text = textWidget->GetText();
   }
}


/*****************************************************************************/
Candera::ArrayProperty<FeatStd::Float> TextWaitAnimationWidget2D::calKeyFrames(Candera::Vector2 textNodeEndPos, Candera::Int32 textLength, Candera::Int32 singleCharWidth)
{
   Candera::ArrayProperty<FeatStd::Float> keyFrame;

   for (int index = 0; index <= textLength; index++)
   {
      keyFrame.Add((textNodeEndPos.GetX() + static_cast<FeatStd::Float>(singleCharWidth * index)));
      keyFrame.Add(textNodeEndPos.GetY());
   }
   return keyFrame;
}


/*****************************************************************************/
Candera::Int16 TextWaitAnimationWidget2D::calculateTextWidth(Candera::String string)
{
   TextWidget2D* textWidget = dynamic_cast<TextWidget2D*>(GetTextWidget());
   if (textWidget)
   {
      TextNode2D* textNode = dynamic_cast<Candera::TextNode2D*>(textWidget->GetNode());
      if (textNode && !textNode->GetStyle().PointsToNull())
      {
         TextRendering::CursorTextMeasureContext measureContext(*textNode->GetStyle(), 0);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(string)
         TextRendering::TextRenderer().Render(measureContext, TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(textNode->GetStyle()), TextRendering::TextProperties(string.GetCString()));
         SECURE_FEATSTD_STRING_ACCESS_END()
         const TextRendering::TextRect& textRect = measureContext.GetTextRectangle();
         TextRendering::TextSize tsize = TextRendering::TextSize(static_cast<Int16>(Math::Minimum<Int16>(textRect.GetLeft(), 0) + textRect.GetRight()), static_cast<Int16>(Math::Minimum<Int16>(textRect.GetTop(), 0) + textRect.GetHeight()));
         //returns the width of the text
         return tsize.GetWidth();
      }
   }
   return TextRendering::TextSize().GetWidth();
}
