/* ***************************************************************************************
* FILE:          TextHighlightWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextHighlightWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "TextHighlightWidget2D.h"

#include <algorithm>

#include <CanderaPlatform/Device/Common/Effects/TextBrush.h>
#include <FeatStd/Platform/Generic/GenericString.h>
#include <Widgets/2D/ControlTemplate/TCloneTraverser.h>
#include <Widgets/utils/IndexRangeSet.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextHighlightWidget2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace text {

CGI_WIDGET_RTTI_DEFINITION(TextHighlightWidget2D);


/*****************************************************************************/
class HighlightNodeCloneTraverser : public TCloneTraverser<Candera::Node2D>
{
      typedef TCloneTraverser<Candera::Node2D> Base;
   public:
      HighlightNodeCloneTraverser() : Base() {}

   protected:
      virtual TraverserBase::TraverserAction OnClone(const Candera::Node2D& node, Candera::Node2D& nodeClone)
      {
         Base::OnClone(node, nodeClone);

         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&nodeClone);
         if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
         {
            Candera::TextBrush* textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
            if (textBrush != NULL)
            {
               textBrush->Text().Set("", NULL);
               textBrush->PreprocessedText().Set(hmibase::widget::text::GlyphDataIterator());
            }

            //Candera::TextNode2D* textNode = Candera::Dynamic_Cast<Candera::TextNode2D*>(renderNode);
            //if (textNode != NULL)
            //{
            //   textNode->SetPreprocessedText(TextRendering::TextRenderArgs::SharedPointer());
            //   textNode->SetText(FeatStd::String());
            //}
         }
         return TraverserBase::StopTraversingForDescendants;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(HighlightNodeCloneTraverser);
};


/*****************************************************************************/
TextHighlightWidget2D::TextHighlightWidget2D() : Base(),
   _invalid(true), _highlightNode(NULL), _lastTextBrushDataUpdateCount(0)
{
}


/*****************************************************************************/
TextHighlightWidget2D::~TextHighlightWidget2D()
{
   if (_highlightNode != NULL)
   {
      if ((_highlightNode->GetEffect(0) != NULL))
      {
         Candera::TextBrush* highlightTextBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(_highlightNode->GetEffect(0)->GetBrushEffect2D());
         if (highlightTextBrush != NULL)
         {
            highlightTextBrush->PreprocessedText().Set(GlyphDataIterator());
         }
      }

      _highlightNode = NULL;
   }
}


/*****************************************************************************/
void TextHighlightWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();

   if (_highlightNode != NULL)
   {
      if ((_highlightNode->GetEffect(0) != NULL))
      {
         Candera::TextBrush* highlightTextBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(_highlightNode->GetEffect(0)->GetBrushEffect2D());
         if (highlightTextBrush != NULL)
         {
            highlightTextBrush->PreprocessedText().Set(GlyphDataIterator());
         }
      }

      _highlightNode = NULL;
   }
   _textBrushData.Release();
   _lastTextBrushDataUpdateCount = 0;
   _invalid = true;
}


/*****************************************************************************/
void TextHighlightWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _lastTextBrushDataUpdateCount = 0;
   _invalid = true;
}


/*****************************************************************************/
typedef std::vector<const CanderaGlyphData*> GlyphDataPtrVector;
static void filterGlyphsByMask(const FeatStd::TChar* text, const GlyphDataPtrVector& sortedGlyphs, const Candera::String& mask, GlyphDataPtrVector& filteredGlyphs)
{
   if (text == NULL || text[0] == '\0' || mask.IsEmpty())
   {
      return;
   }

   FeatStd::SizeType textLength =  FeatStd::Internal::Generic::GenericString::Length(text);
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "filterGlyphsByMask maskCodePoint=%u, maskLength=%u, mask=%100s, textLength=%u, text=%s",
                       mask.GetCodePointCount(), mask.GetCharCount(), mask.GetCString(), textLength, text));

   for (GlyphDataPtrVector::const_iterator it = sortedGlyphs.begin(); it != sortedGlyphs.end();)
   {
      const CanderaGlyphData* g = (*it);
      if ((g != NULL) && (g->GetCharacterPosition() < static_cast<int>(textLength)))
      {
         if (FeatStd::Internal::String::ComparePartial(text + g->GetCharacterPosition(), mask.GetCString(), mask.GetCharCount()) == 0)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "filterGlyphsByMask found match pos=%d", g->GetCharacterPosition()));
            for (FeatStd::SizeType index = 0; (index < mask.GetCodePointCount()) && (it != sortedGlyphs.end()); ++index, ++it)
            {
               const CanderaGlyphData* g = (*it);
               if (g != NULL)
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "filterGlyphsByMask addGlyph charPos=%d", g->GetCharacterPosition()));
                  filteredGlyphs.push_back(g);
               }
            }
            //continue to prevent double incrementation of the iterator
            continue;
         }
      }
      ++it;
   }
}


/*****************************************************************************/
static void filterGlyphsByRange(const FeatStd::TChar* text, const GlyphDataPtrVector& sortedGlyphs, const Candera::String& rangeText, GlyphDataPtrVector& filteredGlyphs)
{
   if (text == NULL || text[0] == '\0' || rangeText.IsEmpty())
   {
      return;
   }

   FeatStd::SizeType textLength = FeatStd::Internal::Generic::GenericString::Length(text);
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "filterGlyphsByRange rangeText=%100s, textLength=%u, text=%s", rangeText.GetCString(), textLength, text));

   hmibase::widget::utils::IndexRangeSet* rangeSet = NULL;
   if (sortedGlyphs.size() <= 32)
   {
      rangeSet = FEATSTD_NEW(hmibase::widget::utils::BitSetIndexRangeSet<32>);
   }
   else if (sortedGlyphs.size() <= 64)
   {
      rangeSet = FEATSTD_NEW(hmibase::widget::utils::BitSetIndexRangeSet<64>);
   }
   else if (sortedGlyphs.size() <= 128)
   {
      rangeSet = FEATSTD_NEW(hmibase::widget::utils::BitSetIndexRangeSet<128>);
   }
   else
   {
      rangeSet = FEATSTD_NEW(hmibase::widget::utils::DynamicIndexRangeSet);
   }

   if (rangeSet != NULL)
   {
      hmibase::widget::utils::IndexRangeSetFactory::fillRanges(*rangeSet, rangeText.GetCString());

      size_t index = 0;
      for (GlyphDataPtrVector::const_iterator it = sortedGlyphs.begin(); it != sortedGlyphs.end(); ++it, ++index)
      {
         const CanderaGlyphData* g = (*it);
         if ((g != NULL) && rangeSet->contains(index))
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "filterGlyphsByRange addGlyph index=%u, charPos=%d", index, g->GetCharacterPosition()));
            filteredGlyphs.push_back(g);
         }
      }

      FEATSTD_DELETE(rangeSet);
      rangeSet = NULL;
   }
}


/*****************************************************************************/
void TextHighlightWidget2D::Update()
{
   Base::Update();

   if (_invalid || _textBrushData.PointsToNull() || (_textBrushData->_UpdateCount != _lastTextBrushDataUpdateCount))
   {
      //get main render node and text brush
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
      Candera::TextBrush* textBrush = NULL;
      if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
      {
         textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
      }

      if ((textBrush != NULL) && _textBrushData.PointsToNull())
      {
         _textBrushData = DynamicPropertiesAccessor::GetTextBrushData(*textBrush);
         _lastTextBrushDataUpdateCount = 0;
      }

      //get highlight render node
      if ((_highlightNode == NULL) && (renderNode != NULL) && (textBrush != NULL))
      {
         //search a descendant node with the same name as this widget
         _highlightNode = Candera::Dynamic_Cast<Candera::RenderNode*>(renderNode->GetDescendant(GetName()));
         if (_highlightNode == NULL)
         {
            //if not found then clone the main render node
            HighlightNodeCloneTraverser cloneTraverser;
            cloneTraverser.Traverse(*renderNode);
            _highlightNode = Candera::Dynamic_Cast<Candera::RenderNode*>(cloneTraverser.GetRootClone());
            if (_highlightNode != NULL)
            {
               //_highlightNode->SetName("");
               renderNode->AddChild(_highlightNode);
               _highlightNode->Upload();
            }
         }
      }

      //get highlight text brush
      Candera::TextBrush* highlightTextBrush = NULL;
      if ((_highlightNode != NULL) && (_highlightNode->GetEffect(0) != NULL))
      {
         highlightTextBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(_highlightNode->GetEffect(0)->GetBrushEffect2D());
      }

      if ((textBrush != NULL) && (highlightTextBrush != NULL))
      {
         //get the glyphs from the main render node and sort them based on character position
         GlyphDataPtrVector sortedGlyphs;

         bool updateTextBrush = false;
         if (!_textBrushData.PointsToNull())
         {
            for (TextBrushData::GlyphDataVector::Iterator it = _textBrushData->_GlyphDataVector.Begin(); it != _textBrushData->_GlyphDataVector.End(); ++it)
            {
               GlyphData& g = (*it);
               sortedGlyphs.push_back(&g);
               g.m_hidden = false;
            }
         }
         else
         {
            for (Candera::TextRendering::PreprocessingContext::Iterator it = textBrush->PreprocessedText().Get(); it.IsValid(); ++it)
            {
               const CanderaGlyphData& g = (*it);
               sortedGlyphs.push_back(&g);

               //reset hidden flag of the main render node glyphs
               const GlyphData* g2 = dynamic_cast<const GlyphData*>(&g);
               if ((g2 != NULL) && g2->m_hidden)
               {
                  const_cast<GlyphData*>(g2)->m_hidden = false;
                  updateTextBrush = true;
               }
            }
         }

         std::stable_sort(sortedGlyphs.begin(), sortedGlyphs.end(), TextUtils::CompareGlyphsByCharacterPosition<const CanderaGlyphData>);

         //filter these sorted glyphs to get those required for highlight
         const FeatStd::TChar* text = textBrush->Text().Get();
         GlyphDataPtrVector filteredGlyphs;
         if (GetMode() == enHighlightMode::Mask)
         {
            filterGlyphsByMask(text, sortedGlyphs, GetText(), filteredGlyphs);
         }
         else
         {
            filterGlyphsByRange(text, sortedGlyphs, GetText(), filteredGlyphs);
         }

         //check if highlight glyphs rendered are different from the filtered & sorted ones
         bool areGlyphsDifferent = false;
         if (filteredGlyphs.size() == _highlightGlyphVectorData.size())
         {
            GlyphDataPtrVector::const_iterator it = filteredGlyphs.begin();
            BaseGlyphDataVector::const_iterator it2 = _highlightGlyphVectorData.begin();
            for (; (it != filteredGlyphs.end()) && (it2 != _highlightGlyphVectorData.end()); ++it, ++it2)
            {
               const CanderaGlyphData* g = (*it);
               const BaseGlyphData& g2 = (*it2);
               if ((g == NULL) || !g2.matches(*g))
               {
                  areGlyphsDifferent = true;
                  break;
               }
            }
         }
         else
         {
            areGlyphsDifferent = true;
         }

         //update highlight text brush if required
         if (areGlyphsDifferent)
         {
            _highlightGlyphVectorData.clear();
            for (GlyphDataPtrVector::const_iterator it = filteredGlyphs.begin(); it != filteredGlyphs.end(); ++it)
            {
               const CanderaGlyphData* g = (*it);
               if (g != NULL)
               {
                  _highlightGlyphVectorData.push_back(*g);

                  //set hidden flag of the main render node glyphs
                  const GlyphData* g2 = dynamic_cast<const GlyphData*>(g);
                  if ((g2 != NULL) && !g2->m_hidden)
                  {
                     const_cast<GlyphData*>(g2)->m_hidden = true;
                     updateTextBrush = true;
                  }
               }
            }
            highlightTextBrush->PreprocessedText().Set(TextUtils::CreateGlyphDataIterator(_highlightGlyphVectorData));
         }

         //reset text if required
         bool isTextChanged = false;
         if ((highlightTextBrush->Text().Get() != NULL) && (highlightTextBrush->Text().Get()[0] != '\0'))
         {
            highlightTextBrush->Text().Set("", NULL);
            isTextChanged = true;
         }

         //update color if required
         bool isColorChanged = false;
         if (highlightTextBrush->Color().Get() != GetColor())
         {
            highlightTextBrush->Color().Set(GetColor());
            isColorChanged = true;
         }

         //update brush if required
         if (areGlyphsDifferent || isColorChanged || isTextChanged)
         {
            highlightTextBrush->Update();
         }

         //hidden flag of the main render node glyphs was changed
         if (updateTextBrush)
         {
            textBrush->PreprocessedText().SetChanged(true);
            textBrush->Update();
         }
      }

      if (!_textBrushData.PointsToNull())
      {
         _lastTextBrushDataUpdateCount = _textBrushData->_UpdateCount;
      }

      _invalid = false;
   }
}


/*****************************************************************************/
bool TextHighlightWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const TextHighlightWidget2D* original = CLONEABLE_WIDGET_CAST<const TextHighlightWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetColor(original->GetColor());
      SetMode(original->GetMode());
      SetText(original->GetText());

      cloned = true;
   }
   return cloned;
}


}
}


}
