/* ***************************************************************************************
* FILE:          TextBaseLineOffsetWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextBaseLineOffsetWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/TextBrushBlend.h"
#include "TextBaseLineOffsetWidget2D.h"
#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/TextArea/TextAreaWidget2D.h>

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/BaseLineLayouter.h>
#else
#include <Candera/Engine2D/Layout/BaseLineLayouter.h>
#endif


#include <Candera/System/Mathematics/Math.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextBaseLineOffsetWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TextBaseLineOffsetWidget2D);


/*****************************************************************************/
TextBaseLineOffsetWidget2D::TextBaseLineOffsetWidget2D()
   : Base(), _textBrush(NULL), _textAreaWidget(NULL), _cachedTextHeight(-1.0f)
{
}


/*****************************************************************************/
TextBaseLineOffsetWidget2D::~TextBaseLineOffsetWidget2D()
{
   _textBrush = NULL;
   _textAreaWidget = NULL;
}


/*****************************************************************************/
void TextBaseLineOffsetWidget2D::OnNodeChanged()
{
   _textBrush = NULL;
   _textAreaWidget = NULL;
   _cachedTextHeight = -1.0f;

   triggerUpdate();
}


/*****************************************************************************/
void TextBaseLineOffsetWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _cachedTextHeight = -1.0f;
}


/*****************************************************************************/
void TextBaseLineOffsetWidget2D::UpdateTextBrush()
{
   if ((_textBrush == NULL) && (_textAreaWidget == NULL))
   {
      //search first for the text brush
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
      if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
      {
         _textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
      }

      //if text brush is not found then search for the text area widget
      if (_textBrush == NULL)
      {
         _textAreaWidget = WidgetFinder::FindAncestorWidget<TextAreaWidget2D>(this);
      }
   }
}


/*****************************************************************************/
bool TextBaseLineOffsetWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   if (!Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      return false;
   }

   const TextBaseLineOffsetWidget2D* original = CLONEABLE_WIDGET_CAST<const TextBaseLineOffsetWidget2D*>(originalWidget);
   if (original == NULL)
   {
      return false;
   }

   SetFontLineHeight(original->GetFontLineHeight());
   SetSingleLineOffset(original->GetSingleLineOffset());
   SetMultipleLinesOffset(original->GetMultipleLinesOffset());

   return true;
}


/*****************************************************************************/
void TextBaseLineOffsetWidget2D::Update()
{
   Base::Update();

   UpdateTextBrush();
   if (GetNode() != NULL)
   {
      enum { InvalidValue, SingleLine, MultipleLines } option = InvalidValue;

      if (_textBrush != NULL)
      {
         Candera::Rectangle boundingRectangle;
         _textBrush->GetBoundingRectangle(boundingRectangle);
         if (!Candera::Math::FloatAlmostEqual(boundingRectangle.GetHeight(), _cachedTextHeight))
         {
            FontLineHeightType fontLineHeight = GetFontLineHeight();
            //if font line height is not specified we use the value from the font metrics
            if ((fontLineHeight <= 0) && !_textBrush->Style().Get().PointsToNull())
            {
               fontLineHeight = _textBrush->Style().Get()->GetMetrics().lineHeight;
            }

            if ((fontLineHeight > 0) && (boundingRectangle.GetHeight() >= 0.0f))
            {
               //to avoid extra calculations and to improve the performance we cache the text height of the last update of the brush
               _cachedTextHeight = boundingRectangle.GetHeight();

               //we consider that the text has multiple lines if the height of the text box is greater than font line height.
               option = (_cachedTextHeight > static_cast<float>(fontLineHeight)) ? MultipleLines : SingleLine;
            }
         }
      }
      else if (_textAreaWidget != NULL)
      {
         option = (_textAreaWidget->GetEffectiveNumberOfLines() > 1) ? MultipleLines : SingleLine;
      }
      else
      {
         //nothing to do
      }

      Candera::Node2D* parentNode = GetNode()->GetParent();
      if ((parentNode != NULL) && (dynamic_cast<Candera::BaseLineLayouter*>(parentNode->GetLayouter()) != NULL))
      {
         switch (option)
         {
            case SingleLine:
               if (Candera::BaseLineLayouter::GetBaseLineOffset(*parentNode) != GetSingleLineOffset())
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update set SingleLineOffset=%.0f on group=%75s %s",
                                      GetSingleLineOffset(), HMIBASE_TO_STRING_PN2D(parentNode), HMIBASE_TO_STRING_VW(this)));

                  Candera::BaseLineLayouter::SetBaseLineOffset(*parentNode, GetSingleLineOffset());
                  InvalidateLayout();
               }
               break;
            case MultipleLines:
               if (Candera::BaseLineLayouter::GetBaseLineOffset(*parentNode) != GetMultipleLinesOffset())
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update Set MultipleLineOffset=%.0f on group=%75s %s",
                                      GetMultipleLinesOffset(), HMIBASE_TO_STRING_PN2D(parentNode), HMIBASE_TO_STRING_VW(this)));

                  Candera::BaseLineLayouter::SetBaseLineOffset(*parentNode, GetMultipleLinesOffset());
                  InvalidateLayout();
               }
               break;
            default:
               break;
         }
      }
   }
}
