/* ***************************************************************************************
* FILE:          TextAreaWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextAreaWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "TextAreaWidget2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateMap.h"
#include <Widgets/2D/BaseWidget2D.h>
#include "Candera/TextEngine/TextLayoutStrategy.h"

#define BOUNDING_RECTANGLE_WITH_TEXTRENDER   // for CGI 3.2, 3.3 and deprecate warning for CGI 3.1

using namespace Candera;

const Candera::UInt8 TextAreaWidget2D::c_Max_Text_Params = 6;
const Candera::TChar TextAreaWidget2D::c_Text_Param_Mark = '%';

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextAreaWidget2D.cpp.trc.h"
#endif
#include <Candera/TextEngine/GlyphTextMeasureContext.h>


CGI_WIDGET_RTTI_DEFINITION(TextAreaWidget2D);


/****************************************************************************
*    Function    : TextAreaWidget2D
*    Description : Constructor
*    Parameters  : void
****************************************************************************/
TextAreaWidget2D::TextAreaWidget2D() :
   m_oParameterMark(&c_Text_Param_Mark, 1/*length*/),
   m_bIsScrollEnabled(false),
   m_flScrollOffset(0.0F),
   m_u32ScrollStartTime(0),
   m_effectiveNumberOfLines(0),
   m_layoutingRect(),
   m_basePoint(0.0f, 0.0f)
{
   // Set non default initial values for properties
   SetTruncationText("...");
   SetEnabled(true);
   SetDisabledTextColor(Color(0.5F, 0.5F, 0.5F, 1.0F));
   SetHighlightTextColor(Color(1.0F, 0.0F, 0.0F, 1.0F));
   SetActiveTextColor(Color(0.2F, 0.5F, 1.0F, 1.0F));
   SetScrollSpeed(100);
   SetScrollStartDelay(1000);
   SetScrollEndDelay(1000);
   // create chunks
   m_pNormalColorChunk1 = FEATSTD_NEW(RichText::ColorChunk);
   m_pNormalColorChunk2 = FEATSTD_NEW(RichText::ColorChunk);
   m_pHighlightColorChunk = FEATSTD_NEW(RichText::ColorChunk);
   m_pTextChunk1 = FEATSTD_NEW(RichText::TextChunk);
   m_pTextChunk2 = FEATSTD_NEW(RichText::TextChunk);
   m_pTextChunk3 = FEATSTD_NEW(RichText::TextChunk);
   // add chunks to richtext
   m_oRichText.vAddChunk(m_pNormalColorChunk1);
   m_oRichText.vAddChunk(m_pTextChunk1);
   m_oRichText.vAddChunk(m_pHighlightColorChunk);
   m_oRichText.vAddChunk(m_pTextChunk2);
   m_oRichText.vAddChunk(m_pNormalColorChunk2);
   m_oRichText.vAddChunk(m_pTextChunk3);
}


/****************************************************************************
*    Function    : ~TextAreaWidget2D
*    Description : Destructor
*    Parameters  : void
****************************************************************************/
TextAreaWidget2D::~TextAreaWidget2D()
{
   FEATSTD_LINT_CURRENT_SCOPE(423, "Chunks will be freed by BaseClass::m_oRichText destructor")
   m_pNormalColorChunk1 = 0;
   m_pNormalColorChunk2 = 0;
   m_pHighlightColorChunk = 0;
   m_pTextChunk1 = 0;
   m_pTextChunk2 = 0;
   m_pTextChunk3 = 0;
}


class FixedNumberOfLinesLayoutStrategy : public Candera::TextRendering::TextLayoutStrategy
{
   public:
      FixedNumberOfLinesLayoutStrategy(const Char* text, Int16 numberOfLines) : m_currentLine(0), m_numberOfLines(numberOfLines), m_text(text), m_cursor(text), m_lastCursor(text), m_isTruncated(false)
      {
      }
      virtual Action OnInit(Pass pass)
      {
         FEATSTD_UNUSED(pass);
         m_currentLine = 0;
         m_cursor = 0;
         m_lastCursor = m_text;
         m_cursor = m_text;
         m_isTruncated = false;
         return (m_numberOfLines > 0) ? Continue : Break;
      }

      virtual Action OnLineBreak(const Char* textCursor, LineBreakType type)
      {
         FEATSTD_UNUSED(type);
         if (m_currentLine > m_numberOfLines)
         {
            m_isTruncated = true;
            return Break;
         }

         if (m_numberOfLines > m_currentLine)
         {
            m_lastCursor = m_cursor;
            m_currentLine++;
            m_cursor = textCursor;
         }

         if (m_numberOfLines == m_currentLine)
         {
            m_currentLine++;
            m_cursor = textCursor;
         }

         return Continue;
      }

      bool IsTruncated() const
      {
         return m_isTruncated;
      }

      const Char* GetLastCursor() const
      {
         return m_lastCursor;
      }

      const Char* GetCursor() const
      {
         return m_cursor;
      }
      FeatStd::Int16 GetEffectiveNumberOfLines()
      {
         return (true == m_isTruncated || (0 == strlen(m_cursor))) ? static_cast<FeatStd::Int16>(m_currentLine - 1) : m_currentLine;
      }
   private:
      FeatStd::Int16 m_currentLine;
      FeatStd::Int16 m_numberOfLines;
      const Char* m_text;
      const Char* m_cursor;
      const Char* m_lastCursor;
      bool m_isTruncated;
      FEATSTD_MAKE_CLASS_UNCOPYABLE(FixedNumberOfLinesLayoutStrategy);
};


/****************************************************************************
*    Function    : InitWidget
*    Description : Called when widget is initialized. Overridden from
*                  WidgetBase.
*    Parameters  : Candera::AssetProvider*
*    Return      : void
****************************************************************************/
void TextAreaWidget2D::InitWidget()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextAreaWidget2D: [%s]. InitWidget called.", GetName()));
   Base::InitWidget();
}


/****************************************************************************
*    Function    : Update
*    Description : Called on for each render cycle. Overridden from
*                  WidgetBase.
*    Parameters  : void
*    Return      : void
****************************************************************************/
void TextAreaWidget2D::Update()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextAreaWidget2D: [%s]. Update called.", GetName()));
   Base::Update();
}


/****************************************************************************
*    Function    : CultureChanged
*    Description : Called on Change of culture.
*    Parameters  : void
*    Return      : void
****************************************************************************/
void TextAreaWidget2D::CultureChanged()
{
   m_bIsRelayout = true;
   Invalidate();
}


/****************************************************************************
*    Function    : OnMessage
*    Description : Handles message processing. Overridden from
*                  Framework Widget.
*    Parameters  : const Courier::Message&
*    Return      : bool
****************************************************************************/
bool TextAreaWidget2D::OnMessage(const Courier::Message& msg)
{
   bool bIsProcessed = false;
   bIsProcessed = Base::OnMessage(msg);
   _TODO("Check ScrollMsg in CGI 3")
   //PARAM_UNUSED(msg);

   return bIsProcessed;
}


/****************************************************************************
*     Function    : CloneFrom
*     Description : required for FlexList cloning
****************************************************************************/
bool TextAreaWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const TextAreaWidget2D* original = CLONEABLE_WIDGET_CAST<const TextAreaWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetTextWidgetMode(original->GetTextWidgetMode());
      SetItemOrder(original->GetItemOrder());
      SetDisabledTextColor(original->GetDisabledTextColor());
      SetHighlightTextColor(original->GetHighlightTextColor());
      SetActiveTextColor(original->GetActiveTextColor());
      SetScrollSpeed(original->GetScrollSpeed());
      SetScrollStartDelay(original->GetScrollStartDelay());
      SetScrollEndDelay(original->GetScrollEndDelay());
      SetHighlightMode(original->GetHighlightMode());
      SetHighlightStartIndex(original->GetHighlightStartIndex());
      SetHighlightCount(original->GetHighlightCount());
      SetHighlightText(original->GetHighlightText());
      SetTruncationMode(original->GetTruncationMode());
      SetTruncationText(original->GetTruncationText());
      SetEnableDynamicText(original->GetEnableDynamicText());
      SetDisabledTextColor(original->GetDisabledTextColor());
      SetHighlightTextColor(original->GetHighlightTextColor());
      SetActiveTextColor(original->GetActiveTextColor());
      SetItemOrder(original->GetItemOrder());
      SetScrollSpeed(original->GetScrollSpeed());
      SetScrollStartDelay(original->GetScrollStartDelay());
      SetScrollEndDelay(original->GetScrollEndDelay());
      SetTextParameter1(original->GetTextParameter1());
      SetTextParameter2(original->GetTextParameter2());
      SetTextParameter3(original->GetTextParameter3());
      SetTextParameter4(original->GetTextParameter4());
      SetTextParameter5(original->GetTextParameter5());
      SetTextParameter6(original->GetTextParameter6());
      SetMultiLineLayouting(original->GetMultiLineLayouting());
      SetTextWrapMode(original->GetTextWrapMode());
      SetLinespacingFactor(original->GetLinespacingFactor());
      SetMaximumNumberOfLines(original->GetMaximumNumberOfLines());
      cloned = true;
   }
   return cloned;
}


/****************************************************************************
*     Function    : OnChanged
*     Description : Called from the Update function for each property that
*                   has been changed. Overridden from RichTextWidgetBase.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::OnChanged(Candera::UInt32 propertyId)
{
   switch (propertyId)
   {
      case TruncationModePropertyId:
      case TruncationTextPropertyId:
      case HighlightModePropertyId:
      case HighlightStartIndexPropertyId:
      case HighlightCountPropertyId:
      case HighlightTextPropertyId:
      case EnableDynamicTextPropertyId:
      case TextParameter1PropertyId:
      case TextParameter2PropertyId:
      case TextParameter3PropertyId:
      case TextParameter4PropertyId:
      case TextParameter5PropertyId:
      case TextParameter6PropertyId:
         m_bIsRelayout = true;
         break;

      case EnabledPropertyId:
      case TextWidgetModePropertyId:
      case DisabledTextColorPropertyId:
      case HighlightTextColorPropertyId:
      case ActiveTextColorPropertyId:
         m_bIsRerender = true;
         break;

      case ItemOrderPropertyId:
      case ScrollSpeedPropertyId:
      case ScrollStartDelayPropertyId:
      case ScrollEndDelayPropertyId:
         break;

      default:
         Base::OnChanged(propertyId);
   }

   Invalidate();
}


/****************************************************************************
*     Function    : bTruncateText
*     Description : Perform text truncation based on given parameters
*     Parameters  : const FeatStd::String&, FeatStd::String& sDestString, TruncationMode enTruncationMode
*     Return      : bool
****************************************************************************/
bool TextAreaWidget2D::bTruncateText(const FeatStd::String& sSrcString, FeatStd::String& sDestString, TruncationMode enTruncationMode) const
{
   const Candera::Vector2 oTextAreaSize = GetTextAreaSize();
   bool bIsTextChanged = false;

   // perform no truncation for text area sizes < 1
   if (Int32(oTextAreaSize.GetX() + 0.5F) < 1)
   {
      sDestString = sSrcString;
      return bIsTextChanged;
   }

   switch (enTruncationMode)
   {
      case SoftTruncation:
      {
         // calculate original text size
         TextRendering::LayoutingOptions layoutingOptions;
         FixedNumberOfLinesLayoutStrategy layoutStrategy(sSrcString.GetCString(), GetMaximumNumberOfLines());
         if (true == GetMultiLineLayouting())
         {
            layoutingOptions.SetMultilineTextEnabled(GetMultiLineLayouting());
            layoutingOptions.SetWordWrapEnabled(!(RT_WrapNone == GetTextWrapMode()));
            Candera::TextRendering::PixelSize lineHeight = Candera::TextRendering::PixelSize((!GetStyle().PointsToNull()) ? GetStyle()->GetMetrics().lineHeight * GetLinespacingFactor() : 1);
            layoutingOptions.SetLineSpacing(lineHeight);
            layoutingOptions.SetSize(Candera::TextRendering::TextSize((Candera::Int16)GetTextAreaSize().GetX(), (Candera::Int16)GetTextAreaSize().GetY()));
            layoutingOptions.SetLayoutStrategy(&layoutStrategy);
         }
         else
         {
            FEATSTD_UNUSED(layoutStrategy);
         }
#ifdef BOUNDING_RECTANGLE_WITH_TEXTRENDER
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
         TextRendering::GlyphTextMeasureContext context(0);
         TextRendering::CursorTextMeasureContext cursorContext((!GetStyle().PointsToNull()) ? *GetStyle() : *StyleType());
         context.SetNextContext(&cursorContext);
         TextRendering::TextRenderer().Render(context, /*TextRendering::LayoutingOptions()*/layoutingOptions, TextRendering::ShapingOptions((!GetStyle().PointsToNull()) ? GetStyle() : StyleType()), TextRendering::TextProperties(sSrcString.GetCString()));
         TextRendering::PixelPosition i16Width = context.GetTextRectangle().GetWidth();

         m_layoutingRect.SetLeft(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetLeft()));
         m_layoutingRect.SetTop(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetTop()));
         m_layoutingRect.SetWidth(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetWidth()));
         m_layoutingRect.SetHeight(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetHeight()));
         if (GetMultiLineLayouting())
         {
            m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
         }
#else
         TextRendering::MeasuringOptions oMeasuringOptions(TextRendering::MeasuringOptions::DefaultFinalAdvance, TextRendering::MeasuringOptions::DefaultTransversalSize);

         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
         TextRendering::PixelPosition i16Width = TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(GetStyle()),
                                                 TextRendering::TextProperties(sSrcString.GetCString()), oMeasuringOptions).GetWidth();
         if (GetMultiLineLayouting())
         {
            m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
         }
#endif
         if (false == GetMultiLineLayouting())
         {
            if (i16Width > Int16(oTextAreaSize.GetX()))
            {
               // if text doesn't fit perform a binary search to find the the amount of codepoints still fitting into the text rectangle
               const TChar* pText = sSrcString.GetCString();
               UInt32 u32Start = 0;
               UInt32 u32End = sSrcString.GetCodePointCount();
               UInt32 u32Pos = 0;

               while ((u32End - u32Start) > 1)
               {
                  // calculate next codepoint count that should be checked
                  u32Pos = (u32Start + u32End) / 2;
                  // check width and shift the search boundaries accordingly

#ifdef BOUNDING_RECTANGLE_WITH_TEXTRENDER
                  TextRendering::GlyphTextMeasureContext context(0);
                  TextRendering::TextRenderer().Render(context, /*TextRendering::LayoutingOptions()*/layoutingOptions, TextRendering::ShapingOptions(GetStyle()), TextRendering::TextProperties(pText, TextRendering::TextLength(u32Pos)));
                  i16Width = context.GetTextRectangle().GetWidth();
#else
                  i16Width = TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(GetStyle()),
                             TextRendering::TextProperties(pText, TextRendering::TextLength(u32Pos)), oMeasuringOptions).GetWidth();
#endif
                  if (i16Width > Int16(oTextAreaSize.GetX()))
                  {
                     u32End = u32Pos;
                  }
                  else
                  {
                     u32Start = u32Pos;
                  }
               }
               // create the final string based upon the biggest (u32Start) fitting codepoint count found
               sDestString = FeatStd::String(pText, u32Start);
               bIsTextChanged = true;
            }
            else
            {
               // use original string
               sDestString = FeatStd::String(sSrcString.GetCString());
            }
         }
         else
         {
            Candera::String remString = Candera::String(layoutStrategy.GetCursor());
            FeatStd::SizeType u32RemTextSize = remString.GetCharCount();
            if (u32RemTextSize > 0)
            {
               FeatStd::SizeType u32TextSize = sSrcString.GetCharCount();
               FeatStd::SizeType charsToFit = u32TextSize - u32RemTextSize + 1;           // Copy till the characters where line break is present
               TChar* pTextBuffer = FEATSTD_NEW_ARRAY(TChar, charsToFit);
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               RichText::Utf8Encoding::u32Copy(pTextBuffer, static_cast<UInt32>(charsToFit), sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();

               sDestString = FeatStd::String(pTextBuffer);
               // cleanup of temporary buffer
               FEATSTD_DELETE_ARRAY(pTextBuffer);
            }
            else
            {
               // use the original string
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               sDestString = FeatStd::String(sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();
            }
         }
         SECURE_FEATSTD_STRING_ACCESS_END();
      }
      break;

      case TextTruncation:
      {
         const FeatStd::String oTruncationText = GetTruncationText();
         // calculate original text size
         TextRendering::LayoutingOptions layoutingOptions;
         FixedNumberOfLinesLayoutStrategy layoutStrategy(sSrcString.GetCString(), GetMaximumNumberOfLines());
         if (true == GetMultiLineLayouting())
         {
            layoutingOptions.SetMultilineTextEnabled(GetMultiLineLayouting());
            layoutingOptions.SetWordWrapEnabled(!(RT_WrapNone == GetTextWrapMode()));
            Candera::TextRendering::PixelSize lineHeight = Candera::TextRendering::PixelSize((!GetStyle().PointsToNull()) ? GetStyle()->GetMetrics().lineHeight * GetLinespacingFactor() : 1);
            layoutingOptions.SetLineSpacing(lineHeight);
            layoutingOptions.SetSize(Candera::TextRendering::TextSize((Candera::Int16)GetTextAreaSize().GetX(), (Candera::Int16)GetTextAreaSize().GetY()));
            layoutingOptions.SetLayoutStrategy(&layoutStrategy);
         }
         else
         {
            FEATSTD_UNUSED(layoutStrategy);
         }
#ifdef BOUNDING_RECTANGLE_WITH_TEXTRENDER
         TextRendering::PixelPosition i16Width;
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
         TextRendering::GlyphTextMeasureContext context(0);
         TextRendering::CursorTextMeasureContext cursorContext((!GetStyle().PointsToNull()) ? *GetStyle() : *StyleType());
         context.SetNextContext(&cursorContext);

         TextRendering::TextRenderer().Render(context, /*TextRendering::LayoutingOptions()*/layoutingOptions, TextRendering::ShapingOptions((!GetStyle().PointsToNull()) ? GetStyle() : StyleType()), TextRendering::TextProperties(sSrcString.GetCString()));
         i16Width = context.GetTextRectangle().GetWidth();

         m_layoutingRect.SetLeft(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetLeft()));
         m_layoutingRect.SetTop(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetTop()));
         m_layoutingRect.SetWidth(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetWidth()));
         m_layoutingRect.SetHeight(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetHeight()));

         if (GetMultiLineLayouting())
         {
            m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
         }
#else
         TextRendering::MeasuringOptions oMeasuringOptions(TextRendering::MeasuringOptions::DefaultFinalAdvance, TextRendering::MeasuringOptions::DefaultTransversalSize);

         TextRendering::PixelPosition i16Width;
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
         i16Width = TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(GetStyle()),
                    TextRendering::TextProperties(sSrcString.GetCString()), oMeasuringOptions).GetWidth();
         if (GetMultiLineLayouting())
         {
            m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
         }
#endif
         SECURE_FEATSTD_STRING_ACCESS_END();

         if (false == GetMultiLineLayouting())
         {
            if (i16Width > Int16(oTextAreaSize.GetX()))
            {
               // if text doesn't fit perform a binary search to find the the amount of codepoints still fitting into the text rectangle while also
               // considering the truncation text
               FeatStd::SizeType u32TextBufferSize = sSrcString.GetCharCount() + GetTruncationText().GetCharCount() + 1;
               TChar* pTextBuffer = FEATSTD_NEW_ARRAY(TChar, u32TextBufferSize);

               if (pTextBuffer != 0)
               {
                  UInt32 u32Start = 0;
                  UInt32 u32End = sSrcString.GetCodePointCount();
                  UInt32 u32Pos = 0;

                  while ((u32End - u32Start) > 1)
                  {
                     // calculate next codepoint count that should be checked
                     u32Pos = (u32Start + u32End) / 2;
                     // create a temporary string containing the the original plus the truncation text
                     SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
                     RichText::Utf8Encoding::u32Copy(pTextBuffer, static_cast<UInt32>(u32TextBufferSize), sSrcString.GetCString());
                     SECURE_FEATSTD_STRING_ACCESS_END();
                     TChar* pInsertPos = pTextBuffer;
                     const TChar* copInsertPos = const_cast<const TChar*>(pInsertPos);

                     for (UInt32 u32Index = 0; u32Index < u32Pos; u32Index++)
                     {
                        RichText::Utf8Encoding::u32Advance(copInsertPos);
                     }

                     pInsertPos = const_cast<TChar*>(copInsertPos);
                     SECURE_FEATSTD_STRING_ACCESS_BEGIN(oTruncationText);
                     RichText::Utf8Encoding::u32Copy(pInsertPos, static_cast<UInt32>(u32TextBufferSize - u32Pos), oTruncationText.GetCString());
                     SECURE_FEATSTD_STRING_ACCESS_END();
                     // check width and shift the search boundaries accordingly

#ifdef BOUNDING_RECTANGLE_WITH_TEXTRENDER
                     TextRendering::GlyphTextMeasureContext context(0);
                     TextRendering::TextRenderer().Render(context, /*TextRendering::LayoutingOptions()*/layoutingOptions, TextRendering::ShapingOptions(GetStyle()), TextRendering::TextProperties(pTextBuffer));
                     i16Width = context.GetTextRectangle().GetWidth();

#else
                     i16Width = TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(GetStyle()),
                                TextRendering::TextProperties(pTextBuffer), oMeasuringOptions).GetWidth();
#endif
                     if (i16Width > Int16(oTextAreaSize.GetX()))
                     {
                        u32End = u32Pos;
                     }
                     else
                     {
                        u32Start = u32Pos;
                     }
                  }

                  // create the final string based upon the biggest (u32Start) fitting codepoint count found
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
                  RichText::Utf8Encoding::u32Copy(pTextBuffer, static_cast<UInt32>(u32TextBufferSize), sSrcString.GetCString());
                  SECURE_FEATSTD_STRING_ACCESS_END();
                  TChar* pInsertPos = pTextBuffer;
                  const TChar* copInsertPos = const_cast<const TChar*>(pInsertPos);

                  for (UInt32 u32Index = 0; u32Index < u32Start; u32Index++)
                  {
                     RichText::Utf8Encoding::u32Advance(copInsertPos);
                  }

                  pInsertPos = const_cast<TChar*>(copInsertPos);
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(oTruncationText);
                  RichText::Utf8Encoding::u32Copy(pInsertPos, static_cast<UInt32>(u32TextBufferSize - u32Start), oTruncationText.GetCString());
                  SECURE_FEATSTD_STRING_ACCESS_END()
                  sDestString = FeatStd::String(pTextBuffer);
                  // cleanup of temporary buffer
                  FEATSTD_DELETE_ARRAY(pTextBuffer);
                  bIsTextChanged = true;
               }
               else
               {
                  // use the original string
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
                  sDestString = FeatStd::String(sSrcString.GetCString());
                  SECURE_FEATSTD_STRING_ACCESS_END();
               }
            }
            else
            {
               // use the original string
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               sDestString = FeatStd::String(sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();
            }
         }
         else
         {
            Candera::String remString = Candera::String(layoutStrategy.GetCursor());
            FeatStd::SizeType u32RemTextSize = remString.GetCharCount();
            if (u32RemTextSize > 0)
            {
               FeatStd::SizeType u32TextSize = sSrcString.GetCharCount();
               FeatStd::SizeType charsToFit = u32TextSize - u32RemTextSize - GetTruncationText().GetCharCount();           // space for truncation text, taken roughly ... TODO to improve this
               TChar* pTextBuffer = FEATSTD_NEW_ARRAY(TChar, charsToFit + GetTruncationText().GetCharCount() + 1);
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               RichText::Utf8Encoding::u32Copy(pTextBuffer, static_cast<UInt32>(charsToFit), sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();

               TChar* pInsertPos = pTextBuffer;
               const TChar* copInsertPos = const_cast<const TChar*>(pInsertPos);

               for (UInt32 u32Index = 0; u32Index <= charsToFit; u32Index++)
               {
                  RichText::Utf8Encoding::u32Advance(copInsertPos);
               }

               pInsertPos = const_cast<TChar*>(copInsertPos);
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(oTruncationText);
               RichText::Utf8Encoding::u32Copy(pInsertPos, static_cast<UInt32>(GetTruncationText().GetCharCount() + 1), oTruncationText.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();
               sDestString = FeatStd::String(pTextBuffer);
               // cleanup of temporary buffer
               FEATSTD_DELETE_ARRAY(pTextBuffer);
            }
            else
            {
               // use the original string
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               sDestString = FeatStd::String(sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();
            }
         }
      }

      break;

      case HardTruncation: // fall through
      {
         if (true == GetMultiLineLayouting())
         {
            TextRendering::LayoutingOptions layoutingOptions;
            FixedNumberOfLinesLayoutStrategy layoutStrategy(sSrcString.GetCString(), GetMaximumNumberOfLines());
            layoutingOptions.SetMultilineTextEnabled(GetMultiLineLayouting());
            layoutingOptions.SetWordWrapEnabled(!(RT_WrapNone == GetTextWrapMode()));
            Candera::TextRendering::PixelSize lineHeight = Candera::TextRendering::PixelSize((!GetStyle().PointsToNull()) ? GetStyle()->GetMetrics().lineHeight * GetLinespacingFactor() : 1);
            layoutingOptions.SetLineSpacing(lineHeight);
            layoutingOptions.SetSize(Candera::TextRendering::TextSize((Candera::Int16)GetTextAreaSize().GetX(), (Candera::Int16)GetTextAreaSize().GetY()));
            layoutingOptions.SetLayoutStrategy(&layoutStrategy);

#ifdef BOUNDING_RECTANGLE_WITH_TEXTRENDER
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
            TextRendering::GlyphTextMeasureContext context(0);
            TextRendering::CursorTextMeasureContext cursorContext((!GetStyle().PointsToNull()) ? *GetStyle() : *StyleType());
            context.SetNextContext(&cursorContext);
            TextRendering::TextRenderer().Render(context, /*TextRendering::LayoutingOptions()*/layoutingOptions, TextRendering::ShapingOptions((!GetStyle().PointsToNull()) ? GetStyle() : StyleType()), TextRendering::TextProperties(sSrcString.GetCString()));
            //TextRendering::PixelPosition i16Width = context.GetTextRectangle().GetWidth();

            m_layoutingRect.SetLeft(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetLeft()));
            m_layoutingRect.SetTop(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetTop()));
            m_layoutingRect.SetWidth(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetWidth()));
            m_layoutingRect.SetHeight(static_cast<FeatStd::Float>(cursorContext.GetTextRectangle().GetHeight()));

            if (GetMultiLineLayouting())
            {
               m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
            }
#else
            TextRendering::MeasuringOptions oMeasuringOptions(TextRendering::MeasuringOptions::DefaultFinalAdvance, TextRendering::MeasuringOptions::DefaultTransversalSize);

            SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
            TextRendering::PixelPosition i16Width = TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(GetStyle()),
                                                    TextRendering::TextProperties(sSrcString.GetCString()), oMeasuringOptions).GetWidth();
            if (GetMultiLineLayouting())
            {
               m_effectiveNumberOfLines = layoutStrategy.GetEffectiveNumberOfLines();
            }
#endif
            SECURE_FEATSTD_STRING_ACCESS_END();
            Candera::String remString = Candera::String(layoutStrategy.GetCursor());
            FeatStd::SizeType u32RemTextSize = remString.GetCharCount();
            if (u32RemTextSize > 0)
            {
               FeatStd::SizeType u32TextSize = sSrcString.GetCharCount();
               FeatStd::SizeType charsToFit = u32TextSize - u32RemTextSize + 1;           // Copy till the characters where line break is present
               TChar* pTextBuffer = FEATSTD_NEW_ARRAY(TChar, charsToFit);
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
               RichText::Utf8Encoding::u32Copy(pTextBuffer, static_cast<UInt32>(charsToFit), sSrcString.GetCString());
               SECURE_FEATSTD_STRING_ACCESS_END();

               sDestString = FeatStd::String(pTextBuffer);
               // cleanup of temporary buffer
               FEATSTD_DELETE_ARRAY(pTextBuffer);
            }
         }
         else
         {
            // use the original string
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
            sDestString = FeatStd::String(sSrcString.GetCString());
            SECURE_FEATSTD_STRING_ACCESS_END();
         }
      }
      break;
      default:
         // return original string
         sDestString = sSrcString;
   }

   return bIsTextChanged;
}


/****************************************************************************
*     Function    : vCalculateScrollOffset
*     Description : Calculate the offset of the text when scrolling
*     Parameters  : void
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vCalculateScrollOffset()
{
   UInt16 u16ScrollSpeed = GetScrollSpeed();

   if ((GetAnimationTimeDispatcher() != 0) && (u16ScrollSpeed > 0) && (m_pTextArea != 0))
   {
      UInt16 u16ScrollStartDelay = GetScrollStartDelay();
      UInt16 u16ScrollEndDelay = GetScrollEndDelay();
      RichText::Paragraph* pParagraph = m_pTextArea->pGetParagraph(0);

      if ((pParagraph != 0) && (pParagraph->cooGetContentSize().GetWidth() > m_pTextArea->cooGetEffectiveSize().GetWidth()))
      {
         UInt32 u32Now = GetAnimationTimeDispatcher()->GetWorldTimeMs();

         // set the scroll start time if not set signaling a new start
         if (m_u32ScrollStartTime == 0)
         {
            m_u32ScrollStartTime = GetAnimationTimeDispatcher()->GetWorldTimeMs();
         }

         // Calculate time delta from scroll start till now
         UInt32 u32CurrentScrollTime = u32Now - m_u32ScrollStartTime;

         if (u32CurrentScrollTime > u16ScrollStartDelay)
         {
            // after initial delay calculate how far into the scrolling has progressed
            UInt32 u32ActiveScrollTime = u32CurrentScrollTime - u16ScrollStartDelay;
            UInt32 u32MaxScrollTime = UInt32((Float(pParagraph->cooGetContentSize().GetWidth() - m_pTextArea->cooGetEffectiveSize().GetWidth()) / Float(u16ScrollSpeed)) * 1000.0F);

            if (u32ActiveScrollTime < u32MaxScrollTime)
            {
               // if scrolling is in its 'active' phase calculate the scrolling offset based on the current scrolling progress
               m_flScrollOffset = Float(u32ActiveScrollTime) * (Float(u16ScrollSpeed) / 1000.0F);
               m_bIsRerender = true;
            }
            else
            {
               if (u32ActiveScrollTime > (u32MaxScrollTime + u16ScrollEndDelay))
               {
                  //if scrolling has ended reset to 'default' state
                  m_u32ScrollStartTime = 0;
                  m_flScrollOffset = 0.0F;
                  m_bIsScrollEnabled = false;
                  m_bIsRerender = true;
               }
               else
               {
                  // if scrolling reached the end delay phase set final (maximum) scroll offset
                  if (m_flScrollOffset != Float(pParagraph->cooGetContentSize().GetWidth() - m_pTextArea->cooGetEffectiveSize().GetWidth()))
                  {
                     m_flScrollOffset = Float(pParagraph->cooGetContentSize().GetWidth() - m_pTextArea->cooGetEffectiveSize().GetWidth());
                     m_bIsRerender = true;
                  }
               }
            }
         }

         // TODO: without permanent rerendering the update function might not be called again and the scrolling won't progress.
         //       to save performance when the scrolling is in the start/end delay phase a timer should be used instead.
         m_bIsRerender = true;
      }
   }
};


/****************************************************************************
*     Function    : vFindString
*     Description : Search for the start position and character count of
*                   the given sub string in the text.
*     Parameters  : const FeatStd::String&, const FeatStd::String&, Candera::UInt16&, Candera::UInt16&
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vFindString(const FeatStd::String& sText, const FeatStd::String& sSubStr, UInt16& u16StartIndex, UInt16& u16Count) const
{
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(sText);
   const TChar* copText = sText.GetCString();
   const UInt16 u16TextLength = UInt16(sText.GetCodePointCount());
   const UInt16 u16SearchLength = UInt16(sSubStr.GetCodePointCount());
   const UInt16 u16SearchCharCount = UInt16(sSubStr.GetCharCount());

   for (UInt16 u16SearchIndex = 0; u16SearchIndex < u16TextLength; u16SearchIndex++)
   {
      int iRes;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSubStr);
      iRes  = memcmp(copText, sSubStr.GetCString(), u16SearchCharCount);
      SECURE_FEATSTD_STRING_ACCESS_END();
      if (iRes == 0)
      {
         u16StartIndex = u16SearchIndex;
         u16Count = u16SearchLength;
         return;
      }

      RichText::Utf8Encoding::u32Advance(copText);
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   u16StartIndex = 0;
   u16Count = 0;
}


/****************************************************************************
*     Function    : vDoProcessDynamicText
*     Description : Processes the text and substitutes each %x with
*                   the according text parameter
*     Parameters  : const FeatStd::String&, const FeatStd::String&
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vDoProcessDynamicText(const FeatStd::String& sSrcString, FeatStd::String& sDestString) const
{
   std::vector<UInt32> oInsertPositions;
   const TChar* pchOriginalString;
   UInt32 u32CurrentCodePoint = 0;
   UInt32 u32CompleteStringSize = UInt32(sSrcString.GetCharCount() + 1);
   // create text parameter table
   const FeatStd::String* pParameterStrings[c_Max_Text_Params + 1];
   pParameterStrings[0] = &GetTextParameter1();
   pParameterStrings[1] = &GetTextParameter2();
   pParameterStrings[2] = &GetTextParameter3();
   pParameterStrings[3] = &GetTextParameter4();
   pParameterStrings[4] = &GetTextParameter5();
   pParameterStrings[5] = &GetTextParameter6();
   pParameterStrings[6] = &m_oParameterMark;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
   pchOriginalString = sSrcString.GetCString();

   // IMPROVEMENT: use char based approach instead of codepoints to improve performance

   // loop through string to find all insert positions and also calculate the complete string size
   while (pchOriginalString[0] != TChar(0))
   {
      if (pchOriginalString[0] == TChar(c_Text_Param_Mark))
      {
         // check if the following value is within specified range
         if ((pchOriginalString[1] >= TChar('1')) && (pchOriginalString[1] <= TChar('0' + c_Max_Text_Params)))
         {
            UInt32 u32ParamIndex = UInt32(pchOriginalString[1] - TChar('1'));
            // add position then parameter index
            oInsertPositions.push_back(u32CurrentCodePoint);
            oInsertPositions.push_back(u32ParamIndex);
            RichText::Utf8Encoding::u32Advance(pchOriginalString);
            u32CurrentCodePoint++;
            u32CompleteStringSize += UInt32(pParameterStrings[u32ParamIndex]->GetCharCount());
         }
         else
         {
            // check for another marker, if true replace the two marker with one
            if (pchOriginalString[1] == TChar(c_Text_Param_Mark))
            {
               // add position then parameter index
               oInsertPositions.push_back(u32CurrentCodePoint);
               oInsertPositions.push_back(c_Max_Text_Params);
               RichText::Utf8Encoding::u32Advance(pchOriginalString);
               u32CurrentCodePoint++;
               u32CompleteStringSize++;
            }
         }
      }

      RichText::Utf8Encoding::u32Advance(pchOriginalString);
      u32CurrentCodePoint++;
   }
   SECURE_FEATSTD_STRING_ACCESS_END();

   // allocate the complete string
   TChar* pchCompleteString = FEATSTD_NEW_ARRAY(TChar, u32CompleteStringSize);

   if (pchCompleteString != 0)
   {
      pchCompleteString[u32CompleteStringSize - 1] = TChar(0);
      // create complete string by copying all string segments
      TChar* pchDestString = pchCompleteString;
      UInt32 u32RemainingStringSize = u32CompleteStringSize;
      UInt32 u32ProcessedCount = 0;
      const UInt uInsertPositionCount = UInt(oInsertPositions.size());
      UInt32 u32ByteCount;

      SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSrcString);
      pchOriginalString = sSrcString.GetCString();

      for (UInt uIndex = 0; uIndex < uInsertPositionCount; uIndex += 2)
      {
         // copy segment before the text parameter
         u32ByteCount = RichText::Utf8Encoding::u32Copy(pchDestString, u32RemainingStringSize, pchOriginalString, oInsertPositions[uIndex] - u32ProcessedCount);
         pchDestString = &pchDestString[u32ByteCount];
         pchOriginalString = &pchOriginalString[u32ByteCount];
         u32RemainingStringSize -= u32ByteCount;
         // copy text parameter
         UInt32 u32TextParameterIndex = oInsertPositions[uIndex + 1];
         u32ByteCount = RichText::Utf8Encoding::u32Copy(pchDestString, u32RemainingStringSize, pParameterStrings[u32TextParameterIndex]->GetCString());
         pchDestString = &pchDestString[u32ByteCount];
         pchOriginalString = &pchOriginalString[2];
         u32RemainingStringSize -= u32ByteCount;
         u32ProcessedCount = oInsertPositions[uIndex] + 2;
      }

      // copy remaining string
      u32ByteCount = RichText::Utf8Encoding::u32Copy(pchDestString, u32RemainingStringSize, pchOriginalString);
      SECURE_FEATSTD_STRING_ACCESS_END();

      FEATSTD_DEBUG_ASSERT(u32ByteCount == u32RemainingStringSize - ((oInsertPositions.size() * 2 / 2) + 1));
      FEATSTD_UNUSED(u32ByteCount);
      // set new string
      sDestString = pchCompleteString;
      // cleanup
      FEATSTD_DELETE_ARRAY(pchCompleteString);
   }
   else
   {
      sDestString = sSrcString;
   }
}


/****************************************************************************
*     Function    : vUpdateNode
*     Description : Update all node related properties here
*     Parameters  : void
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vUpdateNode()
{
   Base::vUpdateNode();
}


/****************************************************************************
*     Function    : bIsReLayoutNeeded
*     Description : Checks if relayouting is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool TextAreaWidget2D::bIsReLayoutNeeded()
{
   return Base::bIsReLayoutNeeded();
}


/****************************************************************************
*     Function    : bIsReRenderNeeded
*     Description : Checks if rerendering is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool TextAreaWidget2D::bIsReRenderNeeded()
{
   if (m_bIsScrollEnabled)
   {
      vCalculateScrollOffset();
   }

   return Base::bIsReRenderNeeded() || m_bIsScrollEnabled;
}


/****************************************************************************
*     Function    : vLayoutRichText
*     Description : Will do the layouting of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after layouting.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vLayoutRichText()
{
   const FeatStd::String* pInputString = &GetText();
   // insert dynamic texts
   FeatStd::String sDynamicString;

   if (GetEnableDynamicText())
   {
      vDoProcessDynamicText(*pInputString, sDynamicString);
      pInputString = &sDynamicString;
   }

   // Calculate highlighting start position and character count
   UInt16 u16HighlightStartIndex = GetHighlightStartIndex();
   UInt16 u16HighlightCount = GetHighlightCount();

   if (GetHighlightMode() == TextMode)
   {
      vFindString(*pInputString, GetHighlightText(), u16HighlightStartIndex, u16HighlightCount);
   }

   // truncate string
   FeatStd::String sTruncatedString;
   bool bIsTruncated;

   if (m_bIsScrollEnabled == true)
   {
      bIsTruncated = bTruncateText(*pInputString, sTruncatedString, HardTruncation);
   }
   else
   {
      bIsTruncated = bTruncateText(*pInputString, sTruncatedString, GetTruncationMode());
   }

   // prevent highlighting of the truncation text
   if ((GetTruncationMode() == TextTruncation) && (bIsTruncated))
   {
      UInt16 u16CompleteTextLength = UInt16(sTruncatedString.GetCodePointCount());
      UInt16 u16TruncationLength = UInt16(GetTruncationText().GetCodePointCount());

      if (u16CompleteTextLength > u16TruncationLength)
      {
         UInt16 u16TextLength = UInt16(u16CompleteTextLength - u16TruncationLength);

         if (u16HighlightStartIndex > u16TextLength)
         {
            u16HighlightCount = 0;
         }

         if (UInt32(u16HighlightStartIndex + u16HighlightCount) > u16TextLength)
         {
            if (u16TextLength > u16HighlightStartIndex)
            {
               u16HighlightCount = UInt16(u16TextLength - u16HighlightStartIndex);
            }
            else
            {
               u16HighlightCount = 0;
            }
         }
      }
      else
      {
         u16HighlightCount = 0;
      }
   }

   // split text into chunks based on the used highlight start position and character count
   UInt32 u32ByteCount = 0;
   UInt32 u32BufferSize = UInt32(sTruncatedString.GetCharCount() + 1);
   TChar* pBuffer = FEATSTD_NEW_ARRAY(TChar, u32BufferSize);

   if (pBuffer != 0)
   {
      if (m_pTextChunk1 != 0)
      {
         pBuffer[0] = TChar(0);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sTruncatedString);
         u32ByteCount = RichText::Utf8Encoding::u32Copy(pBuffer, u32BufferSize, sTruncatedString.GetCString(), u16HighlightStartIndex);
         SECURE_FEATSTD_STRING_ACCESS_END();
         m_pTextChunk1->vSetText(FeatStd::String(pBuffer));
      }

      if (m_pTextChunk2 != 0)
      {
         pBuffer[0] = TChar(0);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sTruncatedString);
         u32ByteCount += RichText::Utf8Encoding::u32Copy(pBuffer, u32BufferSize, &(sTruncatedString.GetCString()[u32ByteCount]), u16HighlightCount);
         SECURE_FEATSTD_STRING_ACCESS_END();
         m_pTextChunk2->vSetText(FeatStd::String(pBuffer));
      }

      if (m_pTextChunk3 != 0)
      {
         pBuffer[0] = TChar(0);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sTruncatedString);
         u32ByteCount += RichText::Utf8Encoding::u32Copy(pBuffer, u32BufferSize, &(sTruncatedString.GetCString()[u32ByteCount]));
         SECURE_FEATSTD_STRING_ACCESS_END();
         m_pTextChunk3->vSetText(FeatStd::String(pBuffer));
      }

      FEATSTD_DELETE_ARRAY(pBuffer);
   }

   // set paragraph size
   if (m_pDefaultParagraphChunk != 0)
   {
      TextAreaSizeType oTextAreaSize = GetTextAreaSize();
      m_pDefaultParagraphChunk->vSetSize(RichText::Size(Int32(oTextAreaSize.GetX()), Int32(oTextAreaSize.GetY())));
      if (true == GetMultiLineLayouting())
      {
         m_pDefaultParagraphChunk->vSetLineSpacingFactor(GetLinespacingFactor());
         m_pDefaultParagraphChunk->vSetTextWrapMode(GetTextWrapMode());
      }
   }

   if (!GetStyle().PointsToNull())
   {
      m_basePoint.SetX((m_layoutingRect.GetLeft() + m_layoutingRect.GetWidth()) * 0.5F);
      const Candera::TextRendering::Metrics& metrics = GetStyle()->GetMetrics();
      Float ascender = static_cast<Float>(metrics.ascender - 1);
      m_basePoint.SetY(m_layoutingRect.GetTop() + ascender);
   }
   if (GetNode() != NULL)
   {
      Candera::Float offset = m_basePoint.GetY() * GetNode()->GetScale().GetY();
      Candera::BaseLineLayouter::SetBaseLineOffset(*GetNode(), offset);
   }
   // perform layout
   Base::vLayoutRichText();
}


/****************************************************************************
*     Function    : vRenderRichText
*     Description : Will do the rendering of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after rendering.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void TextAreaWidget2D::vRenderRichText()
{
   // apply the scrolling offset based on horizontal alignment
   switch (GetHorizontalAlignment())
   {
      case RT_Leading:
         m_oRenderOffset.SetX(m_flScrollOffset * -1.0F);
         break;

      case RT_Trailing:
         m_oRenderOffset.SetX(m_flScrollOffset);
         break;

      default:
         m_oRenderOffset.SetX(0.0F);
   }

   // configure used colors
   if ((m_pNormalColorChunk1 != 0) && (m_pNormalColorChunk2 != 0) && (m_pHighlightColorChunk != 0))
   {
      if (IsEnabled())
      {
         if (GetTextWidgetMode() == Active)
         {
            m_pNormalColorChunk1->SetColor(GetActiveTextColor());
            m_pNormalColorChunk2->SetColor(GetActiveTextColor());
         }
         else
         {
            m_pNormalColorChunk1->SetColor(GetNormalTextColor());
            m_pNormalColorChunk2->SetColor(GetNormalTextColor());
         }

         m_pHighlightColorChunk->SetColor(GetHighlightTextColor());
      }
      else
      {
         m_pHighlightColorChunk->SetColor(GetDisabledTextColor());
         m_pNormalColorChunk1->SetColor(GetDisabledTextColor());
         m_pNormalColorChunk2->SetColor(GetDisabledTextColor());
      }
   }

   // perform rendering
   Base::vRenderRichText();
}


FeatStd::Int16 TextAreaWidget2D::GetEffectiveNumberOfLines() const
{
   return m_effectiveNumberOfLines;
}


Candera::Vector2 TextAreaWidget2D::GetBasePoint() const
{
   return m_basePoint;
}
