/* #***************************************************************************************
* FILE:          TextWidgetUtils.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextWidgetUtils is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "TextWidgetUtils.h"

#include <CanderaPlatform/Device/Common/Effects/TextBrush.h>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextWidgetUtils.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace text {


/*****************************************************************************/
class TextDynamicProperties
{
   public:
      static bool ClearTextBrushData(Candera::TextBrush& object)
      {
         return object.ClearValue(CdaDynamicPropertyInstance(InternalTextBrushData));
      }

      static bool SetTextBrushData(Candera::TextBrush& object, const TextBrushData::SharedPointer& value)
      {
         return object.SetValue(CdaDynamicPropertyInstance(InternalTextBrushData), value);
      }

      static TextBrushData::SharedPointer GetTextBrushData(const Candera::TextBrush& object)
      {
         return object.GetValue(CdaDynamicPropertyInstance(InternalTextBrushData));
      }

      static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* /*host*/)
      {
         return NULL;
      }

   private:
      static const TextBrushData::SharedPointer& DefaultTextBrushData()
      {
         FEATSTD_UNSYNCED_STATIC_OBJECT(TextBrushData::SharedPointer, _defaultValue);
         return _defaultValue;
      }

      CdaDynamicProperties(TextDynamicProperties, Candera::TextBrush);

      CdaDynamicProperty(InternalTextBrushData, TextBrushData::SharedPointer);
      CdaDynamicPropertyDefaultValue(DefaultTextBrushData());
      CdaDynamicPropertyEnd();

      CdaDynamicPropertiesEnd();
};


/*****************************************************************************/
TextBrushData::SharedPointer TextBrushData::Create()
{
   return SharedPointer(FEATSTD_NEW(TextBrushData));
}


/*****************************************************************************/
TextBrushData::SharedPointer TextBrushData::Create(const GlyphDataVector& data)
{
   return SharedPointer(FEATSTD_NEW(TextBrushData)(data));
}


/*****************************************************************************/
TextBrushData::TextBrushData() : _UpdateCount(0)
{
}


/*****************************************************************************/
TextBrushData::TextBrushData(const GlyphDataVector& data) : _GlyphDataVector(data), _UpdateCount(0)
{
}


/*****************************************************************************/
bool DynamicPropertiesAccessor::ClearTextBrushData(Candera::TextBrush& object)
{
   return TextDynamicProperties::ClearTextBrushData(object);
}


/*****************************************************************************/
bool DynamicPropertiesAccessor::SetTextBrushData(Candera::TextBrush& object, const TextBrushData::SharedPointer& value)
{
   return TextDynamicProperties::SetTextBrushData(object, value);
}


/*****************************************************************************/
TextBrushData::SharedPointer DynamicPropertiesAccessor::GetTextBrushData(const Candera::TextBrush& object)
{
   return TextDynamicProperties::GetTextBrushData(object);
}


/*****************************************************************************/
TextBrushData::SharedPointer DynamicPropertiesAccessor::GetOrCreateTextBrushData(Candera::TextBrush& object)
{
   TextBrushData::SharedPointer value(TextDynamicProperties::GetTextBrushData(object));
   if (value.PointsToNull())
   {
      value = TextBrushData::Create();
      TextDynamicProperties::SetTextBrushData(object, value);
   }
   return value;
}


/*****************************************************************************/
size_t TextUtils::GetByteCount(const Candera::TChar* text)
{
   if (text == NULL)
   {
      return 0;
   }

   unsigned char u8StartByte = static_cast<unsigned char>(text[0]);

   if (u8StartByte == 0)
   {
      return 0;
   }
   if (u8StartByte < 0xC0)
   {
      // 1 byte standard
      return 1;
   }

   //todo: this condition needs to be checked
   if (((u8StartByte & 0xC0) == 0x80) || ((u8StartByte >= 0xC0 || u8StartByte < 0x80) == false))
   {
      // Oops: not a starting byte */
      HMI_APP_ASSERT_ALWAYS();
      return 0;
   }
   if (u8StartByte < 0xE0)
   {
      // 2 byte MultiByteChar
      return 2;
   }
   if (u8StartByte < 0xF0)
   {
      // 3 byte MultiByteChar
      return 3;
   }
   if (u8StartByte < 0xF8)
   {
      // 4 byte MultiByteChar
      HMI_APP_ASSERT(u8StartByte < 0xF5); // max unicode: (http://en.wikipedia.org/wiki/UTF-8) In November 2003,UTF-8 was restricted by RFC 3629 to end at U+10FFFF
      return 4;                           // So for a complete check 2. byte would be necessary. if leading byte == 0xF4 the 1. continuation byte must be < 0x90
   }
   HMI_APP_ASSERT_ALWAYS();
   return 0;
}


/*****************************************************************************/
void TextUtils::PrintDebug(const TextBrushData::GlyphDataVector& glyphDataVector, const Candera::TChar* text)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "TextUtils::PrintDebug() glyphCount=%u, text='%s'", glyphDataVector.Size(), text != NULL ? text : "null"));

   char buffer[100];
   for (TextBrushData::GlyphDataVector::ConstIterator it = glyphDataVector.ConstBegin(); it != glyphDataVector.ConstEnd(); ++it)
   {
      const GlyphData& g(*it);
      const char* t = (text != NULL) ? text + g.m_characterPosition : NULL;
      SNPRINTF(buffer, sizeof(buffer), "'%c' [%2d,%u][%3d,%3d,%2d,%2d][%3d,%3d,%3d,%3d]",
               t != NULL ? t[0] : ' ',
               g.m_characterPosition,
               static_cast<unsigned int>(GetByteCount(t)),
               g.m_position.x,
               g.m_position.y,
               g.m_size.width,
               g.m_size.height,
               g.m_left,
               g.m_top,
               g.m_right,
               g.m_bottom);

      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "   %s", buffer));
   }
}


/*****************************************************************************/
template <> bool GlyphDataIteratorFactory::IsGlyphHidden<hmibase::widget::text::GlyphData>(const hmibase::widget::text::GlyphData& g)
{
   return g.m_hidden;
}


}
}


}

namespace Candera {
namespace MetaInfo {

/*****************************************************************************/
template<> struct DataType<hmibase::widget::text::TextBrushData::SharedPointer>
{
   static inline const Char* GetEditor()
   {
      return "";
   }

   static inline bool ConvertToString(hmibase::widget::text::TextBrushData::SharedPointer, Char*, UInt)
   {
      return true;
   }

   static inline bool ConvertFromString(hmibase::widget::text::TextBrushData::SharedPointer, const Char*)
   {
      return true;
   }
};


}
}
