/* #**************************************************************************************
* FILE:          TextWidget2DV2Helper.h
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:	 TextWidget2DV2Helper is part of HMI-Base Widget Library
* COPYRIGHT:     (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*
 *  The class TextWidget2DV2PreprocessingContext is only used to convert a FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>
 * into a Candera::TextRendering::PreprocessingContext::Iterator. Due to access restrictions of serveral classes this is the only possible way.
 * TextWidget2DV2PreprocessingContext::CreateIterator(glyphData) will provide a iterator that can be used as a preprocessed text for a text brush.
 */
class TextWidget2DV2PreprocessingContext : public Candera::TextRendering::PreprocessingContext
{
   public:
      static Candera::TextRendering::PreprocessingContext::Iterator CreateIterator(FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& glyphData)
      {
         return TextWidget2DV2PreprocessingContext(glyphData).GetIterator();
      }

   protected:
      virtual void Measure(Candera::TextRendering::PixelPosition x, Candera::TextRendering::PixelPosition y, const Candera::TextRendering::GlyphBitmap& glyph) override
      {
         FEATSTD_UNUSED(x);
         FEATSTD_UNUSED(y);
         FEATSTD_UNUSED(glyph);
      }

      virtual ReferenceIterator* GetReferenceIterator() const
      {
         return Candera::TextRendering::Internal::PreprocessingContextVectorIterator<FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>, Candera::TextRendering::PreprocessingContext::ReferenceIterator>
                (m_glyphData.ConstBegin(), m_glyphData.ConstEnd()).Clone();
      }

   private:
      FEATSTD_MAKE_CLASS_STATIC(TextWidget2DV2PreprocessingContext);
      FEATSTD_MAKE_CLASS_UNCOPYABLE(TextWidget2DV2PreprocessingContext);

      TextWidget2DV2PreprocessingContext(FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& glyphData) :
         m_glyphData(glyphData)
      {
      }

      FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& m_glyphData;
};


class TextRenderContextStrategy : public Candera::TextRendering::TextRenderContext, public Candera::TextRendering::TextLayoutStrategy
{
   public:
};


class PreprocessedTextLine
{
   public:
      //    FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData> m_glyphs;
      Candera::Vector2 m_size;
};


class PreprocessedText
{
   public:
      FeatStd::Internal::Vector<PreprocessedTextLine> m_lines;
      Candera::Vector2 m_size;
};


class PreprocessingTextRenderContext : public TextRenderContextStrategy
{
   public:

      PreprocessingTextRenderContext(const Candera::TextRendering::SharedStyle::SharedPointer& /*style*//*, FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& textGlyphData*/)
      {
      }

      FEATSTD_MAKE_CLASS_STATIC(PreprocessingTextRenderContext);
      FEATSTD_MAKE_CLASS_UNCOPYABLE(PreprocessingTextRenderContext);
      Candera::TextRendering::LayoutingOptions m_layoutingOptions;
      Candera::TextRendering::ShapingOptions m_shapingOptions;
      Candera::TextRendering::PixelPosition m_ascender;
      Candera::TextRendering::PixelPosition m_descender;
};


/*
 * The TextWidgetTextRenderContext combines the TextRenderContext and the TextLayoutStrategy interfaces. With access to both interfaces and the provided information
 * this class is able to measure the preferred size and the number of lines, truncate and preprocess the text in one linear text processing iteration.
 */
class TextWidgetTextRenderContext : public TextRenderContextStrategy
{
   public:
      enum Truncation
      {
         None,
         Hard,
         Soft,
         Text
      };

      static void SetPreprocessedText(FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& glyphs, Candera::TextRendering::LayoutingOptions& layoutingOptions, Candera::TextNode2D* textNode, Candera::TextBrush* brush)
      {
         const Candera::TextRendering::PreprocessingContext::Iterator it = TextWidget2DV2PreprocessingContext::CreateIterator(glyphs);
         if (0 != textNode)
         {
            HelperTextNode2DLayouter::SetPreprocessedText(*textNode, it, layoutingOptions);
         }
         else
         {
            if (0 != brush)
            {
               brush->PreprocessedText().Set(it);
               brush->CacheArea().Set(Candera::Vector2(layoutingOptions.GetSize().GetWidth(), layoutingOptions.GetSize().GetHeight()));
               brush->Update();
            }
         }
      }

      /*
       * The following parameters have to be provided:
       * + the truncation method,
       * + the brush (as owner of the used style and reference TextRenderContext),
       * + the container in which the preprocessed glyph data has to be stored,
       * + the preprocessed glyph data of the truncation text and
       * + the width of the truncation text
       *
       * Limitations:
       * + the truncation text is currently added to the right side of the last line (independent to the current culture direction).
       * + the instance should only use it once it may not be in a correct state if used twice.
       * + only single line truncation text is allowed (no fail safe checks).
       * + not yet tested with arabic.
       *
       * NOTE: the performance of preprocesing for scrolling can be improved by making an active object aproach out of the preprocessing.
       * as soon as the window boundaries have been reached after a line break the preprocessing can be stopped. the first window can be projected and rendered
       * as part of the update iteration the preprocessing can be continued for some lines and stopped again after those lines. this has to be done for each update
       * call until the complete text is preprocessed. this allows to immediatly show the long text and distribute the cpu load among selveral frames.
       */
      TextWidgetTextRenderContext(
         Truncation truncation,
         const Candera::TextRendering::SharedStyle::SharedPointer& style,
         const Candera::TextRendering::TextRenderContext* reference,
         FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& textGlyphData,
         FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>* truncationTextGlyphData,
         FeatStd::UInt16 truncationTextWidth,
         Candera::enTruncationDirection truncationDirection,
         Candera::Vector2 additionalSize,
         Candera::ScrollBehavior scrollBehavior = Candera::Off,
         FeatStd::UInt32 maxNnumberOfLines = 0,
         bool multiLineLayouting = false) :
         m_shapingOptions(style),
         m_ascender((style.PointsToNull()) ? 0 : style->GetMetrics().ascender),
         m_descender((style.PointsToNull()) ? 0 : style->GetMetrics().descender),
         m_textRectangle(Candera::TextRendering::TextRect::GetMin()),
         m_clipRectangle(Candera::TextRendering::TextRect::GetMax()),
         m_truncation(truncation),
         m_numberOfLines(0),
         m_pendingLine(0),
         m_currentLine(0),
         m_offsetX(0),
         m_offsetY(0),
         m_first(true),
         m_firstInLine(true),
         m_previousX(0),
         m_previousY(0),
         m_reference(reference),
         m_textGlyphData(textGlyphData),
         m_truncationTextGlyphData(truncationTextGlyphData),
         m_truncationTextWidth(truncationTextWidth),
         m_maxNumberOfLines(maxNnumberOfLines),
         m_truncationDirection(truncationDirection),
         m_scrollBehaviour(scrollBehavior),
         m_multiLineLayouting(multiLineLayouting),
         m_bIsSkippedBlit(false),
         m_additionalSize(additionalSize),
         m_charCountEachLineHelper(),
         m_charCount(0),
         m_bIsCurrentLineUpdated(true),
         m_rectTopEachLine(),
         m_bIsTextTruncated(false),
         m_bIsTruncationTextReqForLeft(false),
         m_bisFullTextEnglish(true),
         m_bIsMoreTextThanNumberOfLines(false),
         m_bCultureDependentAlignment(true),
         m_bInitialGlyphProcessing(true)

      {
         m_layoutingOptions.SetLayoutStrategy(this);
         m_bIsBlitCalled = false;
         m_bIsTruncationRequired = false;
      }

      ~TextWidgetTextRenderContext()
      {
         m_bIsBlitCalled = false;
         m_bIsSkippedBlit = false;
      }
      /*
       * Returns the number of lines (only valid after the text has been processed).
       */
      FeatStd::UInt32 GetNumberOfLines() const
      {
         return m_numberOfLines;
      }

      /*
      * Returns the measured text rectangle (only valid after the text has been processed).
      */
      const Candera::TextRendering::TextRect& GetTextRectangle() const
      {
         return m_textRectangle.IsNegative() ? Candera::TextRendering::TextRect::GetDefault() : m_textRectangle;
      }

      /*
      * Returns the layout options (has to be configured before text processing).
      */
      Candera::TextRendering::LayoutingOptions& LayoutingOptions()
      {
         return m_layoutingOptions;
      }

      /*
      * Returns the shaping options (has to be configured before text processing).
      */
      Candera::TextRendering::ShapingOptions& ShapingOptions()
      {
         return m_shapingOptions;
      }
      /*
      * Clears the Pending glyph data , used when calling Render again with different layouting options
      */
      void ClearPendingGlyphData()
      {
         m_pendingGlyphData.Clear();
      }
      /*
      * If any glyph lies outside the clipping rect , SkippedBlit() is called with glyph info,m_bIsSkippedBlit tracks if SkippedBlit() is called
      */
      bool IsBlitSkipped()
      {
         return m_bIsSkippedBlit;
      }

      void SetCultureDependentAlginment(bool cultureDependentAlignment)
      {
         m_bCultureDependentAlignment = cultureDependentAlignment;
      }

      void SetInitialGlyphProcessing(bool initialGlyphProcessing)
      {
         m_bInitialGlyphProcessing = initialGlyphProcessing;
      }

      void SetCurrentLine(Candera::TextRendering::PixelPosition currentLine)
      {
         m_currentLine = currentLine;
      }
      /*
      * Returns the characters in each line(only valid after the text has been processed).
      * Characters are inserted in new line when onlinebreak is called
      */
      std::vector<Candera::UInt32>& GetCharCountInEachLineHelper()
      {
         return m_charCountEachLineHelper;
      }
      /*
      * Returns the Top coordinate/position of text. This will be used in editfield to set the cursor position.
      */
      std::vector<Candera::Int32>& GetRectTopEachLine()
      {
         return m_rectTopEachLine;
      }

      /*
      * Fits Into given rectangle
      */
      bool FitsIntoRectangle()
      {
         if (m_pendingGlyphData.Size() > 0)
         {
            if (((false == m_multiLineLayouting) && (m_currentLine <= m_layoutingOptions.GetSize().GetWidth()) && (GetTextRectangle().GetWidth() <= m_layoutingOptions.GetSize().GetWidth()))
                  || ((true == m_multiLineLayouting) && (GetTextRectangle().GetHeight() <= m_layoutingOptions.GetSize().GetHeight()) && m_bIsMoreTextThanNumberOfLines == false))
            {
               return true;
            }
         }
         else if (((false == m_multiLineLayouting) && (GetTextRectangle().GetWidth() <= m_layoutingOptions.GetSize().GetWidth()))
                  || ((true == m_multiLineLayouting) && ((GetTextRectangle().GetHeight() <= m_layoutingOptions.GetSize().GetHeight()) || (m_layoutingOptions.GetSize().GetHeight() < 0)) && (m_bIsMoreTextThanNumberOfLines == false)))
         {
            return true;
         }
         return false;
      }

      /*
      * Finish the text processing (has to be called after text processing with the TextRenderer is finished).
      * Only required for truncation with Text. Pending glyph data or the trunction text glyph data will be appended.
      */
      void Finish(bool processForArabic = false)
      {
         if (m_charCount > 0)
         {
            m_charCountEachLineHelper.push_back(m_charCount);
            m_charCount = 0;
         }
         m_rectTopEachLine.push_back(static_cast<Candera::Int32>(m_textRectangle.GetBottom()));
         if (m_truncation == Text)
         {
            if ((m_pendingGlyphData.Size() > 0) && (false == m_multiLineLayouting))
            {
               if (m_currentLine <= m_layoutingOptions.GetSize().GetWidth())
               {
                  if (false == Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().PointsToNull())
                  {
                     if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection() == Candera::Globalization::RightToLeft && m_bIsSkippedBlit == true)
                     {
                        if (processForArabic)
                        {
                           AddPendingGlyphData();
                           ProcessGlyphForArabic();
                        }
                        else
                        {
                           AddTruncationTextInLast();
                        }
                        m_bIsSkippedBlit = false;
                     }
                     else
                     {
                        //AddPendingGlyphData() ::the pending glyphs are all visible and no truncation text has to be shown
                        // AddTruncationTextInLast() :: Truncation if glyphs addition exactly matched the maximum size width and hence m_bIsSkippedBlit will be true as there is more text which is outside the clipRect()
                        (false == m_bIsSkippedBlit) ? AddPendingGlyphData() : AddTruncationTextInLast(), m_bIsSkippedBlit = false;
                     }
                  }
                  if ((Candera::Left == m_truncationDirection) && (0 != m_truncationTextGlyphData) && (m_textGlyphData.Size() > 0) && true == m_bIsTruncationTextReqForLeft)
                  {
                     //========For Left Truncation .. Need to revisit this case/check
                     for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
                     {
                        m_textGlyphData.Insert(0, (*m_truncationTextGlyphData)[i]);
                     }
                     m_bIsTextTruncated = true;
                  }
               }
               else
               {
                  // the pending glyphs not complete visible, the truncation glyphs have to be added
                  if (0 != m_truncationTextGlyphData)
                  {
                     if (m_textGlyphData.Size() > 0)
                     {
                        if (Candera::Right == m_truncationDirection)
                        {
                           AddTruncationTextInLast();
                        }
                        else if (true == m_bIsTruncationTextReqForLeft)
                        {
                           //======For Left Truncation ... insert in the beginning
                           for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
                           {
                              m_textGlyphData.Insert(0, (*m_truncationTextGlyphData)[i]);
                           }
                           m_bIsTextTruncated = true;
                        }
                     }
                     else
                     {
                        // only the truncation text is visible
                        for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
                        {
                           m_textGlyphData.Add((*m_truncationTextGlyphData)[i]);
                        }
                        m_bIsTextTruncated = true;
                     }
                  }
               }
            }
            else if ((Candera::Left == m_truncationDirection) && (0 != m_truncationTextGlyphData) && (m_textGlyphData.Size() > 0) && (true == m_bIsTruncationTextReqForLeft))
            {
               //========For Left Truncation .. Need to revisit this case/check
               for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
               {
                  m_textGlyphData.Insert(0, (*m_truncationTextGlyphData)[i]);
               }
               m_bIsTextTruncated = true;
            }
            else if ((true == m_bIsTruncationRequired) && (false == m_multiLineLayouting))
            {
               if (0 != m_truncationTextGlyphData)
               {
                  if (m_textGlyphData.Size() > 0)
                  {
                     if (Candera::Right == m_truncationDirection)
                     {
                        AddTruncationTextInLast();
                     }
                     else if (true == m_bIsTruncationTextReqForLeft)
                     {
                        //======For Left Truncation ... insert in the beginning
                        for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
                        {
                           m_textGlyphData.Insert(0, (*m_truncationTextGlyphData)[i]);
                        }
                        m_bIsTextTruncated = true;
                     }
                  }
               }
            }
         }
      }

      void ProcessGlyphForArabic()
      {
         // The characters are read from last to first (First means the character which has been rendered on Right end as Arabic is RTL)
         // So the m_textGlyphData's last char is first char (rendered at right end) and hence we have to insert the elipses in left end or first glyph in m_textGlyphData
         if (m_textGlyphData.Size() > 0)
         {
            FeatStd::UInt16 truncTextWidth = m_truncationTextWidth;
            if (m_textGlyphData[0].m_position.x < 0)
            {
               truncTextWidth = static_cast<FeatStd::UInt16>(truncTextWidth + (m_textGlyphData[0].m_size.width - m_textGlyphData[0].m_position.x));            // Taking the width which lies between -x to +x in x axis as the char will be intersecting with clipRect as width for char may be eg -2 to +7
            }
            FeatStd::UInt16 widthInGlypData = 0;
            size_t count = 0;
            for (size_t i = 0; i < m_textGlyphData.Size(); ++i)
            {
               widthInGlypData = static_cast<FeatStd::UInt16>(widthInGlypData + m_textGlyphData[i].m_size.width);            // check the width taken by characters which can be removed to fit elipses (...)
               if (widthInGlypData >= truncTextWidth)
               {
                  count = ++i;                                             // These many characters can be removed from glyph data to fit in the elipses
                  break;
               }
            }
            for (size_t i = 0; i < count; ++i)
            {
               m_textGlyphData.Remove(0);                                    // These many characters can be removed from glyph data to fit in the elipses
               m_bIsTextTruncated = true;                                    // characters removed to add the elipses means there is truncation
            }
            //Insert the elipes in start of string
            for (size_t i = 0; i < m_truncationTextGlyphData->Size(); ++i)
            {
               TextWidget2DV2::MinimalGlyphData glyphData = (*m_truncationTextGlyphData)[i];
               glyphData.m_position.x = static_cast<Candera::TextRendering::PixelPosition>(glyphData.m_position.x + ((m_textGlyphData[i].m_position.x - 0) - m_truncationTextWidth));        // elipses to be just next to text and not at the boundary of left most in x axis
               glyphData.m_right = static_cast<Candera::TextRendering::PixelPosition>(glyphData.m_right + ((m_textGlyphData[i].m_position.x - 0) - m_truncationTextWidth));
               m_textGlyphData.Insert(0, glyphData);
            }
         }
      }

      void ProcessGlyphForLeft()
      {
         if (m_textGlyphData.Size() > 0 && ((m_textGlyphData[m_textGlyphData.Size() - 1].m_right) > Candera::TextRendering::PixelSize(m_layoutingOptions.GetSize().GetWidth())))
         {
            FeatStd::SizeType noOfGlyphs = m_textGlyphData.Size();
            FeatStd::UInt16 truncationLength = (Text == m_truncation) ? m_truncationTextWidth : 0;
            FeatStd::SizeType sizeTruncationData = m_truncationTextGlyphData->Size();
            FeatStd::UInt16 truncationRight = 0;
            if (sizeTruncationData  > 0 && truncationLength > 0)
            {
               truncationRight = (*m_truncationTextGlyphData)[sizeTruncationData - 1].m_right;		//This is calculated to reduce the space between end of truncation "..." and text
            }
            FeatStd::Int16 positionFromRight = static_cast<FeatStd::Int16>(m_textGlyphData[m_textGlyphData.Size() - 1].m_right - m_layoutingOptions.GetSize().GetWidth());
            FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>::Iterator it = m_textGlyphData.End();		// Check the m_textGlyphData from right for how many glyphs are possible in the mximum size
            while (it != m_textGlyphData.Begin())
            {
               --it;
               if ((*it).m_position.x <= (positionFromRight + truncationLength))
               {
                  // To be used for HARD truncation
                  FeatStd::Int16 diffInPosition = static_cast<Candera::TextRendering::PixelPosition>((positionFromRight + truncationLength) - (*it).m_position.x);
                  if (diffInPosition && (Hard == m_truncation))
                  {
                     noOfGlyphs--;				// include last glyph also in case of bounding is intersecting for Hard trunc
                  }
                  break;
               }
               noOfGlyphs--;
            }
            if (noOfGlyphs < m_textGlyphData.Size() && truncationRight)
            {
               truncationRight = static_cast<Candera::TextRendering::PixelPosition>(m_textGlyphData[noOfGlyphs].m_position.x - positionFromRight - truncationRight);	//This is calculated to reduce the space between end of truncation "..." and text
            }
            FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData> tempData;
            for (FeatStd::SizeType Dataindex = noOfGlyphs, index = 0; Dataindex < m_textGlyphData.Size(); Dataindex++, index++)		// Add the glyphs within the range of positionFromRight
            {
               tempData.Add(m_textGlyphData[Dataindex]);
               tempData[index].m_position.x = static_cast<Candera::TextRendering::PixelPosition>((m_textGlyphData[Dataindex].m_position.x - (positionFromRight) - truncationRight));			// Each position and Right to be in the positionFromRight
               tempData[index].m_right = static_cast<Candera::TextRendering::PixelPosition>((m_textGlyphData[Dataindex].m_right - (positionFromRight) - truncationRight));
            }
            m_textGlyphData.Clear();
            m_textGlyphData = tempData;
            m_bIsTruncationTextReqForLeft = true;
         }
      }
      bool& IsTextTruncated()
      {
         return m_bIsTextTruncated;
      }
      /*
      * m_bisFullTextEnglish tracks the glyphs direction so if no glyph is having RTL direction means the current text is in Latin
      */
      bool IsTextEnglish()
      {
         return m_bisFullTextEnglish;
      }
   protected:
      virtual Candera::TextRendering::TextLayoutStrategy::Action OnInit(Pass pass)
      {
         FEATSTD_UNUSED(pass);
         UpdateClipRect();
         return Continue;
      }

      /*
      * adds the pending glyph data to the container (only for truncation mode Text)
      */
      void AddPendingGlyphData()
      {
         // the glyphs that are stored in a buffer (because they fit into the box if no truncation text is shown)
         // have to be added to the preprocessed glyphs
         for (FeatStd::SizeType i = 0; i < m_pendingGlyphData.Size(); ++i)
         {
            m_textGlyphData.Add(m_pendingGlyphData[i]);
         }
         m_pendingGlyphData.Clear();
      }

      /*
      * Will be called whenever a line break occurs. It counts the number of lines and checks if the text processing has to be stopped due to exceeding the allowed area by returning Break.
      */
      virtual Candera::TextRendering::TextLayoutStrategy::Action OnLineBreak(const FeatStd::Char* textCursor, Candera::TextRendering::TextLayoutStrategy::LineBreakType type)
      {
         FEATSTD_UNUSED(type);
         FEATSTD_UNUSED(textCursor);
         // increment the number of line counter. after processing the text this number can be used for further actions that depend on the number of lines
         // (like for example a different base line for base line layout)

         Candera::String remText(textCursor);			//This is to understand if there is more text which can not fit in m_maxNumberOfLines

         if (Candera::TextRendering::TextLayoutStrategy::ForcedBreak == type)
         {
            m_bIsMoreTextThanNumberOfLines = true;
         }

         if (!m_bIsBlitCalled)
         {
            return Continue;
         }
         m_charCountEachLineHelper.push_back(m_charCount);
         m_bIsCurrentLineUpdated = true;
         //m_rectTopEachLine.push_back(m_textRectangle.GetTop());
         m_charCount = 0;
         //The m_numberOfLines is increased to check if this is the last line , earlier one more line was rendered.
         ++m_numberOfLines;
         if ((0 != m_maxNumberOfLines) && m_multiLineLayouting && (m_numberOfLines >= m_maxNumberOfLines) && (m_layoutingOptions.GetSize().GetHeight() < 0))
         {
            if ((m_truncation == Text) && (Candera::Right == m_truncationDirection))
               /*&& (m_textRectangle.GetBottom() + m_ascender <= m_layoutingOptions.GetSize().GetHeight()) )*/
            {
               if (!remText.IsEmpty())
               {
                  m_bIsMoreTextThanNumberOfLines = true;
                  // Adding truncation text width to textrect width to render the trunction text properly
                  m_textRectangle.SetRight(static_cast<Candera::TextRendering::PixelPosition>(m_textRectangle.GetRight() + m_truncationTextWidth));
                  AddTruncationTextInLast();
               }
               else if (m_pendingGlyphData.Size() > 0)
               {
                  AddPendingGlyphData();
               }
            }
            if ((m_truncation == None) && (m_numberOfLines >= m_maxNumberOfLines) && !remText.IsEmpty())
            {
               m_bIsMoreTextThanNumberOfLines = true;
            }
            return TextLayoutStrategy::Break;
         }
         /*++m_numberOfLines;*/			//This condition has been moved up to restrict rendering to m_maxNumberOfLines
         // reset the PixelPosition type overflow detection on the x axis
         m_offsetX = 0;
         m_firstInLine = true;
         if ((m_truncation == None)
               || ((m_layoutingOptions.GetSize().GetHeight() < 0) && (true == m_multiLineLayouting))
               || (((m_truncation == Hard) || (m_truncation == Text)) && (m_textRectangle.GetBottom() + m_ascender <= m_layoutingOptions.GetSize().GetHeight()) && (m_numberOfLines < m_maxNumberOfLines))
               || ((m_truncation == Soft) && (m_textRectangle.GetBottom() < m_layoutingOptions.GetSize().GetHeight()) && (m_numberOfLines < m_maxNumberOfLines)))
         {
            if (m_truncation == Text)
            {
               AddPendingGlyphData();
            }
            if ((m_truncation == None) && (m_numberOfLines >= m_maxNumberOfLines) && !remText.IsEmpty())
            {
               m_bIsMoreTextThanNumberOfLines = true;
            }
            // reset the position for the soft truncation method
            m_currentLine = 0;
            return Continue;
         }
         //If GetHeight() is greater than 0 and truncation is enabled we need to add truncation text
         if ((m_truncation == Text) && (Candera::Right == m_truncationDirection) && (true == m_multiLineLayouting)
               /*&& (m_textRectangle.GetBottom() + m_ascender >= m_layoutingOptions.GetSize().GetHeight())*/)
         {
            if (!remText.IsEmpty())
            {
               m_bIsMoreTextThanNumberOfLines = true;
               // Adding truncation text width to textrect width to render the trunction text properly
               m_textRectangle.SetRight(static_cast<Candera::TextRendering::PixelPosition>(m_textRectangle.GetRight() + m_truncationTextWidth));
               AddTruncationTextInLast();
            }
            else if (m_pendingGlyphData.Size() > 0)
            {
               AddPendingGlyphData();
            }
         }
         // the remaining lines are truncated. therefore, we can stop the text processing.
         return TextLayoutStrategy::Break;
      }

      /*
      * Will be called for each glyph. It will skip glyphs that are outside the allowed area by returning Skip.
      */
      virtual Candera::TextRendering::TextLayoutStrategy::Action OnGlyphProcessing(Candera::TextRendering::TextPosition textPosition)
      {
         FEATSTD_UNUSED(textPosition);
         // if ((m_truncation == None) || (m_currentLine < m_layoutingOptions.GetSize().GetHeight()))
         if ((m_truncation == None) || (m_layoutingOptions.GetSize().GetWidth() < 0)
               || (m_currentLine <= m_layoutingOptions.GetSize().GetWidth() || (m_bCultureDependentAlignment == false && true == m_bInitialGlyphProcessing)))
         {
            return Continue;
         }
         if ((Candera::Off != m_scrollBehaviour) || (Candera::Left == m_truncationDirection))	//This is to calculate full text to glyphs
         {
            return Continue;
         }
         return TextLayoutStrategy::Skip;
      };

      /*
      * Updates the clipping rectangle.
      * The TextEngine blit-calls on a glyph are based on the clipping rectangle.
      * Layout options (e.g. word wrap) are connected to 'layouting options - size';
      * blitting is bound to clipping rectangle. When the clip rect is not set and therefore
      * GetClipRect() is always 'max', then glyphs are blitted outside of the maximum boundaries.
      */
      virtual void UpdateClipRect()
      {
         m_clipRectangle = Candera::TextRendering::TextRect::GetMax();
         if (m_scrollBehaviour == Candera::Off && ((None != m_truncation)))
         {
            if (m_layoutingOptions.GetSize().GetWidth() > 0 && (Candera::Left != m_truncationDirection))
            {
               //if(Candera::Left != m_truncationDirection)
               m_clipRectangle.SetLeft(0);
               m_clipRectangle.SetWidth(m_layoutingOptions.GetSize().GetWidth());
            }
            if (m_layoutingOptions.GetSize().GetHeight() > 0)
            {
               m_clipRectangle.SetTop(0);
               m_clipRectangle.SetHeight(m_layoutingOptions.GetSize().GetHeight());
            }
         }
      }

      /*
      * Interface has to be implemented. No limitation required for the implementation of the features.
      */
      virtual const Candera::TextRendering::TextRect& GetClipRect() const override
      {
         return m_clipRectangle;
      }

      /*
      * will be called for each glyph that will not be rendered. We are using in case of arabic language (RTL direction)
      * As in arabic text is read from -x coordinates to 0 so anything outside clipRect will be skipped.
      * If there is any skipped blit , then there is more text than the maximum size
      */
      virtual void SkippedBlit(FeatStd::Int16 x, FeatStd::Int16 y, const Candera::TextRendering::GlyphBitmap& glyph)
      {
         FEATSTD_UNUSED(x);
         FEATSTD_UNUSED(y);
         FEATSTD_UNUSED(glyph);
         m_bIsSkippedBlit = true;
         if (Candera::TextRendering::GlyphBitmap::RightToLeft == glyph.direction)
         {
            m_bisFullTextEnglish = false;
         }
      }

      /*
      * will be called for each glyph that has to be rendered. It measures the preferred area and stores the preprocessed glyph data
      * (if already possible; due to truncation mode Text the glyphs at the line end have to be handled differently).
      */
      virtual void Blit(FeatStd::Int16 x, FeatStd::Int16 y, const Candera::TextRendering::GlyphBitmap& glyph) override
      {
         // measure the bounding box and base line of the glyph
         // NOTE: incomming x = cursorX + glyph.left and y = cursorY - glyph.top
         //In case of alignment other than Vtop/Hleft and Height is given , OnlineBreak is called first
         // This var is used for restricting OnlineBreak to increase m_numberOfLines.
         m_bIsBlitCalled = true;
         m_charCount++;
         Candera::TextRendering::PixelPosition left = static_cast<Candera::TextRendering::PixelPosition>(x - glyph.left);
         Candera::TextRendering::PixelPosition right = static_cast<Candera::TextRendering::PixelPosition>(left + glyph.xadvance);
         Candera::TextRendering::PixelPosition glyphBaseLine = static_cast<Candera::TextRendering::PixelPosition>(y + glyph.top);
         Candera::TextRendering::PixelPosition top = static_cast<Candera::TextRendering::PixelPosition>((glyphBaseLine - m_ascender) + 1);
         Candera::TextRendering::PixelPosition bottom = static_cast<Candera::TextRendering::PixelPosition>(glyphBaseLine - m_descender);

         if (true == m_bIsCurrentLineUpdated)
         {
            m_rectTopEachLine.push_back(static_cast<Candera::Int32>(top));
            m_bIsCurrentLineUpdated = false;
         }
         // prepare the preprocessed glyph data
         TextWidget2DV2::MinimalGlyphData storedGlyphData;
         storedGlyphData.m_fontIdentifier = glyph.fontIdentifier;
         storedGlyphData.m_glyphIndex = glyph.glyphIndex;
         storedGlyphData.m_characterPosition = glyph.characterPosition;
         storedGlyphData.m_position.x = static_cast<Candera::TextRendering::PixelPosition>(x + Candera::TextRendering::PixelPosition(m_additionalSize.GetX()));
         storedGlyphData.m_position.y = static_cast<Candera::TextRendering::PixelPosition>(y + Candera::TextRendering::PixelPosition(m_additionalSize.GetY()));
         // beside the above minimal required glyph data we also need the base line and the top for finalizing the truncation in the truncation mode Text.
         storedGlyphData.m_glyphBaseLine = glyphBaseLine;
         storedGlyphData.m_glyphTop = glyph.top;
         // we perform a PixelPosition type overflow detection to enable text scrolling of long texts
         // (2^31 pixel for the x and y axis; at an extrem value of 480 pixel line height this allows to handle 4473924 text lines)
         if ((!m_firstInLine) && (m_previousX > left))
         {
            // for the first glyph blit call in a line m_previousX is not yet initialized
            // we have detected an overflow of the PixelPosition data type on the x axis
            // therefore we increment the offset
            ++m_offsetX;
         }
         if ((!m_first) && (m_previousY > glyphBaseLine))
         {
            // for the first glyph blit call m_previousY is not yet initialized
            // we have detected an overflow of the PixelPosition data type on the y axis
            // therefore we increment the offset
            ++m_offsetY;
         }
         // update the PixelPosition type overflow detection
         m_firstInLine = false;
         m_first = false;
         m_previousX = left;
         m_previousY = glyphBaseLine;
         // for the scrolling feature we need also the size of the glyph
         storedGlyphData.m_size.width = glyph.width;
         storedGlyphData.m_size.height = glyph.height;
         // we need the right and bottom value to provide scrolling limit values (begin of next page and end of line)
         storedGlyphData.m_right = right;
         storedGlyphData.m_bottom = bottom;
         // we store the current overflow correction values to enable the correct position handling for scrolling
         storedGlyphData.m_offsetX = m_offsetX;
         storedGlyphData.m_offsetY = m_offsetY;
         if (Candera::TextRendering::GlyphBitmap::RightToLeft == glyph.direction)
         {
            m_bisFullTextEnglish = false;
         }
         // test for the truncation settings and add the glyph to the preprocessed glyphs if not truncated
         if (Candera::Off == m_scrollBehaviour)
         {
            if ((m_truncation == None)
                  ||	(m_layoutingOptions.GetSize().GetWidth() < 0)
                  || ((m_truncation == Hard) && (m_currentLine <= m_layoutingOptions.GetSize().GetWidth()))
                  || ((m_truncation == Soft) && (right <= m_layoutingOptions.GetSize().GetWidth()))
                  || ((m_truncation == Text) && (right + m_truncationTextWidth <= m_layoutingOptions.GetSize().GetWidth()))
                  || (Candera::Left == m_truncationDirection))
            {
               // for text truncation the limit has also to consider the fact that the truncation text has to fit into the box.
               static_cast<void>(m_textGlyphData.Add(storedGlyphData));
               m_pendingLine = right;
            }
            else
            {
               // if truncated and the truncation mode is text then we have to store the glph for later processing
               if ((m_truncation == Text))
               {
                  if (right <= m_layoutingOptions.GetSize().GetWidth())
                  {
                     m_bIsTruncationRequired = false;
                     // in case of a line break the glyph has to be rendered because it fits into the box (then the truncation text is in a line after this one).
                     static_cast<void>(m_pendingGlyphData.Add(storedGlyphData));
                  }
                  else
                  {
                     m_bIsTruncationRequired = true;					//This will be used if the this character cannot fit in the given width but there is space for truncation text
                  }
               }
            }
         }
         else
         {
            //This needs to be revisited... done in case of scrolling ,add all text to m_textGlyphData
            static_cast<void>(m_textGlyphData.Add(storedGlyphData));
            m_pendingLine = right;
         }
         // current line marks the spot for the soft truncation
         m_currentLine = right;
         // update the bounding rectangle with the current glyphs bounding rectangle
         m_textRectangle = m_textRectangle.Union(Candera::TextRendering::TextRect(left, top, right, bottom));
      }

      /*
      * Interface has to be implemented. No limitation required for the implementation of the features.
      */
      virtual bool IsRenderingEnabled() const override
      {
         return false;
      }

      /*
      * Interface has to be implemented. No limitation required for the implementation of the features.
      */
      virtual bool IsCachingEnabled() const override
      {
         return (m_reference != 0) ? m_reference->IsCachingEnabled() : true;
      }

      /*
      * Interface has to be implemented. No limitation required for the implementation of the features.
      */
      virtual Candera::TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const
      {
         return (m_reference != 0) ? m_reference->GetGlyphCacheAccess() : 0;
      }

   private:
      class HelperTextNode2DLayouter : public Candera::TextNode2DLayouter
      {
         public:
            static HelperTextNode2DLayouter& GetInstance()
            {
               FEATSTD_UNSYNCED_STATIC_OBJECT(HelperTextNode2DLayouter, s_helperTextNode2DLayouter);
               return s_helperTextNode2DLayouter;
            }

            virtual void Dispose()
            {
            }

            virtual Candera::TextRendering::TextSize OnMeasureText(Candera::TextNode2D& /*node*/, const Candera::TextRendering::TextSize& /*clientArea*/)
            {
               return Candera::TextRendering::TextSize();
            }

            virtual Candera::TextRendering::TextRect OnArrangeText(Candera::TextNode2D& /*node*/, const Candera::TextRendering::TextRect& /*clientArea*/)
            {
               return Candera::TextRendering::TextRect();
            }

            virtual Candera::Layouter* Clone() const
            {
               return &GetInstance();
            }

            static void SetPreprocessedText(Candera::TextNode2D& textNode, const Candera::TextRendering::PreprocessingContext::Iterator& it, Candera::TextRendering::LayoutingOptions& layoutingOptions)
            {
               Candera::TextRendering::TruncationToGlyphIteratorContext::SharedPointer truncation;
               Candera::TextRendering::TextRenderArgs::SharedPointer renderArgs = Candera::TextRendering::TextRenderArgs::Create();
               static_cast<void>(renderArgs->MeasureInit(textNode, layoutingOptions.GetSize(), layoutingOptions, truncation));
               GetInstance().PreprocessTextInternal(renderArgs);
               textNode.SetCustomGlyphInformation(it, renderArgs->GetTextRectangle(), renderArgs->GetLayoutRectangle());
            }
      };

      FEATSTD_MAKE_CLASS_STATIC(TextWidgetTextRenderContext);
      FEATSTD_MAKE_CLASS_UNCOPYABLE(TextWidgetTextRenderContext);

      void AddTruncationTextInLast()
      {
         if (m_textGlyphData.Size() > 0)
         {
            TextWidget2DV2::MinimalGlyphData lastGlyphData = m_textGlyphData[m_textGlyphData.Size() - 1];
            for (FeatStd::SizeType i = 0; i < m_truncationTextGlyphData->Size(); ++i)
            {
               TextWidget2DV2::MinimalGlyphData glyphData = (*m_truncationTextGlyphData)[i];
               glyphData.m_characterPosition = lastGlyphData.m_characterPosition + 1;
               // position at the end of the last line
               glyphData.m_position.x = static_cast<Candera::TextRendering::PixelPosition>(glyphData.m_position.x + m_pendingLine);
               // use the same base line as the last line
               glyphData.m_position.y = static_cast<Candera::TextRendering::PixelPosition>(lastGlyphData.m_glyphBaseLine - glyphData.m_glyphTop);
               m_textGlyphData.Add(glyphData);
            }
         }
         m_bIsTextTruncated = true;
      }

      Candera::TextRendering::LayoutingOptions m_layoutingOptions;
      Candera::TextRendering::ShapingOptions m_shapingOptions;
      Candera::TextRendering::PixelPosition m_ascender;
      Candera::TextRendering::PixelPosition m_descender;
      Candera::TextRendering::TextRect m_textRectangle;
      Candera::TextRendering::TextRect m_clipRectangle;
      Truncation m_truncation;
      Candera::enTruncationDirection m_truncationDirection;
      Candera::ScrollBehavior m_scrollBehaviour;
      Candera::TextRendering::PixelPosition m_pendingLine;
      Candera::TextRendering::PixelPosition m_currentLine;
      FeatStd::Int32 m_offsetX;
      FeatStd::Int32 m_offsetY;
      bool m_first;
      bool m_firstInLine;
      Candera::TextRendering::PixelPosition m_previousX;
      Candera::TextRendering::PixelPosition m_previousY;
      FeatStd::UInt32 m_numberOfLines;
      const Candera::TextRendering::TextRenderContext* m_reference;
      FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>* m_truncationTextGlyphData;
      FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData>& m_textGlyphData;
      FeatStd::Internal::Vector<TextWidget2DV2::MinimalGlyphData> m_pendingGlyphData;
      FeatStd::UInt16 m_truncationTextWidth;
      FeatStd::UInt32 m_maxNumberOfLines;
      bool m_multiLineLayouting;
      bool m_bIsBlitCalled;
      bool m_bIsTruncationRequired;
      bool m_bIsSkippedBlit;
      Candera::Vector2 m_additionalSize;
      std::vector<Candera::UInt32> m_charCountEachLineHelper;
      std::vector<Candera::Int32> m_rectTopEachLine;
      Candera::UInt32 m_charCount;
      bool m_bIsCurrentLineUpdated;
      bool m_bIsTextTruncated;
      bool m_bIsTruncationTextReqForLeft;
      bool m_bisFullTextEnglish;
      bool m_bIsMoreTextThanNumberOfLines;
      bool m_bCultureDependentAlignment;
      bool m_bInitialGlyphProcessing;
};


#ifdef xxxx
namespace MultibyteHelper {
static unsigned int getByteCount(const Candera::TChar* text)
{
   unsigned u8StartByte = (unsigned char)text[0];

   if (u8StartByte == 0)
   {
      return 0;
   }
   if (u8StartByte < 0xC0)
   {
      // 1 byte standard
      return 1;
   }
   if (((u8StartByte & 0xC0) == 0x80) || ((u8StartByte >= 0xC0 || u8StartByte < 0x80) == false))
   {
      // Oops: not a starting byte */
      HMI_APP_ASSERT_ALWAYS();
      return 0;
   }
   if (u8StartByte < 0xE0)
   {
      // 2  byte MultiByteChar
      return 2;
   }
   if (u8StartByte < 0xF0)
   {
      // 3 byte MultiByteChar
      return 3;
   }
   if (u8StartByte < 0xF8)
   {
      // 4 byte MultiByteChar
      HMI_APP_ASSERT(u8StartByte < 0xF5); // max unicode: (http://en.wikipedia.org/wiki/UTF-8) In November 2003,UTF-8 was restricted by RFC 3629 to end at U+10FFFF
      return 4;                           // So for a complete check 2. byte would be necessary. if leading byte == 0xF4 the 1. continuation byte must be < 0x90
   }
   HMI_APP_ASSERT_ALWAYS();
   return 0;
}


}

#endif
