/* ***************************************************************************************
* FILE:          TextStyleConfiguratorImpl.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextStyleConfiguratorImpl is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _HMIBASE_TEXT_STYLE_CONFIGURATOR_IMPL_H_
#define _HMIBASE_TEXT_STYLE_CONFIGURATOR_IMPL_H_

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <Candera/TextEngine/Style.h>
#include <map>
#include "TextStyleConfigurator.h"

namespace hmibase {
namespace widget {
namespace text {

/*TextStyleConfiguratorImpl implements the TextStyleConfigurator. It stores various extension styles registered  for corresponding base style. These extension style are used
  along with the properties of Text widget to get required SharedPointer of SharedStyle to render the text. Currently supported text widgets properties are ShrinkMethod, Bold and Italic*/
class TextStyleConfiguratorImpl : public TextStyleConfigurator
{
   public:
      typedef Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> Style;
      typedef std::map<StyleName, StyleId> StyleEntries;		//These are the main/base styles which are registered , corresponding to these styles we will have extension styles
      typedef std::map<StyleId, StyleName> StyleIdEnteries;		//These are the style Ids and stylename corresponding to base style
      typedef std::map <StyleId, Style&> ExtensionStyleEntries;	//These are the style Ids and SharedPointer of SharedStyle corresponding to base style
      typedef std::map <StyleName, Style> AssetStyleEntries;	//These are the stylename and SharedPointer of SharedStyle read from the asset. Shared pointer of style will be retrieved from this

      TextStyleConfiguratorImpl();
      ~TextStyleConfiguratorImpl();
      static TextStyleConfiguratorImpl& getInstance();

      /*Register base style with an styleId. This styleId is uniquie for each base styles*/
      bool registerStyle(StyleId styleId, StyleName styleName);

      /*Unregister base style for the styleName*/
      bool unregisterStyle(StyleId styleId, StyleName styleName);

      /*Register an extension style for the base style. The styleId is same as the base style*/
      bool registerExtensionStyle(StyleId styleId, ExtensionType extensionType, StyleName styleName);

      /*Unregister an extension style for the styleId*/
      bool unregisterExtensionStyle(StyleId styleId, ExtensionType extensionType);

      /*This stores SharedPointer of SharedStyle from asset in AssetStyleEntries */
      bool readStylesFromAsset(Candera::AssetProvider* assetProvider);

      /*This returns the style id for the corresponding base style*/
      FeatStd::Int16 getTextStyleId(StyleName styleName);

      /*This provides the SharedPointer of SharedStyle for the corresponding style Id and extension type like bold,shrinked etc*/
      bool getExtensionStyleEntries(StyleId styleId, ExtensionType extensionType, Style& style, Candera::AssetProvider* assetProvider);

      /*Provide the count of various maps */
      size_t getStyleEntriesCount();
      size_t getCondensedStyleEntriesCount();
      size_t getShrinkedStyleEntriesCount();
      size_t getAssetStyleEntriesCount();

   private:
      StyleEntries m_styleEntries;
      StyleIdEnteries m_condensedStyleIds;
      StyleIdEnteries m_shrinkedStyleIds;
      StyleIdEnteries m_boldStyleIds;
      StyleIdEnteries m_italicStyleIds;
      StyleIdEnteries m_consdensedBoldStyleIds;
      StyleIdEnteries m_shrinkedBoldStyleIds;
      ExtensionStyleEntries m_condensedStyleEntries;
      ExtensionStyleEntries m_shrinkedStyleEntries;
      ExtensionStyleEntries m_boldStyleEntries;
      ExtensionStyleEntries m_italicStyleEntries;
      ExtensionStyleEntries m_condensedBoldStyleEntries;
      ExtensionStyleEntries m_shrinkedBoldStyleEntries;
      AssetStyleEntries m_assetStyleEnteries;

      static bool s_isAssetReadDone;
};


} // namespace
} // namespace
} // namespace

#endif //_HMIBASE_TEXT_STYLE_CONFIGURATOR_IMPL_H_
