/* ***************************************************************************************
 * FILE:          StepAnimationWidget2D.cpp
 * SW-COMPONENT:  HMI-BASE
 *  DESCRIPTION:  StepAnimationWidget2D is part of HMI-Base Widget Library
 *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */
#include "widget2D_std_if.h"
#include "StepAnimationWidget2D.h"
#include <Widgets/utils/SharedTimerManager.h>

#include <View/CGI/CgiExtensions/AnimationWrapper.h>
#include <CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_ANIMATION
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/StepAnimationWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(StepAnimationWidget2D)


///////////////////////////////////////////////////////////////////////////////
hmibase::widget::util::SharedTimerManager& StepAnimationWidget2D::getSharedTimerManager()
{
   static hmibase::widget::util::SharedTimerManager _instance;
   return _instance;
}


///////////////////////////////////////////////////////////////////////////////
StepAnimationWidget2D::StepAnimationWidget2D() : Base(), _invalid(true), _timerAcquired(false), _acquiredTimerSharedInstance(-1), _keyframeIndex(0)
{
}


///////////////////////////////////////////////////////////////////////////////
StepAnimationWidget2D::~StepAnimationWidget2D()
{
   stopAnimation();
}


///////////////////////////////////////////////////////////////////////////////
bool StepAnimationWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const StepAnimationWidget2D* original = CLONEABLE_WIDGET_CAST<const StepAnimationWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetPropertyAnimation(original->GetPropertyAnimation());
      if (GetPropertyAnimation() == Candera::AnimateWidgetProperty)
      {
         // copy the widget related properties only if we animate a widget
         SetAnimatedWidget(controlTemplateMap.ResolveWidgetClone(original->GetAnimatedWidget()));
         SetWidgetProperty(original->GetWidgetProperty());
         SetChannelCount(original->GetChannelCount());
      }

      SetShouldRun(original->GetShouldRun());
      SetSharedTimerInstance(original->GetSharedTimerInstance());
      SetTimerDelay(original->GetTimerDelay());
      SetEndKeyframeIndex(original->GetEndKeyframeIndex());

      _keyframeValues = original->_keyframeValues;

      cloned = true;
   }
   return cloned;
}


///////////////////////////////////////////////////////////////////////////////
bool StepAnimationWidget2D::OnMessage(const Message& oMsg)
{
   if (Base::OnMessage(oMsg))
   {
      return true;
   }

   bool consumed = false;
   if (TimerExpiredMsg::ID == oMsg.GetId())
   {
      const TimerExpiredMsg* msg = Courier::message_cast<const TimerExpiredMsg*>(&oMsg);
      // check if it is our acquired timer the one which expired
      if ((msg != NULL) && (msg->GetTimer() != NULL)
            && _timerAcquired && getSharedTimerManager().matches(_acquiredTimerSharedInstance, *(msg->GetTimer())))
      {
         checkAndSetKeyframeIndex(msg->GetRepeatCnt());
         animate();
      }
   }

   return consumed;
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _invalid = true;
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();

   _invalid = true;
   triggerUpdate();
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::Update()
{
   Base::Update();

   // update only if it is invalid to avoid continuous updated on wrong config
   if ((GetNode() != NULL) && _invalid)
   {
      if (GetShouldRun())
      {
         startAnimation();
      }
      else
      {
         stopAnimation(true);
      }
      _invalid = false;
   }
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::OnParentViewRenderingEnabled(bool viewLoaded)
{
   Base::OnParentViewRenderingEnabled(viewLoaded);

   if (!viewLoaded)
   {
      // stop animation on view hide
      stopAnimation(true);
   }
   else
   {
      // set invalid on view show so that the next update starts the animation
      _invalid = true;
   }
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::startAnimation()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StartAnimation delay=%u sharedInst=%d effChannelCount=%d valuesCount=%d %s",
                       GetTimerDelay(), GetSharedTimerInstance(), getEffectiveChannelCount(), _keyframeValues.GetCount(),
                       HMIBASE_TO_STRING_VW(this)));

   // stop animation first, it has no effect if already stopped
   stopAnimation();

   // create property setter
   createPropertySetter();

   // acquire timer only if shared timer instance and timer delay are specified
   if (!_propertySetter.PointsToNull() && (GetTimerDelay() > 0))
   {
      _acquiredTimerSharedInstance = GetSharedTimerInstance();
      _timerAcquired = getSharedTimerManager().acquire(_acquiredTimerSharedInstance, GetTimerDelay());
   }
   _keyframeIndex = 0;
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::stopAnimation(bool resetToEndKeyframe)
{
   // check if timer is not already released
   if (_timerAcquired)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StopAnimation sharedInstance=%d %s",
                          GetSharedTimerInstance(), HMIBASE_TO_STRING_VW(this)));

      getSharedTimerManager().release(_acquiredTimerSharedInstance);
      _acquiredTimerSharedInstance = -1;
      _timerAcquired = false;

      if (resetToEndKeyframe && (GetEndKeyframeIndex() >= 0))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StopAnimation Reset to end keyframe index=%d %s",
                             GetEndKeyframeIndex(), HMIBASE_TO_STRING_VW(this)));

         checkAndSetKeyframeIndex(static_cast<FeatStd::UInt32>(GetEndKeyframeIndex()));
         animate();
      }
   }
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::animate()
{
   // the Values array stores the values for all keyframes.
   // depending on the type of the animated property the number of values for each keyframe is between 1 and 4.
   // - 1 value is required for simple types (bool, int, float);
   // - 2 values are required for Vector2;
   // - 3 values are required for Vector3;
   // - 4 values are required for Color, Rectangle and Margin;

   size_t channelCount = getEffectiveChannelCount();
   if ((GetNode() != NULL) && (!_propertySetter.PointsToNull())
         && (_keyframeValues.GetCount() > 0)
         && (_keyframeValues.GetCount() >= channelCount)
         && (_keyframeValues.GetCount() >= static_cast<FeatStd::UInt32>(_keyframeIndex + 1) * channelCount))
   {
      // keyframe values stored the data into an array, if this changes we will have to copy it before passing it to the property setter
      _propertySetter->Set(&_keyframeValues.Get(static_cast<FeatStd::UInt32>(_keyframeIndex) * static_cast<FeatStd::UInt32>(channelCount)));

      Invalidate();
   }
}


///////////////////////////////////////////////////////////////////////////////
static void getWidgetChannelList(FeatStd::Int channelCount, FeatStd::Internal::Vector<Candera::Internal::AnimatedPropertyChannels>& apcList)
{
   // for widget properties the actual values are quite flexible: X, Red, Left and a few others are used for 1st channel; Y, Green, Top for the 2nd channel; etc.
   switch (channelCount)
   {
      default:
      case 1:
         apcList.Add(Candera::Internal::apcDefault);
         break;
      case 2:
         apcList.Add(Candera::Internal::apcX);
         apcList.Add(Candera::Internal::apcY);
         break;
      case 3:
         apcList.Add(Candera::Internal::apcX);
         apcList.Add(Candera::Internal::apcY);
         apcList.Add(Candera::Internal::apcZ);
         break;
      case 4:
         apcList.Add(Candera::Internal::apcRed);
         apcList.Add(Candera::Internal::apcGreen);
         apcList.Add(Candera::Internal::apcBlue);
         apcList.Add(Candera::Internal::apcAlpha);
         break;
   }
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::createPropertySetter()
{
   _propertySetter.Release();

   switch (GetPropertyAnimation())
   {
      case Candera::AnimateAlphaValue:
         _propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::AlphaNode2DPropertySetter>(GetNode());
         break;
      case Candera::AnimatePosition:
         _propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::Transformable2DTranslatePropertySetter>(GetNode());
         break;
      case Candera::AnimateScale:
         _propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::Transformable2DScalePropertySetter>(GetNode());
         break;
      case Candera::AnimateRotation:
         _propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::Transformable2DRotatePropertySetter>(GetNode());
         break;
      case Candera::AnimateRendering:
         _propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::RenderingEnabledNode2DPropertySetter>(GetNode());
         break;
      case Candera::AnimateWidgetProperty:
      {
         Candera::WidgetBase* widget = GetAnimatedWidget();
         if (widget != NULL)
         {
            FeatStd::Internal::Vector<Candera::Internal::AnimatedPropertyChannels> apcList;
            getWidgetChannelList(GetChannelCount(), apcList);

            SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetWidgetProperty());
            _propertySetter = Candera::Internal::AnimationPropertySetterFactory::CreatePropertyMetaInfoSetter(widget->GetMetaInfo(), widget, GetWidgetProperty().GetCString(), apcList);
            if (_propertySetter.PointsToNull())
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter Failed to create widget property setter target=%50s.%10s %s",
                                  HMIBASE_TO_STRING_W(widget), GetWidgetProperty().GetCString(),
                                  HMIBASE_TO_STRING_VW(this)));
            }
            SECURE_FEATSTD_STRING_ACCESS_END();
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter No widget to animate %s", HMIBASE_TO_STRING_VW(this)));
         }
      }
      break;

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter Invalid PropertyAnimation=%d %s",
                            static_cast<int>(GetPropertyAnimation()), HMIBASE_TO_STRING_VW(this)));
         break;
   }
}


///////////////////////////////////////////////////////////////////////////////
size_t StepAnimationWidget2D::getEffectiveChannelCount() const
{
   switch (GetPropertyAnimation())
   {
      default:
      case Candera::AnimateAlphaValue:
      case Candera::AnimateRotation:
      case Candera::AnimateRendering:
         return 1;// simple type (bool, int, float)
      case Candera::AnimatePosition:
      case Candera::AnimateScale:
         return 2;// Vector2 (x, y)
      case Candera::AnimateWidgetProperty:
         return GetChannelCount();// depends on widget property type (between 1 and 4)
   }
}


///////////////////////////////////////////////////////////////////////////////
size_t StepAnimationWidget2D::getEffectiveKeyframeCount() const
{
   size_t channelCount = getEffectiveChannelCount();
   if (channelCount == 0)
   {
      channelCount = 1;
   }
   return _keyframeValues.GetCount() / channelCount;
}


///////////////////////////////////////////////////////////////////////////////
void StepAnimationWidget2D::checkAndSetKeyframeIndex(size_t index)
{
   size_t keyframeCount = getEffectiveKeyframeCount();
   if (keyframeCount == 0)
   {
      keyframeCount = 1;
   }
   _keyframeIndex = static_cast<FeatStd::Int32>(index % keyframeCount);
}
