/* ***************************************************************************************
* FILE:          SpellerWidget2D_Ctrl.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpellerWidget2D_Ctrl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "SpellerWidget2D_Ctrl.h"
#include "Widgets/2D/Speller/SpellerWidget2D.h"
#include "Widgets/2D/Speller/SpellerConstants.h"

using namespace Courier;
using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SPELLER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpellerWidget2D_Ctrl.cpp.trc.h"
#endif


/****************************************************************************
* Req ID      : NA
* Function    : SpellerWidget2D_Ctrl()
* Called by   : HMI Framework when SpellerWidget2D_Ctrl object is created
* Description : Constructor of class where initializations are done
* Parameters  : None
* Return      : None
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
*****************************************************************************/
SpellerWidget2D_Ctrl::SpellerWidget2D_Ctrl():
   m_pWidget(NULL)
{
   m_lockState = false;
}


/****************************************************************************
* Req ID      : NA
* Function    : ~SpellerWidget2D_Ctrl()
* Called by   : HMI Framework when SpellerWidget2D_Ctrl object is deleted
* Description : Destructor of class where we reset necessary variables
*               while moving out of scope
* Parameters  : None
* Return      : None
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
*****************************************************************************/
SpellerWidget2D_Ctrl::~SpellerWidget2D_Ctrl()
{
   m_pWidget = NULL;
}


/****************************************************************************
* Req ID      : NA
* Function    : vSetWidget()
* Called by   : Physical widget SpellerWidget2D(Model in MVC)constructor
* Description : Assigns the Main widget instance pointer to SpellerCtrl class
* Parameters  : SpellerWidget2D*
* Return      : None
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
*****************************************************************************/
void SpellerWidget2D_Ctrl::vSetWidget(SpellerWidget2D* pWidget)
{
   m_pWidget = pWidget;
}


/****************************************************************************
* Req ID      : NA
* Function    : OnTouch
* Called by   : HMI Framework
* Description : Overloaded function to be used only if the widget should perform
*               a custom hit test on a Touch message
* Parameters  : Camera reference and Touch coordinates
* Return      : true  - Touch intersects the node
*               false - Touch did not intersect the node
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
bool SpellerWidget2D_Ctrl::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   COURIER_UNUSED2(camera2D, point);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: OnTouch called", this));
   return false;
}


/****************************************************************************
* Req ID      : NA
* Function    : OnMessage
* Called by   : HMI Framework when a message is received
* Description : The return parameter defines that the message shall be
*               routed to the succeeding message receiving entities of the
*               dispatching path (true) or if the message is fully consumed
*               and any further dispatching of that message shall be stopped
*               (false).
* Parameters  : Message object to be processed.
* Return      : true  - if the message is consumed
*               false - if the message shall be sent to following message
*               receiving entities (default)
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
bool SpellerWidget2D_Ctrl::OnMessage(const Message& msg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: OnMessage with MsgID: [%d]", this, msg.GetId()));

   bool bIsMsgConsumed = false;

   switch (msg.GetId())
   {
      case ButtonReactionMsg::ID:
         bIsMsgConsumed = bOnButtonMsg(msg);
         break;
#ifdef xxxhtcFocusHandlingxxx
         //Rotary event
      case KBD_ROTARY_EVENT::ID:
         bIsMsgConsumed = bOnRotaryMsg(msg);
         break;

#ifdef KBD_OK_EVENT_NOT_LONGER_SUPPORTED // fr83hi todo, code should be removed if no dependences
         //OK or Enter events
      case KBD_OK_EVENT::ID:
         bIsMsgConsumed = bOnOKMsg(msg);
         break;
#endif
         //Joystick KBD events
      case JSK_KBD_EVENT::ID:
         bIsMsgConsumed = bOnJoystickKBDMsg(msg);
         break;
#endif
      default:
         break;
   }

   return bIsMsgConsumed;
}


/***************************************************************************/
bool SpellerWidget2D_Ctrl::bOnButtonMsg(const Message& msg)
{
   bool bMsgRetVal = false;

   const ButtonReactionMsg* pButtonReactionMsg = message_cast<const ButtonReactionMsg*>(&msg);
   if ((!pButtonReactionMsg) || (!m_pWidget) ||
         (::Courier::ViewId(m_pWidget->pcGetActivatedSceneName()) != pButtonReactionMsg->GetView()))
   {
      return false;
   }

   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return false;
   }

   Candera::UInt8 u8Index = 0;

   const Courier::Identifier l_ButtonReactionMsgSender = pButtonReactionMsg->GetSender();
   while (pSpellerBtn->m_pAttachedWidget)
   {
      if ((pSpellerBtn->m_pAttachedWidget) &&
            (l_ButtonReactionMsgSender) == Courier::Identifier(pSpellerBtn->m_pAttachedWidget->GetLegacyName()))
      {
         bMsgRetVal = bHandleButtonMsg(msg, u8Index);
         break;
      }
      else
      {
         pSpellerBtn++;
         u8Index++;
      }
   }

   return bMsgRetVal;
}


/***************************************************************************/
bool SpellerWidget2D_Ctrl::bHandleButtonMsg(const Message& msg, Candera::UInt8 u8Index)
{
   bool bMsgRetVal = false;

   const ButtonReactionMsg* pButtonReactionMsg = message_cast<const ButtonReactionMsg*>(&msg);
   if (!pButtonReactionMsg)
   {
      return false;
   }

   switch (pButtonReactionMsg->GetEnReaction())
   {
      case enPress:
      {
         vOnButtonPress(u8Index);
         bMsgRetVal = true;
         break;
      }

      case enRelease:
      case enShortPressRelease:
      {
         vOnButtonRelease(u8Index, pButtonReactionMsg->GetEnReaction());
         bMsgRetVal = true;
         break;
      }

      case enLongPress:
      case enShortPress:
      {
         vOnButtonShortORLongPress(u8Index, pButtonReactionMsg->GetEnReaction());
         bMsgRetVal = true;
         break;
      }
      case enRepeatPressSecondary:
      {
         onButtonRepeatPressSec(u8Index);
         bMsgRetVal = true;
         break;
      }

      default:
         break;
   }

   return bMsgRetVal;
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vOnButtonPress(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }

   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::UInt8 l_u8CurFocusedIndex = static_cast<Candera::UInt8>(m_pWidget->u32GetCurFocusedIndex());

   if (u8Index != l_u8CurFocusedIndex)
   {
#ifdef VARIANT_S_FTR_ENABLE_SPELLER_WIDGET_SET_FOCUS_ON_BUTTONS
      //ensure to first disable the focus on the current button
      if ((pSpellerBtn[l_u8CurFocusedIndex]).m_pAttachedWidget)
      {
         (pSpellerBtn[l_u8CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
      }

      //now render the focus image on the current touched button
      if ((pSpellerBtn[u8Index]).m_pAttachedWidget)
      {
         pSpellerBtn[u8Index].m_pAttachedWidget->OnFocus();
      }
#endif

      //Store the current focussed button index for future reference
      m_pWidget->vSetCurFocusedIndex(u8Index);
   }
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vOnButtonRelease(Candera::UInt8 u8Index, const enReaction& reaction)
{
   if (!m_pWidget)
   {
      return;
   }

   if (bIsSubSpellerChar(u8Index) && m_pWidget->isBtnReactionToOpenSubSpeller(reaction))
   {
      //activate the sub-speller and change the activation flag
      m_pWidget->vActivateSubSpeller(u8Index);
   }
   else
   {
      //Free mode related actions
      if (FreeMode == m_pWidget->GetMode())
      {
         vHandleFreeModeActions(u8Index);
      }
      //Match related actions
      else if (MatchMode == m_pWidget->GetMode())
      {
         vHandleMatchModeActions(u8Index);
      }

      //deactivate the sub-speller, ideally this should be done only once
      if (m_pWidget->bGetSubSpellerActivationStatus())
      {
         m_pWidget->vDeactivateSubSpeller();
      }
   }

   //Mode independent actions such as layout toggling
   vHandleModeIndependentActions(u8Index);
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vOnButtonShortORLongPress(Candera::UInt8 u8Index, const enReaction& reaction)
{
   if (!m_pWidget)
   {
      return;
   }
   //Launch SUB SPELLER for either SHORT OR LONG PRESS
   if (bIsSubSpellerChar(u8Index) && m_pWidget->isBtnReactionToOpenSubSpeller(reaction))
   {
      //activate the sub-speller and change the activation flag
      m_pWidget->vActivateSubSpeller(u8Index);
   }
   else if (reaction == enLongPress)
   {
      vHandleDeleteLongPress(u8Index);

      //#ifdef SPLR_ENABLE_LONGPRESS_OF_ZERO removed in order to make it property
      if (m_pWidget->GetZeroKeyLongPressed() == true)
      {
         vHandleZeroLongPress(u8Index);
      }

      //deactivate the sub-speller, ideally this should be done only once
      if (m_pWidget->bGetSubSpellerActivationStatus())
      {
         m_pWidget->vDeactivateSubSpeller();
      }
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_005
*               HTC_Speller_REQ_007
* Function    : vHandleFreeModeActions
* Called by   : vOnSpellerBtnRelease
* Description : Function to handle the set of actions if the Speller mode is
                configured in FreeMode
* Parameters  : None
* Return      : None
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D_Ctrl::vHandleFreeModeActions(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::String l_pressedChar = "";

   //TODO("do boundary check")
   switch (pSpellerBtn[u8Index].m_enType)
   {
         //Space character
      case SPLRBTNTYPE_Control_Space:
         l_pressedChar = SPACE_CHAR;
         break;

         //ms_nDelAllCharacter
      case SPLRBTNTYPE_Control_DeleteAll:
         l_pressedChar = DELETEALL_CHAR;
         break;

         //ms_nDelCharacter
      case SPLRBTNTYPE_Control_Delete:
         l_pressedChar = DELETE_CHAR;
         break;

         //ms_nRShCharacter
      case SPLRBTNTYPE_Control_CursorRight_Shift:
         l_pressedChar = SHIFT_CURSOR_RIGHT;
         break;

         //ms_nLShCharacter
      case SPLRBTNTYPE_Control_CursorLeft_Shift:
         l_pressedChar = SHIFT_CURSOR_LEFT;
         break;

         //character buttons
      case SPLRBTNTYPE_Character:
      {
         if (pSpellerBtn[u8Index].m_pTextWidget)
         {
            l_pressedChar = (pSpellerBtn[u8Index].m_pTextWidget->GetText());
         }
      }
      break;

      default:
         break;
   }

   switch (pSpellerBtn[u8Index].m_enType)
   {
      case SPLRBTNTYPE_Control_ToggleLayoutType:
      case SPLRBTNTYPE_Control_ToggleCharCase:
      case SPLRBTNTYPE_Control_Shift_1:
      case SPLRBTNTYPE_Control_Shift_2:
      case SPLRBTNTYPE_Control_Shift_3:
      case SPLRBTNTYPE_Control_Shift_4:
      case SPLRBTNTYPE_Control_Shift_5:
         //nothing to do
         break;

      default:
      {
         //Modify the member variable and update the DataBinding property
         m_pWidget->SetSpellerKeyPressed(l_pressedChar);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(l_pressedChar);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: In FreeMode, updating the databinding to EF '%s'", this, l_pressedChar.GetCString()));
         SECURE_FEATSTD_STRING_ACCESS_END();
         break;
      }
   }
}


void SpellerWidget2D_Ctrl::onButtonRepeatPressSec(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }
   Candera::String l_pressedChar = "";
   if (bIsSubSpellerChar(u8Index))
   {
      //activate the sub-speller and change the activation flag
      m_pWidget->vActivateSubSpeller(u8Index);
   }
   else
   {
      //Free mode related actions
      if (FreeMode == m_pWidget->GetMode())
      {
         if (pSpellerBtn[u8Index].m_enType == SPLRBTNTYPE_Character)
         {
            if (pSpellerBtn[u8Index].m_pTextWidget)
            {
               l_pressedChar = (pSpellerBtn[u8Index].m_pTextWidget->GetText());
               m_pWidget->SetSpellerKeyPressed(l_pressedChar);
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(l_pressedChar);
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: In FreeMode, updating the databinding to EF '%s'", this, l_pressedChar.GetCString()));
               SECURE_FEATSTD_STRING_ACCESS_END();
            }
         }
         else
         {
            if (pSpellerBtn[u8Index].m_enType == SPLRBTNTYPE_Control_Delete)
            {
               l_pressedChar = DELETE_CHAR;
               m_pWidget->SetSpellerKeyPressed(l_pressedChar);
            }

            if (pSpellerBtn[u8Index].m_enType == SPLRBTNTYPE_Control_CursorRight_Shift)
            {
               l_pressedChar = SHIFT_CURSOR_RIGHT;
               m_pWidget->SetSpellerKeyPressed(l_pressedChar);
            }
            if (pSpellerBtn[u8Index].m_enType == SPLRBTNTYPE_Control_CursorLeft_Shift)
            {
               l_pressedChar = SHIFT_CURSOR_LEFT;
               m_pWidget->SetSpellerKeyPressed(l_pressedChar);
            }
         }
      }
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_005
*               HTC_Speller_REQ_007
* Function    : vHandleMatchModeActions
* Called by   : vOnSpellerBtnRelease
* Description : Function to handle the set of actions if the Speller mode is
                configured in MatchMode
* Parameters  : None
* Return      : None
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D_Ctrl::vHandleMatchModeActions(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::String l_pressedChar = "";

   //TODO("do boundary check")
   switch (pSpellerBtn[u8Index].m_enType)
   {
         //Space character
      case SPLRBTNTYPE_Control_Space:
         l_pressedChar = SPACE_CHAR;
         break;

         //ms_nDelAllCharacter
      case SPLRBTNTYPE_Control_DeleteAll:
         l_pressedChar = DELETEALL_CHAR;
         break;

         //ms_nDelCharacter
      case SPLRBTNTYPE_Control_Delete:
         l_pressedChar = DELETE_CHAR;
         break;

         //character buttons
      case SPLRBTNTYPE_Character:
      {
         if (pSpellerBtn[u8Index].m_pTextWidget)
         {
            l_pressedChar = pSpellerBtn[u8Index].m_pTextWidget->GetText();
         }
      }
      break;

      default:
         break;
   }

   //Fire the courier msg & disable buttons only if the button type is delete, deleteAll or character
   if ((SPLRBTNTYPE_Control_Delete == (pSpellerBtn[u8Index].m_enType)) ||
         (SPLRBTNTYPE_Control_DeleteAll == (pSpellerBtn[u8Index].m_enType)) ||
         (SPLRBTNTYPE_Character == (pSpellerBtn[u8Index].m_enType) || SPLRBTNTYPE_Control_Space == (pSpellerBtn[u8Index].m_enType)))
   {
      m_pWidget->postSpellerKeyPressedMsg(l_pressedChar);

      //After key press from Speller, disable all the button nodes
      vToggleAllCharBtnEnableStatus(false);
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_005
*               HTC_Speller_REQ_007
* Function    : vHandleModeIndependentActions
* Called by   : vOnSpellerBtnRelease
* Description : Function to handle the common set of actions to be done irre-
                spective of the Speller mode
* Parameters  : None
* Return      : None
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D_Ctrl::vHandleModeIndependentActions(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }
   vPostPressedControlCharMsg(u8Index);

   //TODO("do boundary check")
   switch (pSpellerBtn[u8Index].m_enType)
   {
         //ms_nSymCharacter
      case SPLRBTNTYPE_Control_ToggleLayoutType:
         m_pWidget->vToggleButtonRelease(SPLRBTNTYPE_Control_ToggleLayoutType);
         break;

         //ms_nShCharacter
      case SPLRBTNTYPE_Control_ToggleCharCase:
         m_pWidget->vToggleButtonRelease(SPLRBTNTYPE_Control_ToggleCharCase);
         break;

      case SPLRBTNTYPE_Control_Shift_1:
      case SPLRBTNTYPE_Control_Shift_2:
      case SPLRBTNTYPE_Control_Shift_3:
      case SPLRBTNTYPE_Control_Shift_4:
      case SPLRBTNTYPE_Control_Shift_5:
         m_pWidget->vToggleButtonRelease(pSpellerBtn[u8Index].m_enType);
         break;

      default:
         break;
   }

   //Fire the message to notify the control character pressed
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vHandleDeleteLongPress(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::String l_pressedChar = "";

   //TODO("do boundary check")
   switch (pSpellerBtn[u8Index].m_enType)
   {
         //On LongPress of ms_nDelCharacter, do the functionality of ms_nDelAllCharacter
      case SPLRBTNTYPE_Control_Delete:
         l_pressedChar = DELETEALL_CHAR;
         break;

      default:
         break;
   }

   //Fire the courier msg & disable buttons only if the button type is delete
   if ((SPLRBTNTYPE_Control_Delete == (pSpellerBtn[u8Index].m_enType)) &&
         (MatchMode == m_pWidget->GetMode()))
   {
      m_pWidget->postSpellerKeyPressedMsg(l_pressedChar);
      //After key press from Speller, disable all the button nodes
      vToggleAllCharBtnEnableStatus(false);
   }
   else if ((SPLRBTNTYPE_Control_Delete == (pSpellerBtn[u8Index].m_enType)) &&
            (FreeMode == m_pWidget->GetMode()))
   {
      //Modify the member variable and update the DataBinding property
      m_pWidget->SetSpellerKeyPressed(l_pressedChar);
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(l_pressedChar);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: In FreeMode, updating the databinding to EF '%s'", this, l_pressedChar.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vHandleZeroLongPress(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::String l_pressedChar = "";
   const Candera::TChar* l_pressedButtonText = "";

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(pSpellerBtn[u8Index].m_pTextWidget->GetText());
   if (pSpellerBtn[u8Index].m_pTextWidget)
   {
      l_pressedButtonText = pSpellerBtn[u8Index].m_pTextWidget->GetText().GetCString();
   }

   //TODO("do boundary check")
   switch (pSpellerBtn[u8Index].m_enType)
   {
         //On LongPress of '0', enter the character '+'
      case SPLRBTNTYPE_Character:
      {
         if ((pSpellerBtn[u8Index].m_pAttachedWidget) &&
               (0 == FeatStd::Internal::TextEncoding::Compare(l_pressedButtonText, "0"))
            )
         {
            l_pressedChar = "+";
         }
      }
      break;

      default:
         break;
   }

   //Fire the courier msg & disable buttons only if the button type is character with Text as '0'
   if ((0 == FeatStd::Internal::TextEncoding::Compare(l_pressedButtonText, "0")) &&
         (MatchMode == m_pWidget->GetMode()))
   {
      m_pWidget->postSpellerKeyPressedMsg(l_pressedChar);
      //After key press from Speller, disable all the button nodes
      vToggleAllCharBtnEnableStatus(false);
   }
   else if ((0 == FeatStd::Internal::TextEncoding::Compare(l_pressedButtonText, "0")) &&
            (FreeMode == m_pWidget->GetMode()))
   {
      //Modify the member variable and update the DataBinding property
      m_pWidget->SetSpellerKeyPressed(l_pressedChar);
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(l_pressedChar);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: In FreeMode, updating the databinding to EF '%s'", this, l_pressedChar.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vPostPressedControlCharMsg(Candera::UInt8 u8Index)
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }

   switch (pSpellerBtn[u8Index].m_enType)
   {
         //check for all possible valid control characters where this message should be fired
      case SPLRBTNTYPE_Control_Ok:
      case SPLRBTNTYPE_Control_Space:
      case SPLRBTNTYPE_Control_ToggleLayoutType:
      case SPLRBTNTYPE_Control_ToggleCharCase:
      case SPLRBTNTYPE_Control_Shift_1:
      case SPLRBTNTYPE_Control_Shift_2:
      case SPLRBTNTYPE_Control_Shift_3:
      case SPLRBTNTYPE_Control_Shift_4:
      case SPLRBTNTYPE_Control_Shift_5:
      case SPLRBTNTYPE_Control_DeleteAll:
      case SPLRBTNTYPE_Control_Delete:
      case SPLRBTNTYPE_Control_CursorRight_Shift:
      case SPLRBTNTYPE_Control_CursorLeft_Shift:
      case SPLRBTNTYPE_Control_CustomButton_1:
      case SPLRBTNTYPE_Control_CustomButton_2:
      case SPLRBTNTYPE_Control_CustomButton_3:
      case SPLRBTNTYPE_Control_CustomButton_4:
      case SPLRBTNTYPE_Control_CustomButton_5:
      {
         //fie the relevant message now with the character as the parameter
         SpellerCtrlKeyPressed* pSpellerCtrlKeyPressed = COURIER_MESSAGE_NEW(SpellerCtrlKeyPressed)(
                  Courier::ViewId(m_pWidget->GetParentView()->GetId().CStr()),
                  Courier::Identifier(pSpellerBtn[u8Index].m_pAttachedWidget->GetName()),
                  m_pWidget->enGetButtonTypeCtrlChar(pSpellerBtn[u8Index].m_enType)
               );

         if (NULL != pSpellerCtrlKeyPressed)
         {
            pSpellerCtrlKeyPressed->Post();
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: fired the courier msg 'SpellerCtrlKeyPressed'", this));
         }
         break;
      }

      default:
         break;
   }
}


#ifdef xxxhtcFocusHandlingxxx
/***************************************************************************/
bool SpellerWidget2D_Ctrl::bOnRotaryMsg(const Message& msg)
{
   //Use rotary events for Cursor movement
   bool bMsgRetVal = false;

   TODO("WrapAround behaviour to be implemented in future");
   const KBD_ROTARY_EVENT* pRotaryMsg = message_cast<const KBD_ROTARY_EVENT*>(&msg);
   if (!pRotaryMsg)
   {
      return false;
   }
   switch (pRotaryMsg->GetDirection())
   {
      case TurnPos:
      {
         //Shift the focus to next button, if any
         vOnRotaryTurnPosMsg();
         bMsgRetVal = true;
         break;
      }
      case TurnNeg:
      {
         //Shift the focus to prev button, if any
         vOnRotaryTurnNegMsg();
         bMsgRetVal = true;
         break;
      }

      default:
         break;
   }

   return bMsgRetVal;
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vOnRotaryTurnPosMsg()
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }
   Candera::UInt32 l_u32CurFocusedIndex = m_pWidget->u32GetCurFocusedIndex();

   //First check if you still have a button on the right to focus
   if ((pSpellerBtn[l_u32CurFocusedIndex + 1]).m_pAttachedWidget)
   {
      //if so then first disable the focus on the current button
      (pSpellerBtn[l_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
      //now render the focus image on the next button to the right
      (pSpellerBtn[++l_u32CurFocusedIndex]).m_pAttachedWidget->OnFocus();

      //Store the current focussed button index for future reference
      m_pWidget->vSetCurFocusedIndex(l_u32CurFocusedIndex);
   }
   else
   {
      //nothing to do, keep focusing the existing button
   }
}


/***************************************************************************/
void SpellerWidget2D_Ctrl::vOnRotaryTurnNegMsg()
{
   if (!m_pWidget)
   {
      return;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return;
   }
   Candera::UInt32 l_u32CurFocusedIndex = m_pWidget->u32GetCurFocusedIndex();

   //First check if you still have a button on the left to focus
   if ((l_u32CurFocusedIndex > 0) && ((pSpellerBtn[l_u32CurFocusedIndex - 1]).m_pAttachedWidget))
   {
      //if so then first disable the focus on the current button
      (pSpellerBtn[l_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
      //now render the focus image on the next button to the right
      (pSpellerBtn[--l_u32CurFocusedIndex]).m_pAttachedWidget->OnFocus();

      //Store the current focussed button index for future reference
      m_pWidget->vSetCurFocusedIndex(l_u32CurFocusedIndex);
   }
   else
   {
      //nothing to do, keep focusing the existing button
   }
}


/***************************************************************************/
bool SpellerWidget2D_Ctrl::bOnOKMsg(const Message& /*msg*/)
{
   bool bMsgRetVal = false;

   if (!m_pWidget)
   {
      return false;
   }
   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (NULL == pSpellerBtn)
   {
      return false;
   }

   Candera::UInt8 l_u8CurFocusedIndex = static_cast<Candera::UInt8>(m_pWidget->u32GetCurFocusedIndex());
   //Check if the button is enabled, only then further processing should be done
   if (
      (NULL == (&pSpellerBtn[l_u8CurFocusedIndex])) ||
      (NULL == ((&pSpellerBtn[l_u8CurFocusedIndex])->m_pAttachedWidget)) ||
      (False == ((&pSpellerBtn[l_u8CurFocusedIndex])->m_pAttachedWidget->IsEnabled()))
   )
   {
      return false;
   }

#ifdef KBD_OK_EVENT_NOT_LONGER_SUPPORTED // fr83hi todo, code should be removed if no dependences
   const KBD_OK_EVENT* pOKMsg = message_cast<const KBD_OK_EVENT*>(&msg);
   if (!pOKMsg)
   {
      return false;
   }
   switch (pOKMsg->GetState())
   {
      case Press:
      {
         //call relevant interface of ButtonWidget2d
         //Further handling done in Speller bHandleButtonMsg()
         pSpellerBtn[l_u8CurFocusedIndex].m_pAttachedWidget->Press();
         bMsgRetVal = true;
         break;
      }

      case Release:
      {
         //call relevant interface of ButtonWidget2d
         //Further handling done in Speller bHandleButtonMsg()
         pSpellerBtn[l_u8CurFocusedIndex].m_pAttachedWidget->Release();
         bMsgRetVal = true;
         break;
      }

      default:
      {
         //handling of Joystick Shorthold & Longhold is not required as calling
         //vOnPress() of ButtonWidget2D should take care of firing ButtonReactionMsg
         //with appropriate parameter as reaction as when relevant
         //if so, then the Speller bHandleButtonMsg() will take care of further handling
         break;
      }
   }
#endif
   return bMsgRetVal;
}


/***************************************************************************/
bool SpellerWidget2D_Ctrl::bOnJoystickKBDMsg(const Message& msg)
{
   bool bMsgRetVal = false;

   const JSK_KBD_EVENT* pJoyStickMsg = message_cast<const JSK_KBD_EVENT*>(&msg);
   if ((NULL != pJoyStickMsg) && (NULL != m_pWidget))
   {
      //calling the function vPostJoyStickMessage() is required so that JACContainerWidget2D
      //can further pass the focus to relevant widget based on the user event (GetKey())
      //Note: doing this should also make the OnLostFocus() of Speller being called if focus
      //is shifted to another widget or another Speller instance.
      m_pWidget->vPostJoyStickMessage(pJoyStickMsg->GetKey(), m_pWidget->GetLegacyName(), pJoyStickMsg->GetState(), m_pWidget->Base::cou8GetAssociatedContFocusOrder());
      bMsgRetVal = true;
   }

   return bMsgRetVal;
}


#endif
/***************************************************************************/
bool SpellerWidget2D_Ctrl::bIsSubSpellerChar(Candera::UInt8 u8Index)
{
   bool bRetValue = false;
   if (!m_pWidget)
   {
      return bRetValue;
   }

   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   if (!pSpellerBtn)
   {
      return bRetValue;
   }

   //TODO("do boundary check for the array")
   if (SUBSPLR_NoSubSplr != (pSpellerBtn[u8Index].m_enSubSplrExist))
   {
      //it's a sub-speller character
      //Refer enSubSplr in SpellerWidget2D.h for further details
      bRetValue = true;
   }

   return bRetValue;
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_001
* Function    : vToggleAllCharBtnEnableStatus
* Called by   : vOnSpellerBtnRelease
* Description : Function to toggle the enable property of all the Speller
*               character buttons
* Parameters  : bool, the enable status received
* Return      : void
* History     : 2013-09-10(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D_Ctrl::vToggleAllCharBtnEnableStatus(const bool bEnableStatus)
{
   if (!m_pWidget && !m_lockState)
   {
      return;
   }

   struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
   while ((pSpellerBtn) && (pSpellerBtn->m_pAttachedWidget))
   {
      if (SPLRBTNTYPE_Character == pSpellerBtn->m_enType || SPLRBTNTYPE_Control_Space == pSpellerBtn->m_enType)
      {
         //toggle only the character buttons
         pSpellerBtn->m_pAttachedWidget->SetEnabled(bEnableStatus);
      }
      else if (SPLRBTNTYPE_Character_Empty == pSpellerBtn->m_enType)
      {
         //ensure to always keep the EMPTY characters as disabled unless these are FILL characters
         pSpellerBtn->m_pAttachedWidget->SetEnabled(false);
      }
      else
      {
         //do nothing as the function buttons should not be controlled by Speller
         //it is upto the project presentation logic to control the enable/disable property of the buttons
         //for instance, like disabling the delete button if related EditField text is empty
      }
      //move to the next button
      pSpellerBtn++;
   }
}


void SpellerWidget2D_Ctrl::vDisableAllCharBtnStatus(const bool bEnableStatus)
{
   if (!m_pWidget)
   {
      return;
   }
   if (bEnableStatus == true)
   {
      m_lockState = true;
      struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
      while ((pSpellerBtn) && (pSpellerBtn->m_pAttachedWidget))
      {
         pSpellerBtn->m_pAttachedWidget->SetEnabled(false);
         pSpellerBtn++;
      }
   }
   else
   {
      m_lockState = false;
      struct SpellerWidget2D::SpellerBtn* pSpellerBtn = m_pWidget->p_stGetSpellerBtn();
      while ((pSpellerBtn) && (pSpellerBtn->m_pAttachedWidget))
      {
         if (SPLRBTNTYPE_Character_Empty == pSpellerBtn->m_enType)
         {
            pSpellerBtn->m_pAttachedWidget->SetEnabled(false);
         }
         else
         {
            pSpellerBtn->m_pAttachedWidget->SetEnabled(true);
         }
         pSpellerBtn++;
      }
      if (MatchMode == m_pWidget->GetMode())
      {
         m_pWidget->vHandleEnableProperty();
      }
   }
}
