/* ***************************************************************************************
* FILE:          ScrollableTextWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScrollableTextWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "ScrollableTextWidget2D.h"
#include "Widgets/2D/WidgetGestureConfig.h"
#include "Widgets/2D/ScrollableText/generated/ScrollableTextWidget2DMessages.h"

using namespace Candera;

const Candera::TChar ScrollableTextWidget2D::c_Text_Command_Mark = '\\';
const Candera::TChar ScrollableTextWidget2D::c_Text_Color_Mark = 'c';
const Candera::TChar ScrollableTextWidget2D::c_Text_Tab_Mark = 't';


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ScrollableTextWidget2D.cpp.trc.h"
#endif

#define MAX_STEP_COUNT 4
#define DEFAULT_SCROLLABLE_TEXT_ID 0
#define SWIPE_UP_ONE_LINE -1
#define SWIPE_DOWN_ONE_LINE 1

CGI_WIDGET_RTTI_DEFINITION(ScrollableTextWidget2D);


/****************************************************************************
*    Function    : ScrollableTextWidget2D
*    Description : Constructor
*    Parameters  : void
****************************************************************************/
ScrollableTextWidget2D::ScrollableTextWidget2D() :
   _listscroll(Candera::Vertical),
   m_pParsedTextChunks(0),
   m_u32ScrollPosition(0),
   m_bIsReparse(false),
   _startIndex(0),
   _layoutcount(0),
   _noOfLayout(0),
   _maxBlockCharCount(0),
   _maxLineCharCount(0),
   _firstPageRendrdCharcount(0),
   _virtualMaxScrollPosition(0.0F),
   _PreviousvirtualMaxScrollPosition(0.0F),
   _contentSize(0),
   _numOfLinesPerTextArea(0),
   _defaultFontLineHeight(0),
   _pageupCount(0),
   _lineupCount(0),
   _currentMaxIndex(0),
   _PreviousMaxIndex(0),
   _lastdragMsgVal(0),
   _dragLastPage(true),
   _listener(0),
   _isLayoutNeeded(false),
   _numLinesToSwipe(0),
   _scrollbarVisibility(false),
   IsChangedScrollPosition(false),
   u16ListChangeType(USHRT_MAX)
{
   // Set non default initial values for properties
   SetLinespacingFactor(1.0F);
   // register widget in helper classes
   m_oTabStopTableListEvent.m_pWidget = this;
   m_oColorTableListEvent.m_pWidget = this;
   SetDrag(true);
   SetSwipe(true);
}


/****************************************************************************
*    Function    : ~ScrollableTextWidget2D
*    Description : Destructor
*    Parameters  : void
****************************************************************************/
ScrollableTextWidget2D::~ScrollableTextWidget2D()
{
   // Note: current chunks will be destroyed by m_oRichText destructor
   m_pParsedTextChunks = 0;
   oInputText = "";
   _listener = 0;
}


/****************************************************************************
*    Function    : Init
*    Description : Called when widget is initialized. Overridden from
*                  WidgetBase.
*    Parameters  : Candera::AssetProvider*
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::InitWidget()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%s]. InitWidget called.", GetName()));
   Base::InitWidget();
}


/****************************************************************************
*    Function    : Update
*    Description : Called on for each render cycle. Overridden from
*                  WidgetBase.
*    Parameters  : void
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::Update()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%s]. Update called.", GetName()));
   Base::Update();
   UpdateSwipe();
   vUpdateScrollPosition();
}


/****************************************************************************
*    Function    : UpdateSwipe
*    Description : Called on for each render cycle for handling swipe
*    Parameters  : void
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::UpdateSwipe()
{
   Candera::Int32 numLinesScroll = _numLinesToSwipe;
   //check whether number of lines to be moved is not equal to zero i.e., swipe action is performed by user or not
   if (numLinesScroll != 0)
   {
      //checks whether configured text area height is > 0 and RichText's logical text area is not equal to 0
      if ((GetTextAreaSize().GetY() > 0.0F) && (m_pTextArea != 0))
      {
         if (numLinesScroll < 0)
         {
            if (_isLayoutNeeded && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_dragLastPage == true) && \
                  (_currentMaxIndex < UInt32(-_virtualMaxScrollPosition)))
            {
               vCopyBlockOfTextforLinedown(_maxLineCharCount);
               _contentSize += UInt32(m_pTextArea->cooGetContentSize().GetHeight());
               u16ListChangeType = ListChangeSwipe;
            }
            else if ((!_isLayoutNeeded) && (!GetStyle().PointsToNull()) && (static_cast<Float>(m_pTextArea->cooGetContentSize().GetHeight()) > GetTextAreaSize().GetY()))
            {
               vSetSingleLayoutRenderOffsetValue(ListChangeDown, 1);
            }
            m_bIsRerender = true;
            ScrollbarNotifyOnContentChanged();
            _numLinesToSwipe = _numLinesToSwipe + 1;
         }
         else  if (numLinesScroll > 0)
         {
            if ((_isLayoutNeeded) && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_startIndex > 0))
            {
               _dragLastPage = true;
               _lineupCount++;
               vCopyBlockOfTextForLineUp();
               _contentSize -= UInt32(m_pTextArea->cooGetContentSize().GetHeight());
               u16ListChangeType = ListChangeUp;
            }
            else if ((!_isLayoutNeeded) && (!GetStyle().PointsToNull()) && (static_cast<Float>(m_pTextArea->cooGetContentSize().GetHeight()) > GetTextAreaSize().GetY()))
            {
               vSetSingleLayoutRenderOffsetValue(ListChangeUp, 1);
            }
            m_bIsRerender = true;
            ScrollbarNotifyOnContentChanged();
            _numLinesToSwipe = _numLinesToSwipe - 1;
         }
      }
   }
}


WidgetGestureConfig ScrollableTextWidget2D::getDefaultGestureConfig() const
{
   return WidgetGestureConfig::getDefaults();
}


/****************************************************************************
*    Function    : OnMessage
*    Description : Handles message processing. Overridden from
*                  Framework Widget.
*    Parameters  : const Courier::Message&
*    Return      : bool
****************************************************************************/
bool ScrollableTextWidget2D::OnMessage(const Courier::Message& msg)
{
   bool bIsProcessed = false;
   if (Base::OnMessage(msg))
   {
      return true;
   }

   switch (msg.GetId())
   {
      case Courier::TouchMsg::ID:
      {
      }
      break;
      case ListChangeMsg::ID:
      {
         //const ScrollableChangeMsg* lMsg = Courier::message_cast<const ScrollableChangeMsg*>(&msg);
         const ListChangeMsg* lMsg = Courier::message_cast<const ListChangeMsg*>(&msg);

         if ((!GetStyle().PointsToNull()) && (GetTextAreaSize().GetY() > 0.0F) && (m_pTextArea != 0) && ((GetScrollableTextId() == DEFAULT_SCROLLABLE_TEXT_ID) || (lMsg->GetListId() == GetScrollableTextId())))
         {
            vSetRenderOffsetValue(lMsg);
            switch (lMsg->GetListChangeType())
            {
               case ListChangeUp:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with action ListChangeUp and value %d.", GetName(), lMsg->GetValue()));

                  if ((_isLayoutNeeded) && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_startIndex > 0))
                  {
                     _dragLastPage = true;
                     _lineupCount++;
                     vCopyBlockOfTextForLineUp();
                     u16ListChangeType = ListChangeUp;
                     _contentSize -= UInt32(m_pTextArea->cooGetContentSize().GetHeight());
                  }
                  bIsProcessed = true;
               }
               break;

               case ListChangeDown:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with action ListChangeDown and value %d.", GetName(), lMsg->GetValue()));

                  if (_isLayoutNeeded && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_dragLastPage == true) && \
                        (_currentMaxIndex < UInt32(-_virtualMaxScrollPosition)))
                  {
                     vCopyBlockOfTextforLinedown(_maxLineCharCount);
                     u16ListChangeType = ListChangeDown;
                     _contentSize += UInt32(m_pTextArea->cooGetContentSize().GetHeight());
                  }
                  bIsProcessed = true;
               }
               break;

               case ListChangePageUp:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with action ListChangePageUp and value %d.", GetName(), lMsg->GetValue()));

                  if ((_isLayoutNeeded) && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_startIndex > 0))
                  {
                     _dragLastPage = true;
                     _pageupCount++;
                     vCopyBlockOfTextForPageUp(_maxBlockCharCount);
                     u16ListChangeType = ListChangePageUp;
                     _contentSize -= UInt32(m_pTextArea->cooGetContentSize().GetHeight());
                  }
                  bIsProcessed = true;
               }
               break;

               case ListChangePageDown:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with action ListChangePageDown and value %d.", GetName(), lMsg->GetValue()));

                  if ((_isLayoutNeeded) && (NULL != m_pParsedTextChunks) && (_layoutcount <= _noOfLayout) && (_currentMaxIndex < (UInt32(-_virtualMaxScrollPosition))))
                  {
                     vCopyBlockOfTextforPageDown(_maxBlockCharCount);
                     u16ListChangeType = ListChangePageDown;
                     _contentSize += UInt32(m_pTextArea->cooGetContentSize().GetHeight());
                  }
                  bIsProcessed = true;
               }
               break;

               case ListChangeSet:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with action ListChangeSet and value %d.", GetName(), lMsg->GetValue()));

                  if ((_isLayoutNeeded) && (_layoutcount <= _noOfLayout))
                  {
                     if ((m_pParsedTextChunks != NULL) && (UInt32(lMsg->GetValue()) >= (UInt32(-_virtualMaxScrollPosition))) && (_dragLastPage == true))

                     {
                        _startIndex = (UInt32(-_virtualMaxScrollPosition)) - _maxLineCharCount;
                        _dragLastPage = false;
                     }

                     else if ((UInt32(lMsg->GetValue()) < (UInt32(-_virtualMaxScrollPosition))))
                     {
                        // Renders at least _maxLineCharCount in the last page
                        if ((UInt32(lMsg->GetValue())) > ((UInt32(-_virtualMaxScrollPosition)) - _maxLineCharCount))
                        {
                           _startIndex = (UInt32(-_virtualMaxScrollPosition)) - _maxLineCharCount;
                           _dragLastPage = false;
                        }
                        else
                        {
                           _dragLastPage = true;
                           _startIndex = UInt32(lMsg->GetValue());
                        }
                     }
                     _pageupCount = 0;
                     _lineupCount = 0;
                     if (_maxBlockCharCount > 0)
                     {
                        GetValidStartIndex(_startIndex);
                        vGetStartIndxWithWordWrap(_startIndex);
                        _layoutcount = (_startIndex / _maxBlockCharCount);
                        vCopyBlockOfText(_maxBlockCharCount);
                        u16ListChangeType = ListChangeSet;
                     }
                  }
                  bIsProcessed = true;
               }
               break;
               default:
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. ListChangeMsg message received with INVALID action and value %d.", GetName(), lMsg->GetValue()));
                  u16ListChangeType = USHRT_MAX;
               }
            }
            m_bIsRerender = true;
         }
      }
      break;

      case Courier::CultureChangeIndMsg::ID:
      {
         OnCultureChangeIndMsg();
      }
      break;

      default:
         break;
   }

   if (bIsProcessed)
   {
      ScrollbarNotifyOnContentChanged();
   }

   Invalidate();

   return bIsProcessed;
}


bool ScrollableTextWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool nodeTouched = false;
   if (!nodeTouched)
   {
      nodeTouched = Base::OnTouch(camera2D, point);
   }
   return nodeTouched;
}


bool ScrollableTextWidget2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool isConsumed = false;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         break;
      }
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
      {
         break;
      }
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         int numLinesScroll = 0;
         if (gestureData._isPoint1VelocityValid)
         {
            if (GetSwipeFriction() != 0)
            {
               numLinesScroll = gestureData._velocity1.y / GetSwipeFriction();
               _numLinesToSwipe = numLinesScroll;
               isConsumed = true;
            }
            else
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D: [%19s]. on swipe gesture GetSwipeFriction =%d.", GetName(), GetSwipeFriction()));
            }
         }
      }
      break;
      default:
         break;
   }
   return isConsumed;
}


bool ScrollableTextWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool isConsumed = false;
   static int startPixelPosition = 0;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         _numLinesToSwipe = 0;
         startPixelPosition = gestureData._pt1.y;
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
      {
         int currentPixelPosition = gestureData._pt1.y;

         if ((startPixelPosition - currentPixelPosition) > _defaultFontLineHeight)
         {
            if (gestureData._velocity1.y < 0)
            {
               _numLinesToSwipe = SWIPE_UP_ONE_LINE;
               startPixelPosition = currentPixelPosition;
            }
         }
         else if ((currentPixelPosition - startPixelPosition) > _defaultFontLineHeight)
         {
            if (gestureData._velocity1.y > 0)
            {
               _numLinesToSwipe = SWIPE_DOWN_ONE_LINE;
               startPixelPosition = currentPixelPosition;
            }
         }
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
      }
      break;
      default:
         break;
   }
   return isConsumed;
}


/******************************************************************************
*  CloneFrom
******************************************************************************/
bool ScrollableTextWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget,
                                       ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ScrollableTextWidget2D* original = CLONEABLE_WIDGET_CAST<const ScrollableTextWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetTextId(original->GetTextId());
      SetFileName(original->GetFileName());
      SetTextSource(original->GetTextSource());
      SetColorTable(original->GetColorTable());
      SetTabStopTable(original->GetTabStopTable());
      SetLinespacingFactor(original->GetLinespacingFactor());
      SetTextWrapMode(original->GetTextWrapMode());
      SetTextWrapMode(original->GetTextWrapMode());
      SetTabStopTable(original->GetTabStopTable());
      SetNormalTextColor(original->GetNormalTextColor());
      SetColorTable(original->GetColorTable());
      cloned = true;
   }
   return cloned;
}


/****************************************************************************
*  Called when Localization/Culture has changed during runtime
****************************************************************************/
void ScrollableTextWidget2D::OnCultureChangeIndMsg()
{
   if (0 != GetParentView())
   {
      m_bIsReparse = true;
   }
}


/****************************************************************************
*    Function    : vSetRenderOffsetValue
*    Description : set the render offset value to scroll and layout the text based
*                    the parameter if layout is needed or not
*    Parameters  : const Courier::ListChangeMsg&
*    Return      : void
****************************************************************************/

void ScrollableTextWidget2D::vSetRenderOffsetValue(const ListChangeMsg* lMsg)
{
   if ((!_isLayoutNeeded) && (!GetStyle().PointsToNull()) && (lMsg != NULL))
   {
      vSetSingleLayoutRenderOffsetValue(lMsg->GetListChangeType(), lMsg->GetValue());
   }
   else
   {
      m_oRenderOffset.SetY(0.0F);
   }
}


/****************************************************************************
*    Function    : vSetSingleLayoutRenderOffsetValue
*    Description : set the render offset value to scroll, swipe and layout the text
*    Parameters  : ListChangeType
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vSetSingleLayoutRenderOffsetValue(ListChangeType enListChangeType, Candera::Int32 i32Value)
{
   TextRendering::Metrics oFontMetrics;
   GetStyle()->GetDefaultFont().GetMetrics(oFontMetrics);
   switch (enListChangeType)
   {
      case ListChangeUp:
      {
         m_oRenderOffset += Candera::Vector2(0.0F, oFontMetrics.lineHeight * GetLinespacingFactor() * Float(i32Value));
      }
      break;
      case ListChangeDown:
      {
         m_oRenderOffset -= Candera::Vector2(0.0F, oFontMetrics.lineHeight * GetLinespacingFactor() * Float(i32Value));
      }
      break;
      case ListChangePageUp:
      {
         if (m_oRenderOffset.GetY() != _virtualMaxScrollPosition)
         {
            m_oRenderOffset += Candera::Vector2(0.0F, static_cast<Candera::Float>(oFontMetrics.lineHeight * _numOfLinesPerTextArea) * Float(i32Value));
         }
         else //added to handle the page-up operation for the last page
         {
            if (m_pTextArea->uGetLineCount() % _numOfLinesPerTextArea == 0)
            {
               m_oRenderOffset += Candera::Vector2(0.0F, static_cast<Candera::Float>(oFontMetrics.lineHeight * _numOfLinesPerTextArea) * Float(i32Value));
            }
            else
            {
               Candera::SizeType offsetDiff = m_pTextArea->uGetLineCount() % _numOfLinesPerTextArea;
               m_oRenderOffset += Candera::Vector2(0.0F, static_cast<Candera::Float>(offsetDiff * oFontMetrics.lineHeight) * Float(i32Value));
            }
         }
      }
      break;
      case ListChangePageDown:
      {
         m_oRenderOffset -= Candera::Vector2(0.0F, static_cast<Candera::Float>(_numOfLinesPerTextArea *  oFontMetrics.lineHeight) * Float(i32Value));
      }
      break;
      case ListChangeSet:
      {
         m_oRenderOffset = Candera::Vector2(0.0F, -Float(i32Value));
      }
      break;
      default:
         break;
   }
   _virtualMaxScrollPosition = (static_cast<Candera::Float>((m_pTextArea->uGetLineCount() - _numOfLinesPerTextArea) * oFontMetrics.lineHeight) * GetLinespacingFactor())  * -1.0F;
   if (m_oRenderOffset.GetY() < _virtualMaxScrollPosition)
   {
      m_oRenderOffset.SetY(_virtualMaxScrollPosition);
   }
   if (m_oRenderOffset.GetY() >= 0.0F)
   {
      m_oRenderOffset.SetY(0.0F);
   }
   _currentMaxIndex = static_cast<UInt32>(-m_oRenderOffset.GetY());
}


/****************************************************************************
*    Function    : IsScrollBarVisible
*    Description : Called by Scrollbar to retrieve data.
*    Parameters  : void
*    Return      : bool
****************************************************************************/
bool ScrollableTextWidget2D::IsScrollBarVisible()
{
   return _scrollbarVisibility;
}


/****************************************************************************
*    Function    : SetListener
*    Description : overwritten FlexScrollable interface
*    Parameters  : FlexScrollableListener* listener
****************************************************************************/
void ScrollableTextWidget2D::SetListener(FlexScrollableListener* listener) // FlexScrollable interface
{
   _listener = listener;
}


void ScrollableTextWidget2D::ScrollbarNotifyOnPositionReached()
{
   if (0 != _listener)
   {
      _listener->OnPositionReached();
   }
}


void ScrollableTextWidget2D::ScrollbarNotifyOnContentChanged()
{
   if (0 != _listener)
   {
      _listener->OnContentUpdated();
   }
}


/****************************************************************************
*    Function    : GetPosition
*    Description : Called by Scrollbar to retrieve data.
*    Parameters  : void
*    Return      : Candera::UInt32
****************************************************************************/
Candera::UInt32 ScrollableTextWidget2D::GetPosition()
{
   //printf("Scroll Bar Pos %u\n", _currentMaxIndex);

   return _currentMaxIndex;
}


Candera::UInt32 ScrollableTextWidget2D::GetFirstFullyVisiblePosition()
{
   //printf("Scroll Bar Pos %u\n", _currentMaxIndex);

   return _currentMaxIndex;
}


/****************************************************************************
*    Function    : GetVisibleSize
*    Description : Called by Scrollbar to retrieve data.
*    Parameters  : void
*    Return      : Candera::UInt32
****************************************************************************/
Candera::UInt32 ScrollableTextWidget2D::GetVisibleSize()
{
   if (GetTextAreaSize().GetY() > 0.0F)
   {
      return UInt32(GetTextAreaSize().GetY() + 0.5F);
   }
   else
   {
      if ((m_pTextArea != 0) && (_contentSize > 0))
      {
         return (_contentSize);
      }

      return 0;
   }
}


/****************************************************************************
*    Function    : GetCompleteSize
*    Description : Called by Scrollbar to retrieve data.
*    Parameters  : void
*    Return      : Candera::UInt32
****************************************************************************/
Candera::UInt32 ScrollableTextWidget2D::GetCompleteSize()
{
   if ((m_pTextArea != 0) && (_contentSize > 0))
   {
      return (_contentSize);
   }

   return 0;
}


/****************************************************************************
*    Function    : Widget2D
*    Description : Called by Scrollbar to retrieve data.
*    Parameters  : void
*    Return      : Candera::Widget2D*
****************************************************************************/
Candera::Widget2D* ScrollableTextWidget2D::GetWidget()
{
   return this;
}


Candera::UInt32 ScrollableTextWidget2D::GetMaxPosition()
{
   return (UInt32(-_virtualMaxScrollPosition));
}


bool ScrollableTextWidget2D::IsCircular()
{
   return false;
}


/****************************************************************************
* overwritten interface FlexScrollable for immediate positioning.
* Immediate positioning means that no animation should be used to scroll to
* the requested position
****************************************************************************/
void ScrollableTextWidget2D::RequestImmediatePositioning(bool /*active*/)
{
}


/****************************************************************************
*     Function    : OnChanged
*     Description : Called from the Update function for each property that
*                   has been changed. Overridden from RichTextWidgetBase.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ScrollableTextWidget2D::OnChanged(Candera::UInt32 propertyId)
{
   switch (propertyId)
   {
      case TextPropertyId:
         if (GetTextSource() == SourceText)
         {
            m_bIsReparse = true;
         }

         break;

      case TextIdPropertyId:
         if (GetTextSource() == SourceTextId)
         {
            m_bIsReparse = true;
         }

         break;

      case FileNamePropertyId:
         if (GetTextSource() == SourceFile)
         {
            m_bIsReparse = true;
         }

         break;

      case TextSourcePropertyId:
         m_bIsReparse = true;
         break;

      case NormalTextColorPropertyId :
         m_bIsReparse = true;
         break;

      case LinespacingFactorPropertyId:
         m_bIsRelayout = true;
         break;

      case TabStopTablePropertyId:
      {
         SizeType u32TabStopTableSize = GetTabStopTable().Count();

         if (u32TabStopTableSize != Courier::ListInterfaceBase::cUnknownQuantity)
         {
            m_oTabStopCache.resize(u32TabStopTableSize);
            GetTabStopTable().SetEventCallback(&m_oTabStopTableListEvent, &TabStopTableListEvent::OnTabStopTableListEvent);
            GetTabStopTable().Request(0, u32TabStopTableSize);
         }
         else
         {
            m_oTabStopCache.resize(0);
         }

         m_bIsRelayout = true;
      }
      break;

      case ColorTablePropertyId:
      {
         SizeType u32ColorTableSize = GetColorTable().Count();

         if (u32ColorTableSize != Courier::ListInterfaceBase::cUnknownQuantity)
         {
            m_oColorCache.resize(u32ColorTableSize);
            GetColorTable().SetEventCallback(&m_oColorTableListEvent, &ColorTableListEvent::OnColorTableListEvent);
            GetColorTable().Request(0, u32ColorTableSize);
         }
         else
         {
            m_oColorCache.resize(0);
         }

         // Optimization: could be done with only a redrawing if a list for all color<->colorchunks is maintained
         m_bIsReparse = true;
      }
      break;

      case TextWrapModePropertyId:
         m_bIsRelayout = true;
         break;

      default:
         Base::OnChanged(propertyId);
         break;
   }
   ScrollbarNotifyOnContentChanged();
   Invalidate();
}


/****************************************************************************
*     Function    : vUpdateNode
*     Description : Update all node related properties here
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vUpdateNode()
{
   Base::vUpdateNode();
}


/****************************************************************************
*     Function    : bIsReLayoutNeeded
*     Description : Checks if relayouting is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool ScrollableTextWidget2D::bIsReLayoutNeeded()
{
   return Base::bIsReLayoutNeeded() || m_bIsReparse;
}


/****************************************************************************
*     Function    : bIsReRenderNeeded
*     Description : Checks if rerendering is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool ScrollableTextWidget2D::bIsReRenderNeeded()
{
   return Base::bIsReRenderNeeded();
}


/****************************************************************************
*     Function    : vLayoutRichText
*     Description : Will do the layouting of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after layouting.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vLayoutRichText()
{
#ifdef RICHTEXT_PERFORMANCE_CHECK
   _tPerf.start();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Time elapse diff (1) = %d", _tPerf.diffMsec()));
#endif

   // reset scrolling position
   m_oRenderOffset = Candera::Vector2(0.0F, 0.0F);
   //reset first page rendered character count to 0
   _firstPageRendrdCharcount = 0;

   // delete old chunks
   m_oRichText.vRemoveChunk(m_pParsedTextChunks, true);
   FEATSTD_DELETE(m_pParsedTextChunks);
   m_pParsedTextChunks = 0;
   // reparse text if needed
   if (m_bIsReparse)
   {
      switch (GetTextSource())
      {
         case SourceText:
            oInputText = GetText();
            break;

         case SourceTextId:
            oInputText = GetTextId();
            break;

         case SourceFile:
         {
            FILE* pFile;
            long int FileSize = 0;
            // open an existing file for reading
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetFileName());
            pFile = fopen(GetFileName().GetCString(), "rb");
            SECURE_FEATSTD_STRING_ACCESS_END();

            if (pFile != 0)
            {
               // Get the number of bytes
               fseek(pFile, 0L, SEEK_END);
               FileSize = ftell(pFile);
               // Reset the file position indicator to the beginning of the file
               fseek(pFile, 0L, SEEK_SET);

               // read data
               if (FileSize > 0)
               {
                  UInt8* pBuffer = FEATSTD_NEW_ARRAY(UInt8, FileSize + 1);

                  if (pBuffer != 0)
                  {
                     fread(pBuffer, sizeof(UInt8), SizeType(FileSize), pFile);

                     pBuffer[FileSize] = 0;
                     oInputText = (Char*)(pBuffer);
                     // printf("FileCharCount %u \n", oInputText.GetCharCount());
                     SetText(oInputText); // For Trace cmd impl to get the text in the active scene
                     FEATSTD_DELETE(pBuffer);
                  }
               }
               // Close File
               fclose(pFile);
            }
         }
         break;

         default:
            break; // do nothing
      }

      vApplyLayoutForRichText();
   }
}


/****************************************************************************
*     Function    : vApplyLayoutForRichText()
*     Description : check if the provided text content fits in a  single or multiple window
*					and does layout for the text content
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vApplyLayoutForRichText()
{
   TextAreaSizeType oTextAreaSize = GetTextAreaSize();
   oOrginalString = oInputText;
   int size = Int32(oTextAreaSize.GetX());
   int sizeY = Int32(oTextAreaSize.GetY());

   if ((!GetStyle().PointsToNull()) && (size > 0))
   {
      TextRendering::Metrics oFontMetrics;
      GetStyle()->GetDefaultFont().GetMetrics(oFontMetrics);
      if (oFontMetrics.lineHeight > 0)
      {
         _defaultFontLineHeight = oFontMetrics.lineHeight;
         // Determine maximum number of characters per line for the given text area size
         _maxLineCharCount = (size / 8); // at least 8 pixels i.e., 1 byte will be occupied by a character
         _numOfLinesPerTextArea = static_cast<UInt16>(sizeY / oFontMetrics.lineHeight);
         // Number of characters that are passed in a single layout/window
         _maxBlockCharCount = _maxLineCharCount * _numOfLinesPerTextArea;
         // Number of layouts for the provided input content
         _noOfLayout = static_cast<UInt32>((oOrginalString.GetCharCount() / _maxBlockCharCount));
         if (_noOfLayout == 0)  // prevent div by zero
         {
            _noOfLayout = 1;
         }
      }
   }
   _virtualMaxScrollPosition = -Float(oOrginalString.GetCharCount()); //filesize
   _startIndex = 0; //reset values for next text update
   _currentMaxIndex = 0;
   _dragLastPage = true;
   if (oOrginalString.GetCharCount() > _maxBlockCharCount)
   {
      //windowing size is layouted instead of all contents.
      _isLayoutNeeded = true;
      if (_layoutcount < _noOfLayout)
      {
         vCopyBlockOfText(_maxBlockCharCount);
      }
   }
   else
   {
      //layout all the content in one shot
      _isLayoutNeeded = false;
      vSendRichTextForLayouting();
   }
}


void ScrollableTextWidget2D::vCopyBlockOfText(Candera::UInt32 BlockCharCount)
{
   FEATSTD_UNUSED(BlockCharCount);
   bool IsMaxBockCharsCopied = isMaxBlockCharsCopied(_startIndex);
   if ((IsMaxBockCharsCopied) && (_layoutcount < _noOfLayout))
   {
      _layoutcount++;
   }
   vSendRichTextForLayouting();

   if (m_pTextArea != 0)
   {
      _contentSize = UInt32(m_pTextArea->cooGetContentSize().GetHeight());
   }

   _currentMaxIndex =  _startIndex;
}


void ScrollableTextWidget2D::vCopyBlockOfTextForPageUp(Candera::UInt32 BlockCharCount)
{
   bool IsBlockCharsCopied = false;
   bool isPageDown = true;
   if ((_layoutcount <= _noOfLayout) && (NULL != m_pParsedTextChunks) && (_startIndex >= BlockCharCount))
   {
      _startIndex = (_pageupCount == 1) ? _startIndex : (_startIndex - m_pParsedTextChunks->u32GetRenderdCharCount());

      GetValidStartIndex(_startIndex);
      if (_startIndex >= BlockCharCount)
      {
         Candera::UInt32 sourceStartIndex = _startIndex - BlockCharCount;
         GetValidStartIndex(sourceStartIndex);
         IsBlockCharsCopied = isMaxBlockCharsCopied(sourceStartIndex);
         isPageDown = false;
         m_pParsedTextChunks->vSetNumOfLinesPerTextArea(_numOfLinesPerTextArea);
      }
      else
      {
         if (_startIndex  > _firstPageRendrdCharcount)
         {
            IsBlockCharsCopied = isProvidedBlockCharsCopied(0, _startIndex);
            isPageDown = false;
            m_pParsedTextChunks->vSetNumOfLinesPerTextArea(_numOfLinesPerTextArea);
         }
         //to layout the first page,where the remaining content to display is small.
         else
         {
            IsBlockCharsCopied = isProvidedBlockCharsCopied(0, _firstPageRendrdCharcount);
            _startIndex = 0;
            isPageDown = true;
            _pageupCount = 0;
         }
      }
   }
   else if ((_layoutcount <= _noOfLayout) && (NULL != m_pParsedTextChunks) && (_startIndex < BlockCharCount))
   {
      _startIndex = (_pageupCount == 1) ? _startIndex : (_startIndex - (m_pParsedTextChunks->u32GetRenderdCharCount()));

      GetValidStartIndex(_startIndex);
      if (_startIndex  > _firstPageRendrdCharcount)
      {
         IsBlockCharsCopied = isProvidedBlockCharsCopied(0, _startIndex);
         isPageDown = false;
         m_pParsedTextChunks->vSetNumOfLinesPerTextArea(_numOfLinesPerTextArea);
      }
      //to layout the first page,where the remaining content to display is small.
      else
      {
         IsBlockCharsCopied = isProvidedBlockCharsCopied(0, _firstPageRendrdCharcount);
         _startIndex = 0;
         isPageDown = true;
         _pageupCount = 0;
      }
   }
   _lineupCount = 0;
   if (IsBlockCharsCopied && (_startIndex < (_layoutcount * _maxBlockCharCount)))
   {
      _layoutcount--;
   }

   vSendRichTextForLayouting(isPageDown);
   if (NULL != m_pParsedTextChunks)
   {
      _currentMaxIndex = _startIndex;
   }
}


void ScrollableTextWidget2D::vCopyBlockOfTextForLineUp()
{
   bool isLineUp = true;
   bool IsMaxBockCharsCopied = false;
   if ((_layoutcount <= _noOfLayout) && (NULL != m_pParsedTextChunks))
   {
      m_pParsedTextChunks->vSetNumOfLinesPerTextArea(_numOfLinesPerTextArea);
      if ((_pageupCount != 0) || (_lineupCount > 1))
      {
         _startIndex = _startIndex - m_pParsedTextChunks->u32GetRenderdCharCount();
      }

      _startIndex = (_startIndex + m_pParsedTextChunks->u32GetRenderdCharCount()) - m_pParsedTextChunks->u32GetLastLineRenderdCharCount();
      GetValidStartIndex(_startIndex);

      if ((_startIndex > _maxBlockCharCount) && (_startIndex > _firstPageRendrdCharcount))
      {
         isLineUp = true;
         Candera::UInt32 _sourceStartIndex = _startIndex - _maxBlockCharCount;
         GetValidStartIndex(_sourceStartIndex);
         IsMaxBockCharsCopied = isMaxBlockCharsCopied(_sourceStartIndex);
      }
      else
      {
         //to handle the case, where the remaining content to display is small.
         IsMaxBockCharsCopied = isProvidedBlockCharsCopied(0, _startIndex);
         if (_startIndex <= _firstPageRendrdCharcount)
         {
            _startIndex = 0;
            _lineupCount = 0;
         }
      }

      _pageupCount = 0;
      m_pParsedTextChunks->vSetIsPageOrLineDown(true);
      m_pParsedTextChunks->vSetLineUp(isLineUp);

      if (IsMaxBockCharsCopied && ((_startIndex) < (_layoutcount * _maxBlockCharCount)))
      {
         _layoutcount--;
      }

      vSendRichTextForLayouting();
      if ((NULL != m_pParsedTextChunks) && (_lineupCount != 0) && (_startIndex > m_pParsedTextChunks->u32GetRenderdCharCount()))
      {
         //added to make sure marker is at correct position
         _currentMaxIndex = _startIndex - m_pParsedTextChunks->u32GetRenderdCharCount();
      }
      else
      {
         _currentMaxIndex = _startIndex;
      }
   }
}


void ScrollableTextWidget2D::vCopyBlockOfTextforPageDown(Candera::UInt32 BlockCharCount)
{
   FEATSTD_UNUSED(BlockCharCount);
   if ((_layoutcount <= _noOfLayout) && (NULL != m_pParsedTextChunks))
   {
      _startIndex = (_pageupCount == 0) ? (_startIndex + m_pParsedTextChunks->u32GetRenderdCharCount()) : _startIndex;
      if (_startIndex == 0)
      {
         _startIndex = _startIndex + m_pParsedTextChunks->u32GetRenderdCharCount();
      }

      GetValidStartIndex(_startIndex);
      m_pParsedTextChunks->vSetIsPageOrLineDown(true);
      _pageupCount = 0;
      _lineupCount = 0;
   }
   bool IsMaxBockCharsCopied = isMaxBlockCharsCopied(_startIndex);
   if (IsMaxBockCharsCopied && (_startIndex > ((_layoutcount)*_maxBlockCharCount)) && (_layoutcount < _noOfLayout))
   {
      _layoutcount++;
   }
   vSendRichTextForLayouting();

   _currentMaxIndex = _startIndex;
}


void ScrollableTextWidget2D::vCopyBlockOfTextforLinedown(Candera::UInt32 BlockCharCount)
{
   if ((_layoutcount <= _noOfLayout) && (NULL != m_pParsedTextChunks))
   {
      _startIndex = (_pageupCount != 0 || _lineupCount != 0) ? (_startIndex - m_pParsedTextChunks->u32GetRenderdCharCount()) : _startIndex;
      _startIndex = _startIndex + m_pParsedTextChunks->u32GetFirstLineRenderdCharCount();
   }
   else if ((NULL != m_pParsedTextChunks) && (m_pParsedTextChunks->u32GetRenderdCharCount()) < (BlockCharCount))
   {
      _startIndex = _layoutcount * _maxBlockCharCount;
   }
   if (_pageupCount != 0)
   {
      _pageupCount = 0;
   }
   if (_lineupCount != 0)
   {
      _lineupCount = 0;
   }
   m_pParsedTextChunks->vSetIsPageOrLineDown(true);

   GetValidStartIndex(_startIndex);
   bool IsMaxBockCharsCopied = isMaxBlockCharsCopied(_startIndex);
   if (IsMaxBockCharsCopied && (_startIndex > (_layoutcount * _maxBlockCharCount)) && (_layoutcount < _noOfLayout))
   {
      _layoutcount++;
   }
   vSendRichTextForLayouting();

   _currentMaxIndex = _startIndex;
}


bool ScrollableTextWidget2D::isMaxBlockCharsCopied(Candera::UInt32 _sourceStartIndex)
{
   Char* l_pCharPtr;
   _maxBlockCharCount = _maxLineCharCount * _numOfLinesPerTextArea;
   GetValidMaxBlockCharCount(_sourceStartIndex, _maxBlockCharCount);
   l_pCharPtr = (Char*) MemoryPlatform::MemAlloc((_maxBlockCharCount + 1) * sizeof(Char));
   if (NULL != l_pCharPtr)
   {
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(oOrginalString);
      const Char* l_pcSource = oOrginalString.GetCString();
      StringPlatform::CopyPartial(l_pCharPtr, &l_pcSource[_sourceStartIndex], _maxBlockCharCount);
      l_pCharPtr[_maxBlockCharCount] = '\0';
      Candera::String s(l_pCharPtr);
      oInputText = s;
      MemoryPlatform::MemFree(l_pCharPtr);
      SECURE_FEATSTD_STRING_ACCESS_END();
      return true;
   }
   else
   {
      return false;
   }
}


bool ScrollableTextWidget2D::isProvidedBlockCharsCopied(Candera::UInt32 startIndex, Candera::UInt32 BlockCharCount)
{
   Char* l_pCharPtr;
   GetValidMaxBlockCharCount(startIndex, BlockCharCount);
   l_pCharPtr = (Char*) MemoryPlatform::MemAlloc((BlockCharCount + 1) * sizeof(Char));
   if (NULL != l_pCharPtr)
   {
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(oOrginalString);
      const Char* l_pcSource = oOrginalString.GetCString();
      StringPlatform::CopyPartial(l_pCharPtr, &l_pcSource[startIndex], BlockCharCount);
      l_pCharPtr[BlockCharCount] = '\0';
      Candera::String s(l_pCharPtr);
      oInputText = s;
      MemoryPlatform::MemFree(l_pCharPtr);
      SECURE_FEATSTD_STRING_ACCESS_END();
      return true;
   }
   else
   {
      return false;
   }
}


void ScrollableTextWidget2D::vSendRichTextForLayouting(bool IsPageDown)
{
   // delete old chunks
   m_oRichText.vRemoveChunk(m_pParsedTextChunks, true);
   FEATSTD_DELETE(m_pParsedTextChunks);
   m_pParsedTextChunks = 0;

   // create new chunks and add them to the rich text
   m_pParsedTextChunks = oParseText(oInputText);
   m_oRichText.vAddChunk(m_pParsedTextChunks);

   if (NULL != m_pParsedTextChunks)
   {
      m_pParsedTextChunks->vSetIsPageOrLineDown(IsPageDown);
   }

   m_bIsReparse = false;

   if (m_pDefaultParagraphChunk != 0)
   {
      // set tab stop list
      m_pDefaultParagraphChunk->vSetTabStopList(m_oTabStopCache);
      // set line spacing
      m_pDefaultParagraphChunk->vSetLineSpacingFactor(GetLinespacingFactor());
      // set text wrapping
      m_pDefaultParagraphChunk->vSetTextWrapMode(GetTextWrapMode());
      // set paragraph size. Only width is required. Height must stay at 0
      TextAreaSizeType oTextAreaSize = GetTextAreaSize();
      m_pDefaultParagraphChunk->vSetSize(RichText::Size(Int32(oTextAreaSize.GetX()), 0));
   }

   // perform layout
   Base::vLayoutRichText();
   ScrollbarNotifyOnContentChanged();
   /// ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "iiiWrapText=%ld ;  iiiWrapText, iiiMeasureText=%ld", iiiWrapText, iiiMeasureText));
}


/****************************************************************************
*     Function    : vRenderRichText
*     Description : Will do the rendering of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after rendering.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vRenderRichText()
{
   //printf("Rendering with offset %f\n", m_oRenderOffset.GetY());
   // perform rendering
   Base::vRenderRichText();
   if (NULL != m_pParsedTextChunks)
   {
      _scrollbarVisibility = (oOrginalString.GetCharCount() > (m_pParsedTextChunks->u32GetRenderdCharCount())) ? true : false;
      if (_scrollbarVisibility == false)
      {
         _currentMaxIndex = UInt32(-_virtualMaxScrollPosition);
      }
      //Sets _currentMaxIndex to _virtualMaxScrollPosition value after rendering the last page . Lage page rendering can happen via PageDown/LineDown/slider/swipe action
      if ((_isLayoutNeeded) && ((u16ListChangeType == ListChangeDown) || (u16ListChangeType == ListChangePageDown) || (u16ListChangeType == ListChangeSet) || (u16ListChangeType == ListChangeSwipe)) && (static_cast<Float>(_startIndex + m_pParsedTextChunks->u32GetRenderdCharCount()) >= -_virtualMaxScrollPosition))
      {
         _currentMaxIndex = UInt32(-_virtualMaxScrollPosition);
      }
      // set the rendered character count of first page
      if ((_isLayoutNeeded) && (_firstPageRendrdCharcount == 0))
      {
         _firstPageRendrdCharcount = _startIndex + m_pParsedTextChunks->u32GetRenderdCharCount();
      }
      ScrollbarNotifyOnContentChanged();
   }
}


/****************************************************************************
*    Function    : ScrollableTextWidget2D::TabStopTableListEvent::OnTabStopTableListEvent
*    Description : This method is called on TabStopTable changes.
*    Parameters  : Courier::ListEvent
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::TabStopTableListEvent::OnTabStopTableListEvent(const Courier::ListEvent& listEvent)
{
   if (m_pWidget != 0)
   {
      if (listEvent.EventType() == Courier::ListEventType::RefreshList)
      {
         m_pWidget->m_oTabStopCache.resize(m_pWidget->GetTabStopTable().Count());
         m_pWidget->GetTabStopTable().Request(0, m_pWidget->GetTabStopTable().Count());
      }
      else if (listEvent.EventType() == Courier::ListEventType::RequestedItem)
      {
         const SizeType listIndex = listEvent.NewIndex();
         const Candera::Int16* value = listEvent.GetItem<Candera::Int16>();

         if ((value != 0) && (listIndex < m_pWidget->m_oTabStopCache.size()))
         {
            m_pWidget->m_oTabStopCache[listIndex] = *value;
         }
      }
   }
}


/****************************************************************************
*    Function    : ScrollableTextWidget2D::ColorTableListEvent::OnColorTableListEvent
*    Description : This method is called on ColorTable changes.
*    Parameters  : Courier::ListEvent
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::ColorTableListEvent::OnColorTableListEvent(const Courier::ListEvent& listEvent)
{
   if (m_pWidget != 0)
   {
      if (listEvent.EventType() == Courier::ListEventType::RefreshList)
      {
         m_pWidget->m_oColorCache.resize(m_pWidget->GetColorTable().Count());
         m_pWidget->GetColorTable().Request(0, m_pWidget->GetColorTable().Count());
      }
      else if (listEvent.EventType() == Courier::ListEventType::RequestedItem)
      {
         const SizeType listIndex = listEvent.NewIndex();
         const Candera::Color* value = listEvent.GetItem<Candera::Color>();

         if ((value != 0) && (listIndex < m_pWidget->m_oColorCache.size()))
         {
            m_pWidget->m_oColorCache[listIndex] = *value;
         }
      }
   }
}


/****************************************************************************
*    Function    : ScrollableTextWidget2D::oParseText
*    Description : Generates a chunk list based on the supplied input string.
*    Parameters  : const FeatStd::String&
*    Return      : RichText::RichTextChunk*
****************************************************************************/

// types only used in pParseText method
// Note: defined here because local types aren't allowed in templates in c++03 and older standards

enum InsertType
{
   InsertNone,
   InsertTab,
   InsertColor,
   InsertLineBreak
};


struct InsertInfo
{
   FEATSTD_LINT_CURRENT_SCOPE(1712, "No default constructor needed.")

   UInt32 m_u32BytePosition;
   UInt32 m_u32CodePointPosition;
   UInt32 m_u32SkipCount;
   InsertType m_enType;
   UInt32 m_u32Index;

   InsertInfo(UInt32 u32BytePosition, UInt32 u32CodePointPosition, UInt32 u32SkipCount, InsertType enType, UInt32 u32Index) :
      m_u32BytePosition(u32BytePosition), m_u32CodePointPosition(u32CodePointPosition), m_u32SkipCount(u32SkipCount), m_enType(enType), m_u32Index(u32Index)
   {
   }
};


RichText::RichTextChunk* ScrollableTextWidget2D::oParseText(const FeatStd::String& oText)
{
   std::vector<InsertInfo> oInsertInfoTable;
   const TChar* pchOriginalString;
   UInt32 u32CurrentBytePosition = 0;
   UInt32 u32CurrentCodePointPosition = 0;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(oText);
   pchOriginalString = oText.GetCString();
   // loop through string to find all insert positions
   while (pchOriginalString[0] != TChar(0))
   {
      // check for commands
      if (pchOriginalString[0] == TChar(c_Text_Command_Mark))
      {
         // check if the following value matches a color or tab mark
         if ((pchOriginalString[1] == c_Text_Color_Mark) || (pchOriginalString[1] == c_Text_Tab_Mark))
         {
            // read the index value of the command and insert the according command into the table
            UInt32 u32SkipCount = 2;
            UInt32 u32Index = 0;

            while ((pchOriginalString[u32SkipCount] >= TChar('0')) && (pchOriginalString[u32SkipCount] <= TChar('9')))
            {
               u32Index *= 10;
               u32Index += (pchOriginalString[u32SkipCount] - TChar('0'));
               u32SkipCount++;
            }

            if (pchOriginalString[1] == c_Text_Color_Mark)
            {
               oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, u32SkipCount, InsertColor, u32Index));
            }
            else
            {
               oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, u32SkipCount, InsertTab, u32Index));
            }

            for (UInt32 u32Index = 0; u32Index < u32SkipCount - 1; u32Index++)
            {
               u32CurrentBytePosition += RichText::Utf8Encoding::u32Advance(pchOriginalString);
               u32CurrentCodePointPosition++;
            }
         }
         else
         {
            // check for line break provided as part of input
            if (pchOriginalString[1] == TChar('n'))
            {
               oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, 2, InsertLineBreak, 0));
               u32CurrentBytePosition += RichText::Utf8Encoding::u32Advance(pchOriginalString);
               u32CurrentCodePointPosition++;
            }
            // check for '\\' commands
            if (pchOriginalString[1] == TChar(c_Text_Command_Mark))
            {
               oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, 1, InsertNone, 0));
               u32CurrentBytePosition += RichText::Utf8Encoding::u32Advance(pchOriginalString);
               u32CurrentCodePointPosition++;
            }
         }
      }
      // check line break
      else if (pchOriginalString[0] == TChar('\r'))
      {
         if (pchOriginalString[1] == TChar('\n'))
         {
            oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, 2, InsertLineBreak, 0));
            u32CurrentBytePosition += RichText::Utf8Encoding::u32Advance(pchOriginalString);
            u32CurrentCodePointPosition++;
         }
      }
      else if (pchOriginalString[0] == TChar('\n'))
      {
         oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, 1, InsertLineBreak, 0));
      }
      //check for '\t' break
      else
      {
         if (pchOriginalString[0] == TChar('\t'))
         {
            oInsertInfoTable.push_back(InsertInfo(u32CurrentBytePosition, u32CurrentCodePointPosition, 1, InsertTab, 1));
         }
      }
      // go to next codepoint
      u32CurrentBytePosition += RichText::Utf8Encoding::u32Advance(pchOriginalString);
      u32CurrentCodePointPosition++;
   }
   SECURE_FEATSTD_STRING_ACCESS_END();

   u32CurrentBytePosition = 0;
   u32CurrentCodePointPosition = 0;
   // create default color chunk
   RichText::RichTextChunk* pCurrentChunk = FEATSTD_NEW(RichText::ColorChunk)(GetNormalTextColor(), 0);
   RichText::RichTextChunk* pFirstChunk = pCurrentChunk;
   // create text and command chunks
   const SizeType uInsertPositionCount = oInsertInfoTable.size();

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(oText);
   pchOriginalString = oText.GetCString();

   for (SizeType uIndex = 0; uIndex < uInsertPositionCount; uIndex++)
   {
      // create text chunk before command
      UInt32 u32TextSize = oInsertInfoTable[uIndex].m_u32CodePointPosition - u32CurrentCodePointPosition;

      if (u32TextSize > 0)
      {
         FeatStd::String oChunkText = FeatStd::String(&pchOriginalString[u32CurrentBytePosition], u32TextSize);
         RichText::TextChunk* pTextChunk = FEATSTD_NEW(RichText::TextChunk)(oChunkText);

         if (pTextChunk != 0)
         {
            if (pCurrentChunk != 0)
            {
               pCurrentChunk->vSetNextChunk(pTextChunk);
               pCurrentChunk = pTextChunk;
            }
            else
            {
               pFirstChunk = pTextChunk;
               pCurrentChunk = pTextChunk;
            }
         }
      }

      // create command chunk
      RichText::RichTextChunk* pCommandChunk = 0;

      switch (oInsertInfoTable[uIndex].m_enType)
      {
         case InsertTab:
         {
            if (oInsertInfoTable[uIndex].m_u32Index > 0)
            {
               pCommandChunk = FEATSTD_NEW(RichText::TabChunk)(((oInsertInfoTable[uIndex].m_u32Index) - 1), (oInsertInfoTable[uIndex].m_u32SkipCount));
            }
         }
         break;

         case InsertColor:
         {
            if (oInsertInfoTable[uIndex].m_u32Index == 0)
            {
               pCommandChunk = FEATSTD_NEW(RichText::ColorChunk)(GetNormalTextColor(), oInsertInfoTable[uIndex].m_u32SkipCount);
            }
            else
            {
               if (oInsertInfoTable[uIndex].m_u32Index <= m_oColorCache.size())
               {
                  pCommandChunk = FEATSTD_NEW(RichText::ColorChunk)(m_oColorCache[oInsertInfoTable[uIndex].m_u32Index - 1], oInsertInfoTable[uIndex].m_u32SkipCount);
               }
            }
         }
         break;

         case InsertLineBreak:
            pCommandChunk = FEATSTD_NEW(RichText::NewLineChunk)(oInsertInfoTable[uIndex].m_u32SkipCount);
            break;

         case InsertNone:
            break;

         default:
            break;
      }

      if (pCommandChunk != 0)
      {
         if ((pCurrentChunk != 0))
         {
            pCurrentChunk->vSetNextChunk(pCommandChunk);
            pCurrentChunk = pCommandChunk;
         }
         else
         {
            pFirstChunk = pCommandChunk;
            pCurrentChunk = pCommandChunk;
         }
      }

      // jump to end of insert position
      u32CurrentBytePosition = oInsertInfoTable[uIndex].m_u32BytePosition + oInsertInfoTable[uIndex].m_u32SkipCount;
      u32CurrentCodePointPosition = oInsertInfoTable[uIndex].m_u32CodePointPosition + oInsertInfoTable[uIndex].m_u32SkipCount;
   }

   // create last text chunk
   if (pchOriginalString[u32CurrentBytePosition] != '\0')
   {
      FeatStd::String oChunkText = FeatStd::String(&pchOriginalString[u32CurrentBytePosition]);
      RichText::TextChunk* pTextChunk = FEATSTD_NEW(RichText::TextChunk)(oChunkText);

      // add to chunk list
      if (pCurrentChunk != 0)
      {
         pCurrentChunk->vSetNextChunk(pTextChunk);
      }
      else
      {
         pFirstChunk = pTextChunk;
      }

      FEATSTD_LINT_NEXT_EXPRESSION(593, "pTextChunk is either pFirstChunk or a following chunk.")
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   return pFirstChunk;
}


/****************************************************************************
*    Function    : ScrollableTextWidget2D::vGetStartIndxWithWordWrap
*    Description : Update the start index  when start index is not the exactly
*                 start of the word when layouting concept is used.
*    Parameters  : Candera::UInt32
*    Return      : void
****************************************************************************/
void ScrollableTextWidget2D::vGetStartIndxWithWordWrap(Candera::UInt32 _startIndx)
{
   const TChar* l_pCharPtr;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(oOrginalString);
   l_pCharPtr = oOrginalString.GetCString();

   if (NULL != l_pCharPtr)
   {
      /*if start index points to middle char of the word or numbers, then navigate backwards until we get start of the word*/
      while ((_startIndx > 0) && (((l_pCharPtr[_startIndx - 1] >= TChar('a')) && (l_pCharPtr[_startIndx - 1] <= TChar('z'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar('A')) && (l_pCharPtr[_startIndx - 1] <= TChar('Z'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar('0')) && (l_pCharPtr[_startIndx - 1] <= TChar('9'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar('!')) && (l_pCharPtr[_startIndx - 1] <= TChar('.'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar(':')) && (l_pCharPtr[_startIndx - 1] <= TChar('@'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar('[')) && (l_pCharPtr[_startIndx - 1] <= TChar('`'))) || \
                                  ((l_pCharPtr[_startIndx - 1] >= TChar('{')) && (l_pCharPtr[_startIndx - 1] <= TChar('~')))))
      {
         _startIndx--;
      }
      _startIndex = _startIndx;

      SECURE_FEATSTD_STRING_ACCESS_END();
   }
}


void ScrollableTextWidget2D::vUpdateScrollPosition()
{
   if (_PreviousvirtualMaxScrollPosition != _virtualMaxScrollPosition)
   {
      _PreviousvirtualMaxScrollPosition = _virtualMaxScrollPosition;
      IsChangedScrollPosition = true;
   }
   if (_PreviousMaxIndex != _currentMaxIndex)
   {
      _PreviousMaxIndex = _currentMaxIndex;
      IsChangedScrollPosition = true;
   }
   if ((IsChangedScrollPosition == true) && (GetPostUpdateMessage() == true))
   {
      IsChangedScrollPosition = false;
      vPostUpdateMessage();
   }
}


void ScrollableTextWidget2D::vPostUpdateMessage()
{
   Courier::View* parentView = GetParentView();
   if (parentView != NULL)
   {
      UInt32 u32MaxScrollPosition = UInt32(-1.0F * _virtualMaxScrollPosition);
      ScrollableTextUpdMsg* msg = COURIER_MESSAGE_NEW(ScrollableTextUpdMsg)(parentView->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), _currentMaxIndex, u32MaxScrollPosition);
      if (msg != NULL)
      {
         bool result = msg->Post();
         if (result)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ScrollableTextWidget2D:[%50s] in view [%50s] successfully posted ScrollableTextUpdMsg with current scroll position = [%u] and Maximum scroll position = [%u]", GetLegacyName(), parentView->GetId().CStr(), _currentMaxIndex, u32MaxScrollPosition));
         }
      }
   }
}


void ScrollableTextWidget2D::GetValidStartIndex(Candera::UInt32& startIndex)
{
   bool IsValidChar = false;
   Candera::UInt32 stepCount = 0;

   while (!IsValidChar && (stepCount < MAX_STEP_COUNT))
   {
      IsValidChar = IsValidByte(startIndex);
      if (!IsValidChar)
      {
         startIndex--;
      }
      stepCount++;
   }

   if (_startIndex > oOrginalString.GetCharCount())
   {
      _startIndex = static_cast<UInt32>(oOrginalString.GetCharCount());
   }
}


void ScrollableTextWidget2D::GetValidMaxBlockCharCount(Candera::UInt32 _sourceStartIndex, Candera::UInt32&   maxBlockCharCount)
{
   bool IsValidChar = false;
   Candera::UInt32 stepCount = 0;

   while (!IsValidChar && (stepCount < MAX_STEP_COUNT))
   {
      IsValidChar = IsValidByte(_sourceStartIndex + maxBlockCharCount);
      if (!IsValidChar)
      {
         maxBlockCharCount++;
      }
      stepCount++;
   }
}


bool ScrollableTextWidget2D::IsValidByte(Candera::UInt32 index)
{
   const TChar* l_pCharPtr = NULL;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(oOrginalString);
   l_pCharPtr = oOrginalString.GetCString();

   if (l_pCharPtr != 0)
   {
      unsigned u8StartByte = (unsigned char)l_pCharPtr[index];
      if (((u8StartByte & 0xC0) == 0x80) || ((u8StartByte >= 0xC0 || u8StartByte < 0x80) == false))
      {
         // Not a starting byte */
         return false;
      }
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   return true;
}
