/* ***************************************************************************************
* FILE:          RichText.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichText is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(RICHTEXT_H)
#define RICHTEXT_H

#ifdef WIN32
#define RICHTEXT_PERFORMANCE_CHECK
extern int rich_txt_perf_counter, rich_txt_perf_WrapText, rich_txt_perf_MeasureText;
#endif

//Header files
#include <vector>
#include <CanderaTypes.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/EngineBase/Common/BitmapTextRenderContext.h>
#include <FeatStd/Util/String.h>
#include <Candera/TextEngine/TextEngine.h>

// Enum data types
#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(RT_TextDirection)   \
        ENUM_DATA_TYPE_ITEM(RT_LeftToRight)  \
        ENUM_DATA_TYPE_ITEM(RT_RightToLeft)  \
        ENUM_DATA_TYPE_ITEM(RT_Automatic)    \
    ENUM_DATA_TYPE_END(RT_TextDirection)
#include <Candera/System/MetaInfo/EnumDataType.h>

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(RT_TextAlignment)  \
    ENUM_DATA_TYPE_ITEM(RT_Leading)         \
    ENUM_DATA_TYPE_ITEM(RT_Middle)          \
    ENUM_DATA_TYPE_ITEM(RT_Trailing)        \
    ENUM_DATA_TYPE_END(RT_TextAlignment)
#include <Candera/System/MetaInfo/EnumDataType.h>

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(RT_TextWrapMode)   \
    ENUM_DATA_TYPE_ITEM(RT_WrapNone)       \
    ENUM_DATA_TYPE_ITEM(RT_WrapWords)      \
    ENUM_DATA_TYPE_ITEM(RT_WrapCharacters)  \
    ENUM_DATA_TYPE_END(RT_TextWrapMode)
#include <Candera/System/MetaInfo/EnumDataType.h>

#define LINECHARCNT 1

namespace RichText {

// forward declarations
class RichTextChunk;
class Line;
class Paragraph;
class TextArea;
class RichText;

// types
typedef Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> tSharedStylePtr;
typedef Candera::Int16 tTabPosition;
typedef std::vector<tTabPosition> tTabStopList;

/****************************************************************************
*     Utf8Encoding
****************************************************************************/
class Utf8Encoding
{
   public:
      static Candera::UInt32 u32Copy(Candera::Char* pDst, Candera::UInt32 u32DstCharCount, const Candera::Char* pSrc);
      static Candera::UInt32 u32Copy(Candera::Char* pDst, Candera::UInt32 u32DstCharCount, const Candera::Char* pSrc, Candera::UInt32 u32CodePointCount);

      static Candera::UInt32 u32CodeLength(const Candera::Char* pCodePoint);
      static Candera::UInt32 u32Advance(const Candera::Char*& pCodePoint);

   private:
      FEATSTD_MAKE_CLASS_STATIC(Utf8Encoding);
};


/****************************************************************************
*     Position
****************************************************************************/
class Position
{
   public:
      Position(FeatStd::Int32 x, FeatStd::Int32 y) : m_x(x), m_y(y) {}

      void SetX(FeatStd::Int32 x)
      {
         m_x = x;
      }
      FeatStd::Int32 GetX() const
      {
         return m_x;
      }

      void SetY(FeatStd::Int32 y)
      {
         m_y = y;
      }
      FeatStd::Int32 GetY() const
      {
         return m_y;
      }

      void Translate(FeatStd::Int32 dx, FeatStd::Int32 dy)
      {
         m_x += dx;
         m_y += dy;
      }

   private:
      FEATSTD_MAKE_CLASS_STATIC(Position);
      FeatStd::Int32 m_x;
      FeatStd::Int32 m_y;
};


/****************************************************************************
*     Size
****************************************************************************/
class Size
{
   public:
      Size(FeatStd::Int32 width, FeatStd::Int32 height) : m_width(width), m_height(height) {}

      void SetWidth(FeatStd::Int32 width)
      {
         m_width = width;
      }
      FeatStd::Int32 GetWidth() const
      {
         return m_width;
      }

      void SetHeight(FeatStd::Int32 height)
      {
         m_height = height;
      }
      FeatStd::Int32 GetHeight() const
      {
         return m_height;
      }

   private:
      FEATSTD_MAKE_CLASS_STATIC(Size);
      FeatStd::Int32 m_width;
      FeatStd::Int32 m_height;
};


/****************************************************************************
*     RichTextRenderer
****************************************************************************/
class RichTextRenderer
{
   public:
      struct TextContext
      {
         TextContext();

         Candera::Color m_oTextColor;
         tSharedStylePtr m_oStyle;
         Candera::TextRendering::Metrics m_stFontMetrics;
         Paragraph* m_pCurrentParagraph;
         Line* m_pCurrentLine;

         static Candera::BitmapTextRenderContext m_oTextRenderContext;

         static Candera::TextRendering::TextRenderer m_oTextRenderer;
      };

      static void vLayout(RichText* pRichText, TextArea** ppTextArea);

      static void vDraw(RichText* pRichText, const Position& cooOffset, const Size& cooSize, Candera::Bitmap::SharedPointer& bitmap);

      //static RichTextChunk* pGetStartChunkForListPageup(RichText* pRichText, Candera::UInt16 i16NumofLinesPerTextArea);
      typedef std::vector<Candera::UInt32> LineList;
      static void vCalculateRenderdCharCnt(RichTextChunk* pChunk, LineList& lineList, Candera::UInt32& LineCharCount);
};


/****************************************************************************
*     Line
****************************************************************************/
class Line
{
   public:
      Line();
      ~Line();

      void vSetKerningCharacter(const Candera::TChar* copKerningChar)
      {
         m_copKerningChar = copKerningChar;
      }
      const Candera::TChar* copGetKerningCharacter() const
      {
         return m_copKerningChar;
      }

      Candera::Int32 i32GetCurrentPosition() const
      {
         return m_i32CurrentPosition;
      }

      const Size& cooGetContentSize() const
      {
         return m_oContentSize;
      }

      void i32SetLineHeight(Candera::Int32 i32LineHeight)
      {
         m_i32LineHeight = i32LineHeight;
      }
      Candera::Int32 i32GetLineHeight() const
      {
         return m_i32LineHeight;
      }

      void vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk);

      void vTranslate(Candera::Int32 i32X, Candera::Int32 i32Y);

      void vEndLine(const RichTextRenderer::TextContext* pTextContext);

      RichTextChunk* pGetFirstChunk()
      {
         return m_pFirstChunk;
      }
      RichTextChunk* pGetLastChunk()
      {
         return m_pFirstChunk;
      }

   private:
      void vDoAlignment();

      RichTextChunk* m_pFirstChunk;
      RichTextChunk* m_pLastChunk;

      Candera::Int32 m_i32CurrentPosition;
      Size m_oContentSize;

      Candera::Int32 m_i32LineHeight;

      const Candera::TChar* m_copKerningChar;
};


/****************************************************************************
*     Paragraph
****************************************************************************/
class Paragraph
{
   public:
      Paragraph();
      ~Paragraph();

      void vSetSize(const Size& cooSize)
      {
         m_oSize = cooSize;
      }
      const Size& cooGetSize() const
      {
         return m_oSize;
      }

      const Size& cooGetContentSize() const
      {
         return m_oContentSize;
      }

      const Size& cooGetEffectiveSize() const
      {
         return m_oEffectiveSize;
      }

      void vSetLineSpacingFactor(Candera::Float flLineSpacingFactor)
      {
         m_flLineSpacingFactor = flLineSpacingFactor;
      }
      Candera::Float coflGetLineSpacingFactor() const
      {
         return m_flLineSpacingFactor;
      }

      void vSetHorizontalAlignment(Candera::RT_TextAlignment enHorizontalAlignment)
      {
         m_enHorizontalAlignment = enHorizontalAlignment;
      }
      Candera::RT_TextAlignment coenGetHorizontalAlignment() const
      {
         return m_enHorizontalAlignment;
      }

      void vSetVerticalAlignment(Candera::RT_TextAlignment enVerticalAlignment)
      {
         m_enVerticalAlignment = enVerticalAlignment;
      }
      Candera::RT_TextAlignment coenGetVerticalAlignment() const
      {
         return m_enVerticalAlignment;
      }

      void vSetTextDirection(Candera::RT_TextDirection enTextDirection)
      {
         m_enTextDirection = enTextDirection;
      }
      Candera::RT_TextDirection coenGetTextDirection() const
      {
         return m_enTextDirection;
      }

      void vSetTabStopList(tTabStopList* pTabStopList)
      {
         m_pTabStopList = pTabStopList;
      }
      const tTabStopList* GetTabStopList() const
      {
         return m_pTabStopList;
      }

      void vSetConstantTabWidth(Candera::Int32 i32ConstantTabWidth)
      {
         m_i32ConstantTabWidth = i32ConstantTabWidth;
      }
      Candera::Int32 i32GetConstantTabWidth() const
      {
         return m_i32ConstantTabWidth;
      }

      void vSetTextWrapMode(Candera::RT_TextWrapMode enTextWrapMode)
      {
         m_enTextWrapMode = enTextWrapMode;
      }
      Candera::RT_TextWrapMode enGetTextWrapMode() const
      {
         return m_enTextWrapMode;
      }

      void vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk);

      void vTranslate(Candera::Int32 i32X, Candera::Int32 i32Y);

      void vEndParagraph(RichTextRenderer::TextContext* pTextContext);

      Candera::SizeType uGetLineCount() const
      {
         return m_oLineList.size();
      }
      Line* pGetLine(Candera::SizeType uIndex)
      {
         return m_oLineList[uIndex];
      }

   private:
      void vDoAlignment();

      void PerformLineSpacing();
      void PerformHorizontalAlignment();
      void PerformVerticalAlignment();

      std::vector<Line*> m_oLineList;

      Size m_oSize;
      Size m_oContentSize;
      Size m_oEffectiveSize;

      Candera::Float m_flLineSpacingFactor;

      Candera::RT_TextAlignment m_enHorizontalAlignment;
      Candera::RT_TextAlignment m_enVerticalAlignment;

      Candera::RT_TextDirection m_enTextDirection;

      tTabStopList* m_pTabStopList;
      Candera::Int32 m_i32ConstantTabWidth;

      Candera::RT_TextWrapMode m_enTextWrapMode;
};


/****************************************************************************
*     TextArea
****************************************************************************/
class TextArea
{
   public:
      TextArea();
      ~TextArea();

      void vSetSize(const Size& cooSize)
      {
         m_oSize = cooSize;
      }
      const Size& cooGetSize() const
      {
         return m_oSize;
      }

      const Size& cooGetContentSize() const
      {
         return m_oContentSize;
      }

      const Size& cooGetEffectiveSize() const
      {
         return m_oEffectiveSize;
      }

      void vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk);

      void vEndTextArea(RichTextRenderer::TextContext* pTextContext);

      Candera::SizeType uGetParagraphCount() const
      {
         return m_oParagraphList.size();
      }
      Paragraph* pGetParagraph(Candera::SizeType uIndex)
      {
         return m_oParagraphList[uIndex];
      }

      Candera::SizeType uGetLineCount();
      Line* pGetLine(Candera::SizeType uIndex);

      Position GetLineCoordinate(Candera::SizeType uIndex);

   private:
      void vDoAlignment();

      std::vector<Paragraph*> m_oParagraphList;

      Size m_oSize;
      Size m_oContentSize;
      Size m_oEffectiveSize;
};


/****************************************************************************
*     RichText
****************************************************************************/
class RichText
{
   public:
      RichText();
      ~RichText();

      void vSetSize(const Size& cooSize)
      {
         m_oSize = cooSize;
      }
      const Size& cooGetSize() const
      {
         return m_oSize;
      }

      void vSetEffectiveSize(const Size& cooEffectiveSize)
      {
         m_oEffectiveSize = cooEffectiveSize;
      }
      const Size& cooGetEffectiveSize() const
      {
         return m_oEffectiveSize;
      }

      void vAddChunk(RichTextChunk* pChunk);
      void vRemoveChunk(RichTextChunk* pChunk, bool bRemoveSubsequent);

      RichTextChunk* pGetFirstChunk()
      {
         return m_pFirstChunk;
      }

      //Added for ScrollableText Widget
      RichTextChunk* pGetLastChunk()
      {
         return m_pLastChunk;
      }
   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(RichText);

      RichTextChunk* m_pFirstChunk;
      RichTextChunk* m_pLastChunk;

      Size m_oSize;
      Size m_oEffectiveSize;
};


/****************************************************************************
*     RichTextChunk
****************************************************************************/
class RichTextChunk
{
   public:
      RichTextChunk();
      virtual ~RichTextChunk();

      virtual void vLayout(RichTextRenderer::TextContext* pContext)
      {
         FEATSTD_UNUSED(pContext);
      }
      virtual void vRender(RichTextRenderer::TextContext* pContext, const Candera::TextRendering::TextCoordinate& cooPosition)
      {
         FEATSTD_UNUSED(pContext);
         FEATSTD_UNUSED(cooPosition);
      }

      void vSetChunkPosition(const Position& cooChunkPosition)
      {
         m_oChunkPosition = cooChunkPosition;
      }
      const Position& cooGetChunkPosition() const
      {
         return m_oChunkPosition;
      }

      void vSetChunkSize(const Size& cooChunkSize)
      {
         m_oChunkSize = cooChunkSize;
      }
      const Size& cooGetChunkSize() const
      {
         return m_oChunkSize;
      }

      RichTextChunk* pGetNextChunk()
      {
         return m_pNextChunk;
      }
      void vSetNextChunk(RichTextChunk* pNextChunk)
      {
         m_pNextChunk = pNextChunk;
      }

      virtual RichTextChunk* pAdvance()
      {
         return m_pNextChunk;
      }

      /*Added for ScrolableTextWidget */
      Candera::UInt32 u32GetRenderdCharCount()
      {
         return m_u32renderdCharCount;
      }

      void vSetRenderdCharCount(Candera::UInt32 u32renderdCharCount)
      {
         m_u32renderdCharCount = u32renderdCharCount;
      }

      /*Added for ScrolableTextWidget */
      Candera::UInt32 u32GetFirstLineRenderdCharCount()
      {
         return m_u32FirstLinerenderdCharCount;
      }

      void vSetFirstLineRenderdCharCount(Candera::UInt32 u32renderdCharCount)
      {
         m_u32FirstLinerenderdCharCount = u32renderdCharCount;
      }

      /*Added for ScrolableTextWidget */
      Candera::UInt32 u32GetLastLineRenderdCharCount()
      {
         return m_u32LastLinerenderdCharCount;
      }

      void vSetLastLineRenderdCharCount(Candera::UInt32 u32renderdCharCount)
      {
         m_u32LastLinerenderdCharCount = u32renderdCharCount;
      }

      bool bGetIsPageOrLineDown()
      {
         return m_bisPageOrLineDown;
      }

      void vSetIsPageOrLineDown(bool bisPageOrLineDown)
      {
         m_bisPageOrLineDown = bisPageOrLineDown;
      }

      Candera::UInt16 u16GetNumOfLinesPerTextArea()
      {
         return m_u16NumOfLinesPerTextArea;
      }
      void vSetNumOfLinesPerTextArea(Candera::UInt16 u16NumOfLinesPerTextArea)
      {
         m_u16NumOfLinesPerTextArea = u16NumOfLinesPerTextArea;
      }
      RichTextChunk* pGetPreviousChunk(RichTextChunk* pChunk, RichText* pRichText)
      {
         RichTextChunk* ptemp;
         ptemp = pRichText->pGetFirstChunk();

         while ((pChunk != NULL) && (NULL != ptemp) && (ptemp->pGetNextChunk() != pChunk))
         {
            ptemp = ptemp->pGetNextChunk();
         }

         return ptemp;
      }
      bool bGetPageUpStartChunk()
      {
         return m_bisPageUpStartChunk;
      }

      void vSetPageUpStartChunk(bool bisPageUpStartChunk)
      {
         m_bisPageUpStartChunk = bisPageUpStartChunk;
      }

      bool bGetIsLineUp()
      {
         return _isLineUp;
      }

      void vSetLineUp(bool bLineUp)
      {
         _isLineUp = bLineUp;
      }

      bool bGetLineUpStartChunk()
      {
         return m_bisLineUpStartChunk;
      }

      void vSetLineUpStartChunk(bool bisLineUpStartChunk)
      {
         m_bisLineUpStartChunk = bisLineUpStartChunk;
      }

   protected:
      Position m_oChunkPosition;
      Size m_oChunkSize;
      bool m_bisPageUpStartChunk;

      RichTextChunk* m_pNextChunk;
      RichText* pRichtext;
      bool m_bisLineUpStartChunk;
   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(RichTextChunk);

      static Candera::UInt32 m_u32renderdCharCount;
      static Candera::UInt32 m_u32FirstLinerenderdCharCount;
      static Candera::UInt32 m_u32LastLinerenderdCharCount;
      static bool m_bisPageOrLineDown;
      static Candera::UInt16 m_u16NumOfLinesPerTextArea;
      static bool _isLineUp;
};


/****************************************************************************
*     ParagraphChunk
****************************************************************************/
class ParagraphChunk : public RichTextChunk
{
   public:
      ParagraphChunk();
      ParagraphChunk(Candera::RT_TextAlignment enHAlignment, Candera::RT_TextAlignment enVAlignment, Candera::RT_TextDirection enTextDirection, const Size& cooSize, Candera::Float flLineSpacingFactor, Candera::RT_TextWrapMode enTextWrapMode, Candera::Int16 i16ConstantTabWidth);

      virtual void vLayout(RichTextRenderer::TextContext* pContext);

      void vSetHorizontalAlignment(Candera::RT_TextAlignment enHorizontalAlignment)
      {
         m_enHorizontalAlignment = enHorizontalAlignment;
      }
      Candera::RT_TextAlignment enGetHorizontalAlignment() const
      {
         return m_enHorizontalAlignment;
      }

      void vSetVerticalAlignment(Candera::RT_TextAlignment enVerticalAlignment)
      {
         m_enVerticalAlignment = enVerticalAlignment;
      }
      Candera::RT_TextAlignment enGetVerticalAlignment() const
      {
         return m_enVerticalAlignment;
      }

      void vSetTextDirection(Candera::RT_TextDirection enTextDirection)
      {
         m_enTextDirection = enTextDirection;
      }
      Candera::RT_TextDirection enGetTextDirection() const
      {
         return m_enTextDirection;
      }

      void vSetSize(const Size& cooSize)
      {
         m_oSize = cooSize;
      }
      const Size& cooGetSize() const
      {
         return m_oSize;
      }

      void vSetLineSpacingFactor(Candera::Float flLineSpacingFactor)
      {
         m_flLineSpacingFactor = flLineSpacingFactor;
      }
      Candera::Float coflGetLineSpacingFactor() const
      {
         return m_flLineSpacingFactor;
      }

      void vSetTextWrapMode(Candera::RT_TextWrapMode enTextWrapMode)
      {
         m_enTextWrapMode = enTextWrapMode;
      }
      Candera::RT_TextWrapMode enGetTextWrapMode() const
      {
         return m_enTextWrapMode;
      }

      void vSetTabStopList(const tTabStopList& oTabStopList)
      {
         m_oTabStopList = oTabStopList;
      }

      void vSetConstantTabWidth(Candera::Int16 i16ConstantTabWidth)
      {
         m_i16ConstantTabWidth = i16ConstantTabWidth;
      }
      Candera::Int16 i16GetConstantTabWidth() const
      {
         return m_i16ConstantTabWidth;
      }

      Candera::Int32 i32GetLineHeight() const
      {
         return m_LineHeight;
      }

      void vSetLineHeight(Candera::Int32 LineHeight)
      {
         m_LineHeight = LineHeight;
      }
   private:
      Candera::RT_TextAlignment m_enHorizontalAlignment;
      Candera::RT_TextAlignment m_enVerticalAlignment;

      Candera::RT_TextDirection m_enTextDirection;

      Size m_oSize;

      Candera::Float m_flLineSpacingFactor;

      Candera::RT_TextWrapMode m_enTextWrapMode;

      tTabStopList m_oTabStopList;
      Candera::Int16 m_i16ConstantTabWidth;
      Candera::Int32 m_LineHeight;
};


/****************************************************************************
*     TextChunk
****************************************************************************/
class TextChunk : public RichTextChunk
{
   public:
      TextChunk();
      TextChunk(const FeatStd::String& oText);
      virtual ~TextChunk();

      virtual void vLayout(RichTextRenderer::TextContext* pContext);

      void SetKerningCharacterForNextBlock(Line* pCurrentLine);

      void CheckCurrentKerningCharacter(const Line* pCurrentLine, const RichTextRenderer::TextContext* pContext);

      virtual void vRender(RichTextRenderer::TextContext* pContext, const Candera::TextRendering::TextCoordinate& cooPosition);

      virtual RichTextChunk* pAdvance();

      void vSetText(const FeatStd::String& cooText)
      {
         m_oText = cooText;
      }
      const FeatStd::String& cooGetText() const
      {
         return m_oText;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(TextChunk);

      const Candera::TextRendering::TextRect cooMeasureText(const RichTextRenderer::TextContext* pContext, const Candera::TChar* copText, Candera::TextRendering::TextLength i32TextLength) const;
      bool IsValidByte(Candera::UInt32 index);
      void vWrapText(const RichTextRenderer::TextContext* pContext, Candera::Int32 i32RemainingWidth);

      FeatStd::String m_oText;

      TextChunk* m_pWrapChild;
      TextChunk* m_pWrapParent;
      friend class RichTextRenderer;
};


/****************************************************************************
*     StyleChunk
****************************************************************************/
class StyleChunk : public RichTextChunk
{
   public:
      StyleChunk();
      StyleChunk(tSharedStylePtr oStyle);

      virtual void vLayout(RichTextRenderer::TextContext* pContext);
      virtual void vRender(RichTextRenderer::TextContext* pContext, const Candera::TextRendering::TextCoordinate& cooPosition);

      tSharedStylePtr oGetStyle() const
      {
         return m_oStyle;
      }
      void vSetStyle(tSharedStylePtr oSharedStyle)
      {
         m_oStyle = oSharedStyle;
      }

   private:
      tSharedStylePtr m_oStyle;
};


/****************************************************************************
*     ColorChunk
****************************************************************************/
class ColorChunk : public RichTextChunk
{
   public:
      ColorChunk();
      ColorChunk(const Candera::Color& color, Candera::UInt32 u32ColorCharCnt);
      virtual void vRender(RichTextRenderer::TextContext* pContext, const Candera::TextRendering::TextCoordinate& cooPosition);
      void SetColor(const Candera::Color& color)
      {
         _color = color;
      }

      void vSetColorCharCnt(Candera::UInt32 u32ColorCharCnt)
      {
         m_u32ColorCharCnt = u32ColorCharCnt;
      }
      Candera::UInt32 u32GetColorCharCnt() const
      {
         return m_u32ColorCharCnt;
      }

   private:
      Candera::Color _color;
      Candera::UInt32 m_u32ColorCharCnt;
};


/****************************************************************************
*     NewLineChunk
****************************************************************************/
class NewLineChunk : public RichTextChunk
{
      // No new functionality needed
   public:
      NewLineChunk()
      {
         _u32CharCnt = 1;
      }

      NewLineChunk(Candera::UInt32 u32CharCnt)
      {
         _u32CharCnt = u32CharCnt;
      }

      Candera::UInt32 getNewLineCharCount()
      {
         return _u32CharCnt;
      }

      virtual ~NewLineChunk()
      {
      }
   private:
      Candera::UInt32 _u32CharCnt;
};


/****************************************************************************
*     TabChunk
****************************************************************************/
class TabChunk : public RichTextChunk
{
   public:
      TabChunk();
      TabChunk(Candera::UInt32 u32TabId, Candera::UInt32 u32TabCharCnt);

      void vSetTabId(Candera::UInt32 u32TabId)
      {
         m_u32TabId = u32TabId;
      }
      Candera::UInt32 u32GetTabId() const
      {
         return m_u32TabId;
      }

      void vSetTabCharCnt(Candera::UInt32 u32TabCharCnt)
      {
         m_u32tabCharCnt = u32TabCharCnt;
      }
      Candera::UInt32 u32GetTabCharCnt() const
      {
         return m_u32tabCharCnt;
      }

      virtual void vLayout(RichTextRenderer::TextContext* pContext);

   private:
      Candera::UInt32 m_u32TabId;
      Candera::UInt32  m_u32tabCharCnt;
};


} // namespace RichText

#endif
