/* ***************************************************************************************
* FILE:          RtTextInStream.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtTextInStream is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/Base.h>
#include <FeatStd/Util/String.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

namespace hmibase {
namespace widget {
namespace richtext {

class TextInStream
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(TextInStream);

      TextInStream()
      {
      }

      virtual ~TextInStream()
      {
      }

      /**
       *  Provides the number of characters that can be retrieved from the stream without blocking waits.
       */
      virtual FeatStd::UInt32 GetAvailableCharCount() const = 0;

      /*
       *  Provides the length of the complete underlying text or 'UNKNOWN_SIZE' if the size of the text is
       *  not known and need to be determined differently.
       */
      virtual FeatStd::UInt32 GetTextLength() const = 0;

      /*
       *  Determines whether or not the end of the stream is reached. This method is required for definite
       *  'end of stream' determinations, because data accessor method may return empty strings for other reasons as well.
       */
      virtual bool IsEndOfStreamReached() const = 0;

      /*
       *  Provides the the UCS4 (UTF-32) encoded character at the current stream position without advancing the stream position.
       *  Subsequent calls of this method therefore provide always the identical result. The method blocks until the character is
       *  available.
       *  The null-character ('\0') is returned, if the current position is at the end of the stream (after the last character).
       */
      virtual FeatStd::UInt32 PeekChar() const = 0;

      /*
       *  Provides the character sequence at the current stream position up to the length indicated by parameter 'count'
       *  without advancing the stream position. Subsequent calls of this method therefore provide always the identical result.
       *  This method blocks until a sufficient number of characters are available or the end of the stream is reached.
       *  The size of the returned string may therefore be less than the requested character count. An empty string is returned,
       *  if the current position is at the end of the stream (after the last character).
       */
      virtual FeatStd::String PeekString(FeatStd::UInt32 count) const = 0;

      /*
       *  Provides the the UCS4 (UTF-32) encoded character at the current stream position and advances the stream position to the
       *  next character. The method blocks until the character is available. The null-character ('\0') is returned, if the
       *  current position is at the end of the stream (after the last character).
       */
      virtual FeatStd::UInt32 ReadChar() = 0;

      /*
       *  Provides the character sequence at the current stream position containing all characters to the end of the stream.
       *  This method blocks until all characters up to the end of the stream are available. An empty string is returned,
       *  if the current position is at the end of the stream (after the last character).
       *  This method can be useful if further processing implementations require a complete string.
       */
      virtual FeatStd::String ReadCompleteText() = 0;

      /*
       *  Provides the character sequence at the current stream position up to the length indicated by parameter 'count'
       *  and advances the stream position to next character following the provided sequence. This method blocks until a
       *  sufficient number of characters are available or the end of the stream is reached. The size of the returned string
       *  may therefore be less than the requested character count. An empty string is returned, if the current position is
       *  at the end of the stream (after the last character).
       */
      virtual FeatStd::String ReadString(FeatStd::UInt32 count) = 0;

      /*
       *  Sets the internal stream position pointer to the start of the stream.
       */
      virtual void Reset() = 0;

   private:
      FEATSTD_SHARED_POINTER_DECLARATION();
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
