/* ***************************************************************************************
* FILE:          RtStringInStream.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtStringInStream is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtStringInStream.h"
#include <FeatStd/Platform/String.h>
#include <FeatStd/MemoryManagement/Heap.h>
#include <hmibase/util/Macros.h>

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::Internal::String;

StringInStream::StringInStream(FeatStd::String string) :
   m_string(string),
   m_it(string.GetCString()),
   m_pos(string.GetCString()),
   m_availableCharCount(GetTextLengthImpl())
{
}


FeatStd::String StringInStream::PeekString(FeatStd::UInt32 count) const
{
   FeatStd::SizeType byteLength = 0;
   FeatStd::Internal::Utf8TextIterator it(m_pos);
   while ((*it != 0) && (count-- > 0))
   {
      byteLength += FeatStd::SizeType(it.Advance());
   }

   FeatStd::TChar* buffer = FEATSTD_NEW_ARRAY(FeatStd::TChar, byteLength + 1);
   FeatStd::Internal::String::CopyPartial(buffer, m_pos, byteLength);
   buffer[byteLength] = '\0';
   FeatStd::String s(buffer);
   FEATSTD_DELETE_ARRAY(buffer);
   return s;
}


FeatStd::UInt32 StringInStream::ReadChar()
{
   FeatStd::UInt32 c = PeekChar();
   m_pos += m_it.Advance();
   return c;
}


FeatStd::String StringInStream::ReadCompleteText()
{
   FeatStd::String s(m_pos);
   m_pos = "";
   m_it = "";
   return s;
}


FeatStd::String StringInStream::ReadString(FeatStd::UInt32 count)
{
   FeatStd::String s(PeekString(count));
   m_pos += s.GetCharCount();
   return s;
}


void StringInStream::Reset()
{
   m_pos = m_string.GetCString();
   m_it = m_string.GetCString();
}


//***********************************************************************************************
#define ANGSTROM_SMALL   "\xC3\xA5"
#define ANGSTROM_CAPITAL "\xE2\x84\xAB"

class StringInStreamTest
{
   public:
      StringInStreamTest()
      {
         const FeatStd::TChar* testString = ANGSTROM_CAPITAL "12" ANGSTROM_SMALL "3456789" ANGSTROM_CAPITAL "abcdefg";
         StringInStream s(testString);
         FeatStd::UInt32 l = s.GetAvailableCharCount();
         FEATSTD_UNUSED(l);
         FEATSTD_DEBUG_ASSERT(l == String::TLength(testString));

         l = s.GetTextLength();
         FEATSTD_DEBUG_ASSERT(l == String::TLength(testString));

         bool end = s.IsEndOfStreamReached();
         FEATSTD_UNUSED(end);
         FEATSTD_DEBUG_ASSERT(!end);

         FeatStd::UInt32 c = s.ReadChar();
         FeatStd::Internal::Utf8TextIterator it(testString);
         FEATSTD_UNUSED(c);
         FEATSTD_DEBUG_ASSERT(c == (*it));   // iterator points to first character in testString;

         c = s.PeekChar();
         it.Advance(); // iterator points to second character in testString;
         FEATSTD_DEBUG_ASSERT(c == (*it));

         FeatStd::String str = s.ReadString(8);
         FEATSTD_DEBUG_ASSERT(str == "12" ANGSTROM_SMALL "34567");
         str = s.PeekString(5);
         FEATSTD_DEBUG_ASSERT(str == "89" ANGSTROM_CAPITAL "ab");

         str = s.ReadCompleteText();
         FEATSTD_DEBUG_ASSERT(str == "89" ANGSTROM_CAPITAL "abcdefg");

         end = s.IsEndOfStreamReached();
         FEATSTD_DEBUG_ASSERT(end);
      }
};


static StringInStreamTest test;
} // namespace richtext
} // namespace widget
} // namespace hmibase
