/* ***************************************************************************************
* FILE:          RichTextWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichTextWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RichTextWidget2D.h"
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <Candera/Engine2D/Core/RenderNode.h>

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/Layouter.h>
#else
#include <Candera/Engine2D/Layout/Layouter.h>
#endif

#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColorBlend.h>
#include <Widgets/2D/RichText/ContentProvider/RtAssetUrlResolver.h>
#include <Widgets/2D/RichText/ContentProvider/RtContentProvider.h>
#include <Widgets/2D/RichText/Engine/RtDynamicProperties.h>
#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/Utils/RtStringInStream.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RichTextWidget2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {


CGI_WIDGET_RTTI_DEFINITION(RichTextWidget2D);

static Candera::AssetProvider* s_defaultAssetProvider = &Candera::DefaultAssetProvider::GetInstance();

RichTextWidget2D::RichTextWidget2D() :
   m_richTextEngine(Engine::Create()),
   m_update(true),
   m_updateOffset(false),
   m_sceneComposerContext(false)
{
   FEATSTD_DEBUG_ASSERT(m_richTextEngine != 0);
}


RichTextWidget2D::~RichTextWidget2D()
{
   DynamicProperties::ClearEngine(GetNode());
}


bool RichTextWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const RichTextWidget2D* original = CLONEABLE_WIDGET_CAST<const RichTextWidget2D*>(originalWidget);
      if (original != 0)
      {
         SetSourceType(original->GetSourceType());

         if (ControlTemplateBinding::IsTextBindable(*this))
         {
            SetSource(ControlTemplateBinding::GetTextValue(*this));
         }
         else
         {
            SetSource(original->GetSource());
         }

         _TODO("consider data binding")
         SetTextStyle(original->GetTextStyle());
         SetStyleSheetUrl(original->GetStyleSheetUrl());
         SetStyle(original->GetStyle());
         SetColorEnabled(original->GetColorEnabled());
         SetColor(original->GetColor());
         SetOffset(original->GetOffset());
         SetMaxNumberOfLines(original->GetMaxNumberOfLines());
         SetHorizontalAlignmentEnabled(original->GetHorizontalAlignmentEnabled());
         SetHorizontalAlignment(original->GetHorizontalAlignment());
         SetBaselineOffset(original->GetBaselineOffset());
         SetAsynchronousRendering(original->GetAsynchronousRendering());
         SetSlicedRendering(original->GetSlicedRendering());
         SetSliceHeight(original->GetSliceHeight());
         SetCultureDependentAlignment(original->GetCultureDependentAlignment());

         cloned = true;
      }
   }
   return cloned;
}


void RichTextWidget2D::Init(Candera::AssetProvider* assetProvider)
{
   Base::Init(assetProvider);
   m_currentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();

   AssetUrlResolver::GetInstance().SetAssetProvider(assetProvider);
   InvalidateLayout();

   if (assetProvider != s_defaultAssetProvider)
   {
      m_sceneComposerContext = true;
   }
}


void RichTextWidget2D::Update()
{
   Base::Update();

   if (m_update && (m_richTextEngine != 0))
   {
      Engine::Data data;
      data.m_markup = Markup::Html;

      if (GetSourceType() == SrcType::String)
      {
         data.m_source = TextInStream::SharedPointer(FEATSTD_NEW(StringInStream)(GetSource()));
      }
      else
      {
         data.m_source = ContentProvider::GetInstance().GetTextResource(0, GetSource().GetCString());
      }

      Candera::TextRendering::SharedStyle::SharedPointer textStyle = GetTextStyle();
      if (textStyle == 0)
      {
         Candera::TextRendering::FontStore* fontStore = Candera::TextRendering::FontEngine::Instance().GetFontStore();
         if (fontStore != 0)
         {
            const FeatStd::Char* faceName = fontStore->GetFaceName(1);
            if ((faceName != 0) && (*faceName != '\0'))
            {
               CustomStyleCache* styleCache = CustomStyleCache::Get(faceName, 20);
               if (styleCache != 0)
               {
                  textStyle = styleCache->GetStyle();
               }
            }
         }
      }
      data.m_textStyle = textStyle;
      data.m_styleSheetUrl = GetStyleSheetUrl();
      data.m_style = GetStyle();
      data.m_colorEnabled = GetColorEnabled();
      data.m_color = GetColor();
      data.m_maxNumberOfLines = GetMaxNumberOfLines();
      data.m_horizontalAlignmentEnabled = GetHorizontalAlignmentEnabled();
      data.m_horizontalAlignment = GetHorizontalAlignment();
      data.m_baselineOffset = GetBaselineOffset();
      data.m_asynchronousRendering = m_sceneComposerContext ? false : GetAsynchronousRendering();
      data.m_slicedRendering = GetSlicedRendering();
      data.m_sliceHeight = GetSliceHeight();
      data.m_view = GetParentView();
      data.m_cultureDependentAlignment = GetCultureDependentAlignment();

      m_richTextEngine->SetData(data);
      InvalidateLayout();
      Invalidate();
      m_update = false;
   }

   if (m_updateOffset)
   {
      m_richTextEngine->GetViewport()->SetOffset(GetOffset());
      InvalidateLayout();
      Invalidate();
      m_updateOffset = false;
   }
}


void RichTextWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case SourceTypePropertyId:
      case SourcePropertyId:
      case TextStylePropertyId:
      case StyleSheetUrlPropertyId:
      case StylePropertyId:
      case ColorEnabledPropertyId:
      case ColorPropertyId:
      case MaxNumberOfLinesPropertyId:
      case HorizontalAlignmentEnabledPropertyId:
      case HorizontalAlignmentPropertyId:
      case BaselineOffsetPropertyId:
      case CultureDependentAlignmentPropertyId:
         m_update = true;
         break;

      case OffsetPropertyId:
         m_updateOffset = true;
         break;

      default:
         break;
   }
}


void RichTextWidget2D::OnBeforeNodeChanged()
{
   Base::OnBeforeNodeChanged();

   RichTextRenderer::SharedPointer renderer = m_richTextEngine->GetRenderer();
   if (renderer != 0)
   {
      renderer->SetRootNode(0);
   }

   DynamicProperties::ClearEngine(GetNode());
}


void RichTextWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();

   Engine& engine = *m_richTextEngine;
   RichTextRenderer::SharedPointer renderer = engine.GetRenderer();
   if (renderer != 0)
   {
      renderer->SetRootNode(GetNode());
   }

   DynamicProperties::SetEngine(GetNode(), &engine);
}


void RichTextWidget2D::CultureChanged()
{
   CulturePtr culture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
   if (m_currentCulture != culture)
   {
      m_currentCulture = culture;
      m_update = true;
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
