/* ***************************************************************************************
* FILE:          RichTextMarqueeWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichTextMarqueeWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RichTextMarqueeWidget2D.h"
#include "RichTextWidget2D.h"
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Widgets/2D/RichText/Engine/RtDynamicProperties.h>
#include <Widgets/2D/WidgetFinder2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RichTextMarqueeWidget2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {


CGI_WIDGET_RTTI_DEFINITION(RichTextMarqueeWidget2D);


RichTextMarqueeWidget2D::RichTextMarqueeWidget2D() :
   m_state(Disabled),
   m_offset(0.0F),
   m_visibleWidth(0.0F),
   m_totalWidth(0.0F),
   m_loopCount(0),
   m_reverse(false),
   m_updateStartStop(false)
{
}


RichTextMarqueeWidget2D::~RichTextMarqueeWidget2D()
{
}


void RichTextMarqueeWidget2D::Update()
{
   Base::Update();

   if (m_updateStartStop)
   {
      ChangeState(AllowRestart() ? WaitStart : Disabled);
   }

   Engine* engine = DynamicProperties::GetEngine(GetNode());
   if (engine != 0)
   {
      FeatStd::Float offsetStart = GetStartEmpty() ? m_visibleWidth : 0.0F;
      FeatStd::UInt32 expired = m_ticks.GetExpiredTime();

      switch (m_state)
      {
         case WaitStart:
            m_offset = offsetStart;
            if (expired >= GetDelayStart())
            {
               m_reverse = false;
               ChangeState(Running);
            }
            break;

         case Running:
         {
            FeatStd::Float end = (GetEndEmpty() ? 0.0F : m_visibleWidth) - m_totalWidth;
            FeatStd::Float distance = 0.001F * FeatStd::Float(GetSpeed() * expired);
            m_offset = m_reverse ? end + distance : offsetStart - distance;
            if ((m_offset <= end) || (m_offset >= offsetStart))
            {
               m_offset = m_reverse ? offsetStart : end;
               ChangeState(WaitEnd);
            }
            break;
         }

         case WaitEnd:
            if (expired >= GetDelayEnd())
            {
               if (GetBounce())
               {
                  if (!m_reverse)
                  {
                     m_reverse = true;
                     ChangeState(Running);
                  }
                  else
                  {
                     m_reverse = false;
                  }
               }

               if (!m_reverse)
               {
                  m_loopCount++;
                  ChangeState(AllowRestart() ? WaitStart : Disabled);
               }
            }
            break;

         case Disabled:
         default:
            m_offset = 0.0F;
            break;
      }

      Viewport::SharedPointer viewport = engine->GetViewport();
      if (viewport != 0)
      {
         viewport->SetOffset(m_offset, 0.0F);
         if (m_state != Disabled)
         {
            Invalidate();
            InvalidateLayout();
         }
      }
   }
}


bool RichTextMarqueeWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const RichTextMarqueeWidget2D* original = CLONEABLE_WIDGET_CAST<const RichTextMarqueeWidget2D*>(originalWidget);
      if (original != 0)
      {
         SetSpeed(original->GetSpeed());
         SetDelayStart(original->GetDelayStart());
         SetDelayEnd(original->GetDelayEnd());
         SetStartEmpty(original->GetStartEmpty());
         SetEndEmpty(original->GetEndEmpty());
         SetRepeatCount(original->GetRepeatCount());

         cloned = true;
      }
   }
   return cloned;
}


void RichTextMarqueeWidget2D::Init(Candera::AssetProvider* assetProvider)
{
   Base::Init(assetProvider);
   OnChanged(EnabledPropertyId);
}


void RichTextMarqueeWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case EnabledPropertyId:
         m_loopCount = 0;
         m_updateStartStop = true;
         break;

      default:
         break;
   }
}


void RichTextMarqueeWidget2D::OnBeforeNodeChanged()
{
   DynamicProperties::ClearMarqueeEnabled(GetNode());
}


void RichTextMarqueeWidget2D::OnNodeChanged()
{
   DynamicProperties::SetMarqueeEnabled(GetNode(), true);
   Invalidate();
}


bool RichTextMarqueeWidget2D::AllowRestart()
{
   bool allowRestart = false;

   Engine* engine = DynamicProperties::GetEngine(GetNode());
   if (engine != 0)
   {
      const RichTextRenderer::SharedPointer& renderer = engine->GetRenderer();
      if (renderer != 0)
      {
         const RichTextRenderer::MarqueeInfo& marqueeInfo = renderer->GetMarqueeInfo();
         if (marqueeInfo.m_isValid)
         {
            m_visibleWidth = marqueeInfo.m_visibleWidth;
            m_totalWidth = marqueeInfo.m_totalWidth;
            m_updateStartStop = false;
            if (m_visibleWidth < m_totalWidth)
            {
               const FeatStd::UInt32 repeatCount = GetRepeatCount();
               allowRestart = IsEnabled() && ((repeatCount == 0) || (m_loopCount < repeatCount));
            }
         }
      }
   }

   return allowRestart;
}


void RichTextMarqueeWidget2D::ChangeState(State state)
{
   m_state = state;
   m_ticks = FeatStd::Internal::Ticks(FeatStd::Internal::TicksType::Now);
   Invalidate();
   InvalidateLayout();
   //printf("State: %s\n", Disabled ? "Disabled" : WaitStart ? "WaitStart" : Running ? "Running" : WaitEnd ? "WaitEnd" : "?");
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
