/* ***************************************************************************************
* FILE:          RtRenderer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtRenderer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtTruncationBitmapTextRenderContext.h"
#include <Widgets/2D/RichText/DocumentModel/RtText.h>
#include <Widgets/2D/RichText/LiteHtml/TruncationTextContext.h>

namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera::TextRendering;


TruncationBitmapTextRenderContext::TruncationBitmapTextRenderContext(const Candera::TextRendering::TextRenderContext* referenceContext /*= 0*/) :
   Base(referenceContext),
   m_truncationRect(0.0F, 0.0F, -1.0F, -1.0F),
   m_text(0),
   m_ellipsisContext(0),
   m_truncation(NoTruncation)
{
}


void TruncationBitmapTextRenderContext::SetTruncation(bool truncate, const Candera::Rectangle& rect, const TruncationTextContext* ellipsisContext, BidiBaseLevel::Enum bidiLevel)
{
   m_truncation = NoTruncation;
   if (truncate)
   {
      if (bidiLevel == BidiBaseLevel::RightToLeft)
      {
         m_truncation = LengthCheckRightToLeft;
      }
      else
      {
         m_truncation = LengthCheckLeftToRight;
      }

      m_truncationRect = rect;
      m_ellipsisContext = ellipsisContext;
   }
}


bool TruncationBitmapTextRenderContext::PrepareBlit(FeatStd::Int16 x, FeatStd::Int16 /* y */, const Candera::TextRendering::GlyphBitmap& glyph)
{
   bool blit = true;

   switch (m_truncation)
   {
      case NoTruncation:
         break;

      case LengthCheckLeftToRight:
      {
         FeatStd::Float ellipsisWidth = (m_ellipsisContext != 0) ? FeatStd::Float(m_ellipsisContext->GetTextWidth()) : 0.0F;

         if (FeatStd::Float(x + glyph.xadvance) > m_truncationRect.GetLeft() + m_truncationRect.GetWidth() - ellipsisWidth)
         {
            if (m_ellipsisContext != 0)
            {
               m_truncation = BlitEllipsis;
               LayoutingOptions layoutingOptions;
               layoutingOptions.SetOffset(TextCoordinate(x, PixelPosition(m_truncationRect.GetTop())));
               (void)TextRenderer().Render(*this, layoutingOptions, ShapingOptions(m_style), TextProperties(m_ellipsisContext->GetIterator()));
            }
            blit = false;
            m_truncation = SkipRemainingGlyphs;
         }
         break;
      }

      case LengthCheckRightToLeft:
      {
         FeatStd::Float ellipsisWidth = (m_ellipsisContext != 0) ? FeatStd::Float(m_ellipsisContext->GetTextWidth()) : 0.0F;
         bool ellipsis = false;
         FeatStd::Float ellipsisX = 0.0F;
         if (glyph.characterPosition == 0)
         {
            ellipsis = true;
            if (FeatStd::Float(x) - ellipsisWidth >= 0)
            {
               ellipsisX = FeatStd::Float(x) - ellipsisWidth;
            }
            else
            {
               blit = false;
               ellipsisX = FeatStd::Float(0);
            }
         }
         else if (FeatStd::Float(x) - ellipsisWidth < 0)
         {
            blit = false;
         }
         else
         {
            ellipsis = true;
            ellipsisX = FeatStd::Float(x) - ellipsisWidth;
         }

         if (ellipsis && m_ellipsisContext != 0)
         {
            m_truncation = BlitEllipsis;
            LayoutingOptions layoutingOptions;
            layoutingOptions.SetOffset(TextCoordinate(PixelPosition(ellipsisX), PixelPosition(m_truncationRect.GetTop())));
            (void)TextRenderer().Render(*this, layoutingOptions, ShapingOptions(m_style), TextProperties(m_ellipsisContext->GetIterator()));
            m_truncation = NoTruncation;
         }
         break;
      }

      case BlitEllipsis:
         break;

      case SkipRemainingGlyphs:
      default:
         blit = false;
         break;
   }
   return blit;
}


void TruncationBitmapTextRenderContext::Blit(FeatStd::Int16 x, FeatStd::Int16 y, const Candera::TextRendering::GlyphBitmap& glyph)
{
   bool blit = PrepareBlit(x, y, glyph);

   if (blit)
   {
      Candera::Color color = m_color;
      if (m_text != 0)
      {
         const Text::HighlightData::SharedPointer& highlightData = m_text->GetHighlightData();

         if (highlightData != 0)
         {
            TruncationTextContext& context = static_cast<TruncationTextContext&>(*(m_text->GetContext()));
            if (highlightData->IsInRange(context.GetCodePointIndex(glyph.characterPosition)))
            {
               color = highlightData->m_textColor;
            }
         }
      }
      SetPenColor(color);
      Base::Blit(x, y, glyph);
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
