/* ***************************************************************************************
* FILE:          RtRenderBufferCache.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtRenderBufferCache is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtRenderBufferCache.h"

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::UInt32;
using FeatStd::Internal::CriticalSectionLocker;

RenderBufferCache::RenderBufferCache() :
   m_renderBuffers(),
   m_nodeUpdateRequestPending(false),
   m_cs()
{
}


RenderBufferCache::~RenderBufferCache()
{
   Clear();
}


UInt32 RenderBufferCache::GetCount()
{
   CriticalSectionLocker locker(&m_cs);
   return m_renderBuffers.Size();
}


RenderBufferCacheItem::SharedPointer RenderBufferCache::Get(UInt32 idx)
{
   RenderBufferCacheItem::SharedPointer cacheItem;
   CriticalSectionLocker locker(&m_cs);

   if (m_renderBuffers.Size() > idx)
   {
      cacheItem = m_renderBuffers[idx];
   }

   return cacheItem;
}


bool RenderBufferCache::TestAndGet(FeatStd::UInt32 idx, RenderBufferCacheItem::SharedPointer& itemPtr)
{
   CriticalSectionLocker locker(&m_cs);
   bool rc = false;
   if (m_renderBuffers.Size() > idx)
   {
      itemPtr = m_renderBuffers[idx];
      rc = true;
   }
   return rc;
}


void RenderBufferCache::Add(const Candera::Bitmap::SharedPointer& renderBufferBitmap)
{
   CriticalSectionLocker locker(&m_cs);
   RenderBufferCacheItem::SharedPointer itemPtr = RenderBufferCacheItem::Create(renderBufferBitmap);
   m_renderBuffers.Add(itemPtr);
}


void RenderBufferCache::Clear()
{
   CriticalSectionLocker locker(&m_cs);
   m_renderBuffers.Clear();
}


bool RenderBufferCache::TestAndSetRenderNodeUpdateRequestPending()
{
   CriticalSectionLocker locker(&m_cs);
   bool isSet = m_nodeUpdateRequestPending;
   m_nodeUpdateRequestPending = true;
   return isSet;
}


void RenderBufferCache::ClearRenderNodeUpdateRequestPending()
{
   CriticalSectionLocker locker(&m_cs);
   m_nodeUpdateRequestPending = false;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
