/* ***************************************************************************************
* FILE:          RtDecorationRenderTraverser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDecorationRenderTraverser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtDecorationRenderTraverser.h"
#include "RtRenderer.h"
#include <Candera/Engine2D/Core/RenderNode.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h>
#include <Widgets/2D/RichText/ContentProvider/RtContentProvider.h>
#include <Widgets/2D/RichText/DocumentModel/RtImage.h>
#include <Widgets/2D/RichText/DocumentModel/RtSpace.h>
#include <Widgets/2D/RichText/Engine/RtEngine.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RtDecorationRenderTraverser.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera;
using namespace Candera::TextRendering;


DecorationRenderTraverser::DecorationRenderTraverser(Engine& engine, Candera::Node2D& node) :
   Base(engine),
   m_node(node)
{
   Node2D* child = node.GetFirstChild();
   while (child != 0)
   {
      Node2D* nextSibling = child->GetNextSibling();
      (void)node.RemoveChild(child);
      child->Dispose();
      child = nextSibling;
   }
}


void DecorationRenderTraverser::OnProcess(const DocElement& docElement, const Candera::Rectangle& /*effectiveRect*/)
{
   Color color;
   (void)docElement.GetStyleParameter(StyleParameterBase::BackgroundColor, color);

   // background boxes already have absolute position set by in DocumentContainer::draw_background, so an empty rectangle is passed.
   DrawBackgroundBoxes(docElement.GetBackgroundBoxList(), color, Candera::Rectangle());
}


bool DecorationRenderTraverser::ProcessSpace(const Space& space, const Candera::Rectangle& effectiveRect)
{
   DrawHighlightBoxes(space.GetHighlightData(), effectiveRect);
   return false;
}


bool DecorationRenderTraverser::ProcessText(const Text& text, const Candera::Rectangle& effectiveRect)
{
   DrawHighlightBoxes(text.GetHighlightData(), effectiveRect);
   return false;
}


bool DecorationRenderTraverser::ProcessImage(const Image& image, const Candera::Rectangle& effectiveRect)
{
   const FeatStd::Char* imageName = image.GetName().GetCString();
   Bitmap::SharedPointer bitmap = ContentProvider::GetInstance().GetBitmap(0, imageName);
   if (bitmap != 0)
   {
      RenderNode* node = RenderNode::Create();
      if (node != 0)
      {
         node->SetName(imageName);
         BitmapBrushBlend::SharedPointer brush = BitmapBrushBlend::Create();
         if (brush != 0)
         {
            BitmapImage2D::SharedPointer bitmapImage = BitmapImage2D::Create();
            if (bitmapImage != 0)
            {
               RichTextRenderer::SetAlphaBlendFactor(brush->GetBlendEffect());

               if (bitmapImage->SetBitmap(bitmap))
               {
                  brush->GetBitmapBrush().Image().Set(bitmapImage);

                  Candera::Vector2 effectivePos(effectiveRect.GetPosition());
                  node->SetPosition(effectivePos);
                  if (node->AddEffect(brush.GetPointerToSharedInstance()))
                  {
                     if (GetEngine().GetData().m_asynchronousRendering)
                     {
                        UploadNodeAsync(node);
                     }
                     else
                     {
                        UploadNode(&m_node, node);
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessImage: node->AddEffect() failed."));
                  }
               }
               else
               {
                  ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessImage: bitmapImage->SetBitmap() failed."));
               }
            }
         }
      }
   }

   return false;
}


void DecorationRenderTraverser::UploadNodeAsync(Candera::Node2D* node)
{
   UploadRequest::SharedPointer uploadRequest = UploadRequest::SharedPointer(FEATSTD_NEW(UploadRequest)(&m_node, node));
   RichTextRenderer::GetUpoadDispatcher().Schedule(uploadRequest);
}


void DecorationRenderTraverser::UploadNode(Candera::Node2D* parentNode, Candera::Node2D* node)
{
   if (node->Upload() && (parentNode != 0))
   {
      if (!parentNode->AddChild(node))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessImage: m_node.AddChild() failed."));
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessImage: node->Upload() failed."));
   }
}


void DecorationRenderTraverser::DrawHighlightBoxes(const Text::HighlightData::SharedPointer& highlightData,
      const Candera::Rectangle& effectiveRect)
{
   if (highlightData != 0)
   {
      DrawBackgroundBoxes(highlightData->m_boxList, highlightData->m_backgroundColor, effectiveRect);
   }
}


void DecorationRenderTraverser::DrawBackgroundBoxes(const DocElement::BoxList& boxList, const Candera::Color& color,
      const Candera::Rectangle& effectiveRect)
{
   if (!boxList.Empty())
   {
      for (DocElement::BoxList::ConstIterator it(boxList.ConstBegin()); it != boxList.ConstEnd(); ++it)
      {
         RenderNode* node = RenderNode::Create();
         if (node != 0)
         {
            SolidColorBrushBlend::SharedPointer brush = SolidColorBrushBlend::Create();
            if (brush != 0)
            {
               SolidColorBrush& solidColorBrush = brush->GetSolidColorBrush();
               solidColorBrush.Color().Set(color);
               solidColorBrush.Size().Set((*it).GetSize());

               RichTextRenderer::SetAlphaBlendFactor(brush->GetBlendEffect());

               Vector2 pos((*it).GetPosition());
               pos += effectiveRect.GetPosition();
               node->SetPosition(pos);
               node->SetName("RT_BackgroundBox");
               if (node->AddEffect(brush.GetPointerToSharedInstance()))
               {
                  if (node->Upload())
                  {
                     if (!m_node.AddChild(node))
                     {
                        ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DrawBackgroundBoxes: node->AddChild() failed."));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DrawBackgroundBoxes: node->Upload() failed."));
                  }
               }
               else
               {
                  ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DrawBackgroundBoxes: node->AddEffect() failed."));
               }
            }
         }
      }
   }
}


void DecorationRenderTraverser::UploadRequest::Execute()
{
   DecorationRenderTraverser::UploadNode(m_parentNode, m_node);
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
