/* ***************************************************************************************
* FILE:          TruncationTextContext.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TruncationTextContext is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "TruncationTextContext.h"
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>
#include <Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h>

namespace hmibase {
namespace widget {
namespace richtext {

TruncationTextContext::TruncateGlyphData::TruncateGlyphData() :
   m_direction(Candera::TextRendering::GlyphBitmap::Implicit),
   m_fontIdentifier(),
   m_glyphIndex(),
   m_characterPosition(),
   m_position(),
   m_advance(),
   m_rect()
{
}


TruncationTextContext::SharedPointer TruncationTextContext::Create(const Candera::TextRendering::TextRenderContext* reference)
{
   return SharedPointer(FEATSTD_NEW(TruncationTextContext)(reference));
}


hmibase::widget::richtext::TruncationTextContext::TruncateGlyphDataIterator TruncationTextContext::GetTruncateGlyphDataIterator() const
{
   return TruncateGlyphDataIterator(m_container.ConstBegin(), m_container.ConstEnd());
}


void TruncationTextContext::ApplyCodePointMap(const FeatStd::TChar* text)
{
   m_codePointMap.Clear();

   Candera::TextRendering::Internal::FullBoundCodePointIterator codePointItor(text, -1);

   Candera::TextRendering::TextLength len = 0;
   Candera::TextRendering::TextLength sumLen = 0;
   do
   {
      m_codePointMap.Add(sumLen);
      len = codePointItor.Advance();
      sumLen += len;
   }
   while (len != 0);
}


FeatStd::UInt32 TruncationTextContext::GetCodePointIndex(Candera::TextRendering::TextPosition characterPosition) const
{
   FeatStd::UInt32 i = 0;
   for (; i < m_codePointMap.Size(); i++)
   {
      if (m_codePointMap[i] == characterPosition)
      {
         break;
      }
   }

   return i;
}


Candera::TextRendering::TextPosition TruncationTextContext::GetCharacterPosition(FeatStd::UInt32 codePointIndex) const
{
   Candera::TextRendering::TextPosition pos = -1;
   if (codePointIndex < m_codePointMap.Size())
   {
      pos = m_codePointMap[codePointIndex];
   }
   return pos;
}


TruncationTextContext::TruncationTextContext(const Candera::TextRendering::TextRenderContext* reference) :
   Base(reference),
   m_textWidth(0)
{
}


void TruncationTextContext::Measure(Candera::TextRendering::PixelPosition x, Candera::TextRendering::PixelPosition y,
                                    const Candera::TextRendering::GlyphBitmap& glyph)
{
   TruncateGlyphData storedGlyphData;

   storedGlyphData.m_fontIdentifier = glyph.fontIdentifier;
   storedGlyphData.m_glyphIndex = glyph.glyphIndex;
   storedGlyphData.m_characterPosition = glyph.characterPosition;
   storedGlyphData.m_position.x = x;
   storedGlyphData.m_position.y = y;
   storedGlyphData.m_advance.width = glyph.xadvance;
   storedGlyphData.m_advance.height = glyph.yadvance;
   storedGlyphData.m_rect.SetLeft(glyph.left);
   storedGlyphData.m_rect.SetWidth(glyph.width);
   storedGlyphData.m_rect.SetTop(glyph.top);
   storedGlyphData.m_rect.SetHeight(glyph.height);
   storedGlyphData.m_direction = glyph.direction;

   static_cast<void>(m_container.Add(storedGlyphData));

   m_textWidth = x + FeatStd::Int(storedGlyphData.m_advance.width) - glyph.left;
}


Candera::TextRendering::PreprocessingContext::ReferenceIterator* TruncationTextContext::GetReferenceIterator() const
{
   return Candera::TextRendering::Internal::PreprocessingContextVectorIterator<DataContainer, PreprocessingContext::ReferenceIterator>
          (m_container.ConstBegin(), m_container.ConstEnd()).Clone();
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
