﻿/* ***************************************************************************************
* FILE:          LiteHtmlContext.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LiteHtmlContext is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "LiteHtmlContext.h"
#include <FeatStd/Platform/Thread.h>
#include <FeatStd/Util/BinarySearch.h>
#include <FeatStd/Util/Hash.h>
#include <Widgets/2D/RichText/ContentProvider/RtContentProvider.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/LiteHtmlContext.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {


LiteHtmlContext::List& LiteHtmlContext::GetList()
{
   FEATSTD_SYNCED_STATIC_OBJECT(List, s_contextList);
   return s_contextList;
}


litehtml::context* LiteHtmlContext::GetContext(Url styleSheetUrl)
{
   static FeatStd::Internal::CriticalSection cs;
   FeatStd::Internal::CriticalSectionLocker locker(&cs);

   litehtml::context* litehtmlContext = 0;
   LiteHtmlContext* context = 0;
   List& list = GetList();

   FeatStd::UInt32 threadId = FeatStd::Internal::Thread::GetCurrentId();

   Key key(threadId, styleSheetUrl);
   const FeatStd::SizeType itemCount = list.Size();
   FeatStd::SizeType insertAt;
   FeatStd::SizeType foundAt = FeatStd::Internal::BinarySearch<LiteHtmlContext::Key>(key, list, itemCount, &insertAt);

   if (foundAt < itemCount)
   {
      litehtmlContext = &list[foundAt]->m_context;
   }
   else
   {
      context = FEATSTD_NEW(LiteHtmlContext)(threadId, styleSheetUrl);
   }

   if (context != 0)
   {
      if (list.Insert(insertAt, SharedPointer(context)))
      {
         litehtmlContext = &context->m_context;
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "LiteHtmlContext::GetContext: Inserting new context in the list failed."));
         FEATSTD_DELETE(context);
      }
   }

   return litehtmlContext;	/* PRQA S 4629 */
}


LiteHtmlContext::LiteHtmlContext(FeatStd::UInt32 threadId, Url styleSheetUrl) :
   m_key(threadId, styleSheetUrl)
{
   TextInStream::SharedPointer css = ContentProvider::GetInstance().GetTextResource(0, styleSheetUrl);

   if (css != 0)
   {
      m_context.load_master_stylesheet(css->ReadCompleteText().GetCString());
   }
   else
   {
      // minimal setting to see white text on black background in Scene Composer
      static const FeatStd::TChar* minimalCss =
         "html {"
         "  color:white;"
         "  white-space:pre;"
         "  text-overflow:ellipsis"
         "}"
         "body {"
         "  display:block"
         "}"
         "br {"
         "  display:inline-block"
         "}";
      m_context.load_master_stylesheet(minimalCss);
   }
}


LiteHtmlContext::Key::Key(FeatStd::UInt32 threadId, Url styleSheetUrl) :
   m_threadId(threadId),
   m_styleSheetUrlHash(FeatStd::Hash::CalcHash(styleSheetUrl))
{
}


template <typename T1, typename T2>
FeatStd::Int Compare2Values(const T1& left1, const T1& right1, const T2& left2, const T2& right2)
{
   return
      (left1 < right1) || ((left1 == right1) && (left2 < right2)) ? -1 :
      (left1 == right1) && (left2 == right2) ? 0 :
      +1;
}


FeatStd::Int LiteHtmlContext::Key::Compare(const Key& left, const LiteHtmlContext::SharedPointer& right)
{
   return Compare2Values(left.m_threadId, right->m_key.m_threadId, left.m_styleSheetUrlHash, right->m_key.m_styleSheetUrlHash);
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
