/* ***************************************************************************************
* FILE:          RtCaretBasedDocManipulator.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtCaretBasedDocManipulator is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/Util/String.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>
#include <Widgets/2D/RichText/Engine/RtCaret.h>
#include <Widgets/2D/RichText/Engine/RtDocUnits.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>
#include <Widgets/2D/RichText/StyleProvider/RtElementStyle.h>

namespace hmibase {
namespace widget {
namespace richtext {

class Engine;

/**
    This class provides functionality to control input caret and
    to manipulate the associated document according to user input.
 */
class CaretBasedDocManipulator: public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);

   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(CaretBasedDocManipulator);

      /** Destructor */
      virtual ~CaretBasedDocManipulator();

      /**
       * Provides the Caret instance associated with this manipulator.
       *
       * @return The smart reference to the associated Caret instance.
       * A Caret instance is always available.
       */
      virtual Caret::SharedPointer GetCaret() = 0;

      /**
       * Provides the paragraph item object of the document on which the
       * Caret instance is currently located.
       *
       * @return The reference to the currently focused paragraph item.
       */
      virtual ParagraphItem::SharedPointer GetItemAtCaret() = 0;

      /**
       * Places the Caret on and inside the paragraph item of the underlying
       * document that matches the position relative to the upper left
       * corner of the document. The on-screen position is automatically
       * converted into document-relative position if parameter
       * 'screenCoordinates' is set to 'true'.
       * The Caret is placed to the next appropriate
       * position between two document units. Because the size of document
       * units define discrete positions for Carets, the actually applied
       * position of the Caret is typically slightly deviating from the
       * requested position.
       * The Caret position is clipped to the content of the document.
       *
       * @note: On-screen coordinates must be converted to document-related
       * coordinates using conversion functions provided by Viewport class
       * if the parameter 'screenCoordinates' is set to false.
       *
       * @param posX The value of the horizontal coordinate in pixel
       *             on which the Caret shall be placed closely.
       * @param posY The value of the vertical coordinate in pixel
       *             on which the Caret shall be placed closely.
       * @param screenCoordinates True (default) if posX and posY
       *             are provided as on-screen coordinates, false if they
       *             are provided relative to the upper left corner of the
       *             document.
       */
      virtual void PlaceCaret(FeatStd::Int32 posX, FeatStd::Int32 posY, bool screenCoordinates) = 0;

      /**
       * Places the Caret in the same way PlaceCaret(UInt32, UInt32, bool) but
       * uses a Vector2 instead of discrete integer values for the coordinates.
       *
       * @param position The requested position of the Caret in pixel.
       * @param screenCoordinates True (default) if position
       *             is provided as on-screen coordinates, false if it
       *             is provided relative to the upper left corner of the
       *             document.
       */
      virtual void PlaceCaret(const Candera::Vector2& position, bool screenCoordinates) = 0;

      /**
       * Provides the position of the Caret either in on-screen coordinates
       * or relative to the upper left corner of the document.
       *
       * @note This is a convenience method for Caret::GetPosition().
       *
       * @param screenCoordinates True (default) to provide the position of the
       *                          Caret relative to the node of the Caret widget
       *                          or false to get it relative to the upper left
       *                          corner of the document.
       * @return The position of the Caret according to 'screenCoordinates' parameter.
       */
      virtual Candera::Vector2 GetCaretPosition(bool screenCoordinates) = 0;

      /**
       * Sets the Caret before the first document unit (e.g. character) of the
       * document. This results in the same position as 'PlaceCaretAtEnd()' for
       * empty documents.
       */
      virtual void PlaceCaretAtBegin() = 0;

      /**
       * Sets the Caret after the last document unit (e.g. character) of the
       * document. This results in the same position as 'PlaceCaretAtBegin()' for
       * empty documents.
       */
      virtual void PlaceCaretAtEnd() = 0;

      /**
       * Moves the Caret for the indicated distance within the document. The
       * movement is clipped to the content of the document.

       * @param count The distance to move the Caret. Positive values moves
       *              it towards the end, negative values towards the begin
       *              of the document.
       * @param unit The unit of the distance to move the Caret. This could be
       *             e.g. a single character, image, item, or a paragraph or
       *             chapter.
       */
      virtual void MoveCaret(FeatStd::Int32 count, DocUnits::Enum unit) = 0;

      /**
       * Inserts the given, non-empty string before the current position
       * of the Caret. Empty texts are silently ignored.
       *
       * @param text The text to insert.
       */
      virtual void InsertBeforeCaret(const FeatStd::String& text) = 0;

      /**
       * Inserts the given, non-empty string before the current position
       * of the Caret and attaches the given style to that text.
       * Empty texts are silently ignored.
       *
       * @param text The text to insert.
       * @param style The style to apply to the text.
       */
      virtual void InsertBeforeCaret(const FeatStd::String& text, const ElementStyle::SharedPointer style) = 0;

      /**
       * Inserts the given, non-empty string after the current position
       * of the Caret. Empty texts are silently ignored.
       *
       * @param text The text to insert.
       */
      virtual void InsertAfterCaret(const FeatStd::String& text) = 0;

      /**
       * Inserts the given, non-empty string after the current position
       * of the Caret and attaches the given style to that text.
       * Empty texts are silently ignored.
       *
       * @param text The text to insert.
       * @param style The style to apply to the text.
       */
      virtual void InsertAfterCaret(const FeatStd::String& text, const ElementStyle::SharedPointer style) = 0;

      /**
       * Removes the given number of atomic document units (e.g. characters and images)
       * before the current position of the Caret.

       * @param count The number of atomic document units to delete.
       */
      virtual void DeleteBeforeCaret(FeatStd::UInt32 count) = 0;

      /**
       * Removes the given number of atomic document units (e.g. characters and images)
       * after the current position of the Caret.

       * @param count The number of atomic document units to delete.
       */
      virtual void DeleteAfterCaret(FeatStd::UInt32 count) = 0;

   protected:
      // Restricted constructor
      CaretBasedDocManipulator();

   private:
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
