/* ***************************************************************************************
* FILE:          RtCaret.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtCaret is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/Mathematics/Vector2.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>
#include <Widgets/2D/RichText/DocumentModel/RtParagraphItem.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
    This class represents caret metrics in relation to the affected document. The
    caret is located within the document where (interactive) editing occurs. It defines the position of the caret within the document by associating it with
    the affected paragraph item and the offset within the item.
    This class does not define any appearance attributes of the caret presentation.
  */
class Caret: public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Caret);

      /** Destructor */
      virtual ~Caret();

      /**
       * Defines whether or not the caret shall be visualized.
       *
       * @param visible True to show the caret, false to hide it.
       */
      virtual void SetVisiblity(bool visible) = 0;

      /**
       * Determines whether or not the caret shall be visualized.
       *
       * @return True if the caret is visible, false otherwise.
       */
      virtual bool GetVisiblity() const = 0;

      /**
       * Provides the caret screen position in pixels relative to the top-left corner of
       * the document.
       *
       * @return The position of the caret regardless whether or not caret is visible.
       * @param screenCoordinates True (default) if posX and posY
       *             are provided as on-screen coordinates, false if they
       *             are provided relative to the upper left corner of the
       *             document.
       * @return The position of the Caret according to 'screenCoordinates' parameter.
       */
      virtual Candera::Vector2 GetPosition(bool screenCoordinates) = 0;

      /**
       * Defines the screen position of the caret within the document in pixels either relative
       * to the top-left corner of the document or relative to the widget of the
       * document. The Caret position is clipped to the content of the document.
       *
       * @param position The position in pixels to set.
       * @param screenCoordinates True (default) if position
       *             is provided as on-screen coordinates, false if it
       *             is provided relative to the upper left corner of the
       *             document.
       */
      virtual void SetPosition(const Candera::Vector2& position, bool screenCoordinates) = 0;

      /**
       * Defines the position of the caret within the document by associating it with
       * the affected paragraph item and the offset within the item.
       *
       * @param item The paragraph item the caret shall be placed to
       * @param offset The offset in document units (typically characters) of the caret position
       *               relative to the start of the associated paragraph item.
       */
      virtual void SetPosition(ParagraphItem::SharedPointer item, FeatStd::UInt32 offset) = 0;

      /**
       * Provides the paragraph item of the document in which the caret is located or a
       * null-object if the caret is not set or the document is empty. Note that in
       * general the caret shall be positioned at the beginning of the next item instead
       * of the end of the previous item, if applicable.
       *
       * @return The paragraph item affected by the Caret.
       */
      virtual ParagraphItem::SharedPointer GetAffectedParagraphItem() = 0;

      /**
       * Provides the position of the caret within the affected item.
       *
       * @return The offset in atomic document units of the caret relative to the begin of the affected paragraph item.
       */
      virtual FeatStd::UInt32 GetAffectedParagraphItemOffset() = 0;

   protected:
      ///Constructor for restricted construction.
      Caret();

   private:
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
