/* ***************************************************************************************
* FILE:          RtViewportImpl.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtViewportImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Widgets/2D/RichText/Engine/RtViewport.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * Implementation of the Viewport interface.
 * This class is not intended being used by clients directly.
 */
class ViewportImpl : public Viewport
{
      FEATSTD_TYPEDEF_BASE(Viewport);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(ViewportImpl);

      static ViewportImpl::SharedPointer Create(Engine* engine);

      virtual ~ViewportImpl();

      virtual void EnableRendering(bool enable) override;
      virtual bool IsRenderingEnabled() const override;

      virtual void SetSize(FeatStd::UInt16 width, FeatStd::UInt16 height) override;
      virtual void SetSize(const Candera::Vector2& size) override;
      virtual const Candera::Vector2& GetSize() const override;

      virtual void SetScaleFactor(FeatStd::Float scaleFactorX, FeatStd::Float scaleFactorY) override;
      virtual void SetScaleFactor(const Candera::Vector2& scaleFactor) override;
      virtual Candera::Vector2 GetScaleFactor() const override;

      virtual Candera::Vector2 ToDocumentPosition(const Candera::Vector2& screenPosition) const override;
      virtual Candera::Vector2 ToViewportPosition(const Candera::Vector2& documentPosition) const override;

      virtual void SetOffset(FeatStd::Float offsetX, FeatStd::Float offsetY) override;
      virtual void SetOffset(const Candera::Vector2& offset) override;
      virtual Candera::Vector2 GetOffset() const override;

   protected:
      ViewportImpl();

   private:
      bool m_renderingEnabled;      // Whether or not rendering of the document is enabled
      Candera::Vector2 m_size;      // The size of the view port defined by presentation.
      Candera::Vector2 m_offset;
};


inline
void ViewportImpl::SetSize(FeatStd::UInt16 width, FeatStd::UInt16 height)
{
   SetSize(Candera::Vector2(FeatStd::Float(width), FeatStd::Float(height)));
}


inline
void ViewportImpl::SetOffset(FeatStd::Float offsetX, FeatStd::Float offsetY)
{
   SetOffset(Candera::Vector2(offsetX, offsetY));
}


inline
void ViewportImpl::SetOffset(const Candera::Vector2& offset)
{
   m_offset = offset;
}


inline
Candera::Vector2 ViewportImpl::GetOffset() const
{
   return m_offset;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
