/* ***************************************************************************************
* FILE:          RtHighlighterImpl.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtHighlighterImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtHighlighter.h>
#include <Widgets/2D/RichText/Engine/RtMarker.h>
#include <Widgets/2D/RichText/Engine/RtDocRange.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/DocumentModel/RtText.h>

namespace hmibase {
namespace widget {
namespace richtext {

class Engine;

/**
 * Implementation of the Highlighter interface.
 * This class is not intended being used by clients directly.
 */
class HighlighterImpl: public Highlighter
{
      FEATSTD_TYPEDEF_BASE(Highlighter);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(HighlighterImpl);

      static HighlighterImpl::SharedPointer Create(Engine* engine);
      virtual ~HighlighterImpl();

      virtual void MarkAll(const FeatStd::String& searchCriteria, ElementStyle::SharedPointer highlightStyle) override;
      virtual void MarkRange(const FeatStd::String& range, ElementStyle::SharedPointer highlightStyle) override;
      virtual void MarkText(const FeatStd::String& searchText, FeatStd::UInt32 maxOccurencies, bool caseSensitive, ElementStyle::SharedPointer highlightStyle) override;
      virtual void ClearAll() override;
      virtual void Update() override;

   protected:
      HighlighterImpl();

   private:

      // A single index-based highlight range criteria
      struct SearchIndex
      {
         FeatStd::UInt32 m_startIdx;
         FeatStd::UInt32 m_endIdx;

         SearchIndex(FeatStd::UInt32 startIdx, FeatStd::UInt32 endIdx);
         bool IsRange() const;
      };

      // A single text-mask based highlight range criteria
      struct SearchText
      {
         FeatStd::String m_mask;
         FeatStd::UInt32 m_maxOccurencies;
         FeatStd::UInt32 m_occurencies;
         bool m_caseSensitive;

         SearchText(FeatStd::String mask, FeatStd::UInt32 maxOccurencies, bool caseSensitive);
      };

      // Get access to protected setter methods.
      class HighlightDocRange : public DocRange
      {
         public:
            using DocRange::SetStart;
            using DocRange::SetEnd;
            using DocRange::AddElement;
      };

      // The list type for search indexes.
      typedef FeatStd::Internal::Vector<SearchIndex> SearchIndexList;
      // The list type for search texts.
      typedef FeatStd::Internal::Vector<SearchText> SearchTextList;

      // The traverser searching characters by their index or index-range
      class IndexSearcher : public DocElementTraverser
      {
            FEATSTD_TYPEDEF_BASE(DocElementTraverser);
         public:
            IndexSearcher(const SearchIndexList& searchList, const HighlighterImpl& highlighter);
            virtual ~IndexSearcher();

         protected:
            virtual bool Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect) override;

         private:
            struct MatchDetectorAttribs
            {
               bool m_isElemAdded;
               bool m_isStartInElem;
               FeatStd::UInt32 m_remainingPartCnt;
               FeatStd::UInt32 m_elemPartCnt;
            };

            void ProcessPotentialMatch(Text* textElem, MatchDetectorAttribs& attribs);

            /** Finalizes the match and returns the number of remaining unprocessed element parts. */
            FeatStd::UInt32 FinalizeMatch(const FeatStd::TChar* text, FeatStd::UInt32 end, FeatStd::UInt32 elemPartCnt);

            // The order list of index and index ranges to find
            const SearchIndexList& m_searchList;
            // The current entry index of the search list.
            FeatStd::SizeType m_curIndex;
            // The processed document character count.
            FeatStd::UInt32 m_processedCharCount;
            // The matching doc range
            HighlightDocRange* m_docRange;
            // The highlighter implementation to notify on hits.
            const HighlighterImpl& m_highlighter;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(IndexSearcher);
      };

      // The traverser searching characters by their index or index-range
      class TextSearcher : public DocElementTraverser
      {
            FEATSTD_TYPEDEF_BASE(DocElementTraverser);
         public:
            TextSearcher(SearchTextList& searchList, const HighlighterImpl& highlighter);
            virtual ~TextSearcher();

         protected:
            virtual bool Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect) override;

         private:
            // The order list of index and index ranges to find
            SearchTextList& m_searchList;
            // The matching doc range
            HighlightDocRange* m_docRange;
            // The highlighter implementation to notify on hits.
            const HighlighterImpl& m_highlighter;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(TextSearcher);

            bool Match(Candera::TextRendering::CodePoint c1, Candera::TextRendering::CodePoint c2, bool caseSensitive) const;
      };

      // Invoked by search traverser upon match.
      void OnMatch(DocRange::SharedPointer match) const;

      struct IndexSearchAttribs
      {
         FeatStd::UInt32 m_startIdx;
         FeatStd::UInt32 m_endIdx;
         bool m_isStartIdxPass;
         bool m_isEndIdxPass;

         IndexSearchAttribs(FeatStd::UInt32 startIdx, FeatStd::UInt32 endIdx, bool isStartIdxPass, bool isEndIdxPass) :
            m_startIdx(startIdx),
            m_endIdx(endIdx),
            m_isStartIdxPass(isStartIdxPass),
            m_isEndIdxPass(isEndIdxPass)
         {}
      };

      // finalizes the creation of a search index and appends it to the given list.
      void FinalizeSearchIndex(SearchIndexList& searchList, IndexSearchAttribs& attribs) const;
      // finalizes the creation of a search text and appends it to the given list.
      void FinalizeSearchIndex(SearchTextList& searchList, FeatStd::TChar* textMask, FeatStd::SizeType& textMaskLength) const;
      // Performs the search on the document
      void DoSearch(const SearchIndexList* searchIndexList, SearchTextList* searchTextList);
      // Process criteria string
      void ProcessCombinedCriteriaText(const FeatStd::String& searchCriteria, SearchIndexList& searchIndexList, SearchTextList& searchTextList) ;
      // Calculates the index search attributes
      void CalcIndexSearchAttribs(FeatStd::TChar criterion, IndexSearchAttribs& searchAttribs);
      // Applies the highlight appearance
      void ApplyHighlighting(Text::HighlightData::SharedPointer highlightData, FeatStd::UInt32 start, FeatStd::UInt32 end, const Text* text) const;

      // The style for highlight presentation.
      ElementStyle::SharedPointer m_highlightStyle;

      FEATSTD_MAKE_CLASS_UNCOPYABLE(HighlighterImpl);
};


inline
HighlighterImpl::SearchIndex::SearchIndex(FeatStd::UInt32 startIdx, FeatStd::UInt32 endIdx) :
   m_startIdx(startIdx),
   m_endIdx(endIdx)
{
}


inline
bool HighlighterImpl::SearchIndex::IsRange() const
{
   return (m_endIdx != m_startIdx);
}


inline
HighlighterImpl::SearchText::SearchText(FeatStd::String mask, FeatStd::UInt32 maxOccurencies, bool caseSensitive) :
   m_mask(mask),
   m_maxOccurencies(maxOccurencies),
   m_occurencies(0),
   m_caseSensitive(caseSensitive)
{
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
