/* ***************************************************************************************
* FILE:          RtDocAccessorImpl.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocAccessorImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtDocAccessorImpl.h"

#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Vector2.h>

#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElementTraverser.h>

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::UInt8;
using FeatStd::UInt16;
using FeatStd::Int32;
using FeatStd::UInt32;
using FeatStd::Float;

using Candera::Rectangle;
using Candera::Vector2;


DocAccessorImpl::ElementLocator::ElementLocator(const Vector2& targetPoint) :
   Base(),
   m_targetPoint(targetPoint),
   m_hitRect(),
   m_hitElement()
{
}


bool DocAccessorImpl::ElementLocator::Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect)
{
   if (effectiveRect.Contains(m_targetPoint))
   {
      // hit test passed
      if ((m_hitElement == 0)
            || ((effectiveRect.GetLeft() >= m_hitRect.GetLeft()) && (effectiveRect.GetTop() >= m_hitRect.GetTop())
                && (effectiveRect.GetWidth() < m_hitRect.GetWidth()) && (effectiveRect.GetHeight() < m_hitRect.GetHeight())))
      {
         // the bounding rect of a child element must be fully contained within the one of the parent element
         m_hitElement = DocElement::SharedPointer(const_cast<DocElement*>(&docElement));
         m_hitRect = effectiveRect;
      }
   }
   return false;
}


FEATSTD_RTTI_DEFINITION(DocAccessorImpl, Base);

DocAccessorImpl::SharedPointer DocAccessorImpl::Create(Engine* engine)
{
   DocAccessorImpl::SharedPointer sp = DocAccessorImpl::SharedPointer(FEATSTD_NEW(DocAccessorImpl));
   if (!sp.PointsToNull())
   {
      sp->AttachEngine(engine);
   }
   return sp;
}


DocAccessorImpl::DocAccessorImpl() :
   Base(),
   m_srcDocMarkup(Markup::Invalid)
{
}


DocAccessorImpl::~DocAccessorImpl()
{
}


void DocAccessorImpl::SetDocument(Markup::Enum markup, Document::SharedPointer document)
{
   m_srcDocMarkup = markup;
   m_document = document;
}


Document::SharedPointer DocAccessorImpl::GetDocument()
{
   return  m_document;
}


Chapter::SharedPointer DocAccessorImpl::GetOutline()
{
   Chapter::SharedPointer rootChapter;
   if (m_document != 0)
   {
      rootChapter = m_document->GetBody();
   }
   return rootChapter;
}


Chapter::SharedPointer DocAccessorImpl::GetChapter(DocElement::SharedPointer element)
{
   return FindInHierarchy<Chapter>(element);
}


Paragraph::SharedPointer DocAccessorImpl::GetParagraph(DocElement::SharedPointer element)
{
   return FindInHierarchy<Paragraph>(element);
}


DocElement::SharedPointer DocAccessorImpl::GetElementAt(Int32 posX, Int32 posY, bool screenCoordinates)
{
   return GetElementAt(Vector2(Float(posX), Float(posY)), screenCoordinates);
}


DocElement::SharedPointer DocAccessorImpl::GetElementAt(const Vector2& position, bool screenCoordinates)
{
   DocElement::SharedPointer hitElement;
   Vector2 targetPoint = position;
   if (screenCoordinates)
   {
      if ((0 != GetEngine()) && (!GetEngine()->GetViewport().PointsToNull()))
      {
         targetPoint = GetEngine()->GetViewport()->ToDocumentPosition(position);
      }
   }
   if (m_document != 0)
   {
      Rectangle docBounds = m_document->GetRect();
      if ((targetPoint.GetX() <= 0.0F) && (targetPoint.GetY() <= 0.0F))
      {
         // position is left and above of the document
         // -> set caret before the first character
         hitElement = m_document->GetFirstLeafParagraphItem();
      }
      else if ((targetPoint.GetX() >= docBounds.GetWidth()) && (targetPoint.GetY() >= docBounds.GetHeight()))
      {
         // position is right and below of the document
         // -> set caret at the end of the document
         hitElement = m_document->GetLastLeafParagraphItem();
      }
      else
      {
         // clip to document boundaries
         targetPoint.SetX(Candera::Math::Maximum(0.0F, Candera::Math::Minimum(docBounds.GetWidth(), targetPoint.GetX())));
         targetPoint.SetY(Candera::Math::Maximum(0.0F, Candera::Math::Minimum(docBounds.GetHeight(), targetPoint.GetY())));

         ElementLocator locator(targetPoint);
         (void)m_document->Process(locator, Candera::Rectangle());
         hitElement = locator.GetHitElement();
      }
   }
   return  hitElement;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
