/* ***************************************************************************************
* FILE:          RtCaretImpl.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtCaretImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtCaretImpl.h"
#include <Widgets/2D/RichText/Engine/RtEngine.h>

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::UInt8;
using FeatStd::UInt16;
using FeatStd::UInt32;
using Candera::Rectangle;
using Candera::Vector2;

FEATSTD_RTTI_DEFINITION(CaretImpl, Base);


CaretImpl::SharedPointer CaretImpl::Create(Engine* engine)
{
   CaretImpl::SharedPointer sp = CaretImpl::SharedPointer(FEATSTD_NEW(CaretImpl));
   if (!sp.PointsToNull())
   {
      sp->AttachEngine(engine);
   }
   return sp;
}


CaretImpl::CaretImpl() :
   Base(),
   m_visible(false),
   m_affectedItemOffset(0)
{
}


CaretImpl::~CaretImpl()
{
}


void CaretImpl::SetVisiblity(bool visible)
{
   if (m_visible != visible)
   {
      m_visible = visible;
   }
}


bool CaretImpl::GetVisiblity() const
{
   return m_visible;
}


Vector2 CaretImpl::GetPosition(bool screenCoordinates)
{
   Vector2 pos(m_position);
   if (screenCoordinates)
   {
      Engine* engine = GetEngine();
      if (0 != engine)
      {
         const Viewport::SharedPointer viewport = engine->GetViewport();
         if (viewport != 0)
         {
            pos = viewport->ToViewportPosition(pos);
         }
      }
   }
   return  pos;
}


void CaretImpl::SetPosition(const Vector2& position, bool screenCoordinates)
{
   Vector2 pos(position);
   if (screenCoordinates)
   {
      Engine* engine = GetEngine();
      if (engine != 0)
      {
         const Viewport::SharedPointer viewport = engine->GetViewport();
         if (viewport != 0)
         {
            pos = viewport->ToDocumentPosition(position);
         }
      }
   }
   if (m_position != pos)
   {
      m_position = pos;
      FindParagraphItemAt(m_position, screenCoordinates);
   }
}


void CaretImpl::SetPosition(const ParagraphItem::SharedPointer item, UInt32 offset)
{
   if ((item != 0) && (item != m_affectedItem) && (offset != m_affectedItemOffset))
   {
      Rectangle itemRect = item->GetRect();
      _TODO("Implement")
      //m_position.SetY(itemRect.GetTop() + item->GetBaselineHeight());
      //m_position.SetX(itemRect.GetLeft() + item->GetAtomWidth(offset));
      //m_affectedItemOffset = (offset > item->GetAtomCount());
   }
}


ParagraphItem::SharedPointer CaretImpl::GetAffectedParagraphItem()
{
   return  m_affectedItem;
}


UInt32 CaretImpl::GetAffectedParagraphItemOffset()
{
   return  m_affectedItemOffset;
}


void CaretImpl::FindParagraphItemAt(const Vector2& position, bool screenCoordinates)
{
   DocAccessor::SharedPointer docAccessor;

   Engine* engine = GetEngine();
   if (0 != engine)
   {
      docAccessor = engine->GetDocAccessor();
   }
   // note: flatend code due to coding guidlines requriement for nesting level.
   if (docAccessor != 0)
   {
      m_affectedItem = Candera::Dynamic_Cast<ParagraphItem::SharedPointer>(docAccessor->GetElementAt(position, screenCoordinates));
   }
   if ((docAccessor != 0) && (m_affectedItem != 0))
   {
      UInt16 startIdx = 0;
      UInt16 endIdx = m_affectedItem->GetPartCount();
      UInt16 curIdx = endIdx / 2;
      if (0 == endIdx)
      {
         m_affectedItemOffset = 0;
      }
      else
      {
         Rectangle curBound = m_affectedItem->GetPartBoundaries(curIdx);
         FeatStd::Float curLeft = curBound.GetLeft();
         FeatStd::Float curRight = curLeft + curBound.GetWidth();
         while ((curLeft > position.GetX()) || (curRight < position.GetX()))
         {
            // position is not within horizontal part boundary
            if (curLeft > position.GetX())
            {
               // current left boundary edge is right of position -> iterate to the left
               endIdx = curIdx;      // limit relevant area to the left side
            }
            else if (curRight < position.GetX())
            {
               // current right boundary edge is left of position -> iterate to the right
               startIdx = UInt16(curIdx + 1);   // limit relevant area to the right side
            }
            curIdx = UInt16(startIdx + endIdx) / 2;  // test inside the half of the relevant area
            curBound = m_affectedItem->GetPartBoundaries(curIdx);
            curLeft = curBound.GetLeft();
            curRight = curLeft + curBound.GetWidth();
         }
         // the position is within the horizontal boundary of the current index
         // determine whether use the left or the right boundary
         m_affectedItemOffset = curIdx + ((position.GetX() - curLeft) <= (curRight - position.GetX()) ? 0 : 1);
      }
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
