/* ***************************************************************************************
* FILE:          RtText.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtText is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Widgets/2D/RichText/Utils/RtConfig.h>
#include <Widgets/2D/RichText/DocumentModel/RtParagraphItem.h>
#include <FeatStd/Util/String.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Widgets/2D/RichText/LiteHtml/DocumentContainer.h>
#include <Widgets/2D/RichText/LiteHtml/TruncationTextContext.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocHelper.h>

namespace Candera {
namespace TextRendering {
class Font;
class MaximalPreprocessingContext;
class AlignTextMeasureContext;
}


}

namespace hmibase {
namespace widget {
namespace richtext {

/** The paragraph item representing a single and inseparable text block. */
class Text : public ParagraphItem
{
      FEATSTD_TYPEDEF_BASE(ParagraphItem);

   public:
      FEATSTD_RTTI_DECLARATION();

      struct HighlightRange
      {
         FeatStd::UInt32 m_start;
         FeatStd::UInt32 m_end;

         typedef FeatStd::Internal::Vector<HighlightRange> List;

         HighlightRange();
         HighlightRange(FeatStd::UInt32 start, FeatStd::UInt32 end);
      };

      struct HighlightData
      {
         HighlightRange::List m_rangeList;
         DocElement::BoxList m_boxList;
         Candera::Color m_textColor;
         Candera::Color m_backgroundColor;

         typedef FeatStd::MemoryManagement::SharedPointer<HighlightData> SharedPointer;

         FEATSTD_SHARED_POINTER_DECLARATION();

         bool IsInRange(FeatStd::UInt32 index) const;
      };

      struct TruncationData
      {
         bool m_isTruncationLine;
         bool m_isLastInTruncationLine;
         bool m_hasMoreLines;
         bool m_truncationNeeded;
         Candera::TextRendering::BidiBaseLevel::Enum m_baseBidiLevel;
         Candera::TextRendering::BidiBaseLevel::Enum m_elementBidiLevel;

         TruncationData();
      };

      Text(const FeatStd::String& text = "");
      virtual ~Text();

      void SetText(const FeatStd::String& text);
      const FeatStd::String& GetText() const;

      void SetContext(const TruncationTextContext::SharedPointer& context);
      const TruncationTextContext::SharedPointer& GetContext() const;

      /** Provides the number of atomic parts - single letters of the text - of this paragraph item.
          @return The number of letters of this text. */
      virtual FeatStd::UInt16 GetPartCount() const override;

      /** Retrieves the position the 'partIdx'th letter of this text.
          @param partIdx  The index of the letter of which the position is retrieved. The index is clipped to 'GetPartCount'.
          @return         The position of the left upper corner of the addressed letter. */
      virtual Candera::Vector2 GetPartPosition(FeatStd::UInt16 partIdx) const override;

      /** Retrieves the bounding rectangle the 'partIdx'th letter of this text.
          @param partIdx  The index of the letter of which the bounding rectangle is retrieved. The index is clipped to 'GetPartCount'.
          @return         The bounding rectangle relative to the left upper corner of the item. */
      virtual Candera::Rectangle GetPartBoundaries(FeatStd::UInt16 partIdx) const override;

      void SetHighlightData(const HighlightData::SharedPointer& hightlightData);

      const HighlightData::SharedPointer& GetHighlightData() const;

      void SetTruncationData(const TruncationData& truncationData);
      const TruncationData& GetTruncationData() const;

#if defined (RICHTEXT_LOGGING_ENABLED)
      virtual void Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine = false,
                       StreamIterationData& data = StreamIterationData::defaultData) const override;
#endif
   private:
      FeatStd::String m_text;
      TruncationTextContext::SharedPointer m_textContext;
      HighlightData::SharedPointer m_highlightData;
      TruncationData m_truncationData;
};


inline
Text::HighlightRange::HighlightRange() :
   m_start(0),
   m_end(0)
{
}


inline
Text::HighlightRange::HighlightRange(FeatStd::UInt32 start, FeatStd::UInt32 end) :
   m_start(start),
   m_end(end)
{
}


inline
void Text::SetText(const FeatStd::String& text)
{
   m_text = text;
}


inline
const FeatStd::String& Text::GetText() const
{
   return m_text;
}


inline
void Text::SetContext(const TruncationTextContext::SharedPtr& context)
{
   m_textContext = context;
}


inline
const TruncationTextContext::SharedPtr& Text::GetContext() const
{
   return m_textContext;
}


inline
void Text::SetHighlightData(const HighlightData::SharedPointer& hightlightData)
{
   m_highlightData = hightlightData;
}


inline
const hmibase::widget::richtext::Text::HighlightData::SharedPointer& Text::GetHighlightData() const
{
   return m_highlightData;
}


inline
Text::TruncationData::TruncationData() :
   m_isTruncationLine(false),
   m_isLastInTruncationLine(false),
   m_hasMoreLines(false),
   m_truncationNeeded(false),
   m_baseBidiLevel(Candera::TextRendering::BidiBaseLevel::Implicit),
   m_elementBidiLevel(Candera::TextRendering::BidiBaseLevel::Implicit)
{
}


inline
void Text::SetTruncationData(const TruncationData& truncationData)
{
   m_truncationData = truncationData;
}


inline
const Text::TruncationData& Text::GetTruncationData() const
{
   return m_truncationData;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
