/* ***************************************************************************************
* FILE:          RtDocHelper.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocHelper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Widgets/2D/RichText/Utils/RtConfig.h>
#include <Candera/TextEngine/Style.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElementTraverser.h>

#if defined(RICHTEXT_LOGGING_ENABLED)
#include <iostream>
#include <sstream>
#endif

namespace hmibase {
namespace widget {
namespace richtext {

class CustomStyleCache
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(CustomStyleCache);

      static CustomStyleCache* Get(const FeatStd::Char* faceName, FeatStd::Int32 height);

      const Candera::TextRendering::SharedStyle::SharedPointer& GetStyle() const;

   private:
      typedef FeatStd::Internal::Vector<CustomStyleCache::SharedPointer> Cache;

      struct Key
      {
         FeatStd::String m_faceName;
         FeatStd::Int32 m_height;

         Key(const FeatStd::Char* faceName, FeatStd::Int32 height);

         static FeatStd::Int Compare(const Key& left, const CustomStyleCache::SharedPointer& right);
      };

      Key m_key;
      Candera::TextRendering::SharedStyle::SharedPointer m_style;

      FEATSTD_SHARED_POINTER_CREATE_DECLARATION();
      FEATSTD_SHARED_POINTER_DECLARATION();

      CustomStyleCache();
      bool Init(const FeatStd::Char* faceName, FeatStd::Int32 height);
};


inline
const Candera::TextRendering::SharedStyle::SharedPointer& CustomStyleCache::GetStyle() const
{
   return m_style;
}


#if defined(RICHTEXT_LOGGING_ENABLED)
void RemoveLastLine(std::ostream& stream);

#define RICHTEXT_LOG_BEGIN(stream) \
   do { \
      std::ostringstream stream

#define RICHTEXT_LOG(stream, message) \
      stream << message

#define RICHTEXT_LOG_END(stream) \
      stream.flush(); \
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%s", stream.str().c_str())); \
   } while (false)

#define __ARG_0(_0, ...) _0
#define _ARG_0(args) __ARG_0 args

#define __ARG_1(_0, _1, ...) _1
#define _ARG_1(args) __ARG_1 args

#define RICHTEXT_LOG_DOCELEMENT(...) \
   Document* doc = GetDocument(); \
   if ((doc != 0) && (_ARG_0((__VA_ARGS__)) != 0)) { \
      _ARG_0((__VA_ARGS__))->Log(doc->m_creationStream, Candera::Rectangle(), _ARG_1((__VA_ARGS__, false))); \
   } \
   else { \
      FEATSTD_DEBUG_FAIL(); \
   }

struct StreamIterationData
{
   static StreamIterationData defaultData;

   size_t indent;
   const FeatStd::Char* name;
   bool deep;

   StreamIterationData() :
      indent(0),
      name("---"),
      deep(false)
   {
   }
};


class StreamTraverser : public DocElementTraverser
{
   public:
      StreamTraverser(std::ostream& stream, StreamIterationData& data = StreamIterationData::defaultData);

   protected:
      virtual bool Process(const class DocElement& object, const Candera::Rectangle& effectiveRect) override;

   private:
      std::ostream& m_stream;
      StreamIterationData& m_data;

      StreamTraverser(const StreamTraverser&);
      StreamTraverser& operator =(const StreamTraverser&);
};


#else
#define RICHTEXT_LOG_BEGIN(stream)
#define RICHTEXT_LOG_END(stream)
#define RICHTEXT_LOG_DOCELEMENT(...)
#define RICHTEXT_LOG(stream, message)
#endif
} // namespace richtext
} // namespace widget
} // namespace hmibase
