/* ***************************************************************************************
* FILE:          RtAssetUrlResolver.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtAssetUrlResolver is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtAssetUrlResolver.h"
#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <FeatStd/Util/StaticObject.h>
#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/Utils/RtStringInStream.h>
#include <hmibase/util/Macros.h>

namespace hmibase {
namespace widget {
namespace richtext {

AssetUrlResolver& AssetUrlResolver::GetInstance()
{
   FEATSTD_SYNCED_STATIC_OBJECT(AssetUrlResolver, s_assetUrlResolver);
   return s_assetUrlResolver;
}


AssetUrlResolver::AssetUrlResolver() :
   m_assetProvider(0)
{
}


Candera::Bitmap::SharedPointer AssetUrlResolver::GetBitmap(Url baseUrl, Url resourceUrl)
{
   Candera::Bitmap::SharedPointer bitmap;
   UrlSegments urlSegments;
   SplitUrl(baseUrl, resourceUrl, urlSegments);
   if (m_assetProvider != 0)
   {
      bitmap = m_assetProvider->GetBitmapById(GetId(Candera::BitmapLib, urlSegments.authority, urlSegments.path));
   }

   return bitmap;
}


TextInStream::SharedPointer AssetUrlResolver::GetTextResource(Url baseUrl, Url resourceUrl)
{
   TextInStream::SharedPointer textResource;

   UrlSegments urlSegments;
   SplitUrl(baseUrl, resourceUrl, urlSegments);
   if (m_assetProvider != 0)
   {
      Candera::ResourceDataHandle hdl = m_assetProvider->GetRawResourceById(GetId(Candera::RawResourceLib, urlSegments.authority, urlSegments.path));
      if (hdl.m_accessor.IsValid())
      {
         Candera::ResourceObject<FeatStd::TChar> obj(hdl);
         textResource = TextInStream::SharedPointer(FEATSTD_NEW(StringInStream)(obj.GetData()));
      }
   }

   return textResource;
}


Candera::TextRendering::SharedStyle::SharedPointer AssetUrlResolver::GetTextStyle(Url baseUrl, Url resourceUrl)
{
   Candera::TextRendering::SharedStyle::SharedPointer style;

   UrlSegments urlSegments;
   SplitUrl(baseUrl, resourceUrl, urlSegments);
   if (m_assetProvider != 0)
   {
      style = m_assetProvider->GetTextStyleById(GetId(Candera::TextStyleLib, urlSegments.authority, urlSegments.path));
   }

   return style;
}


Candera::Id AssetUrlResolver::GetId(Candera::AssetLib assetLib, const FeatStd::String& authority, const FeatStd::String& path) const
{
   Candera::Id id(0);

   if (authority == "path")
   {
      id = Candera::Internal::AssetProviderFunctions::GetIdByName(m_assetProvider, assetLib, path.GetCString());
   }
   else
   {
      if (authority == "id")
      {
         FeatStd::UInt32 libraryId = 0;
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
         if (FeatStd::Internal::String::StringScanf(path.GetCString(), "%d", &libraryId) > 0)
#else
         if (::sscanf(path.GetCString(), "%d", &libraryId) > 0)
#endif
         {
            id = CDA_LIBRARY_ASSETID(0, libraryId);
         }
      }
   }

   return id;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
