/* ***************************************************************************************
* FILE:          ProgressBarWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ProgressBarWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/ProgressBar/ProgressBarWidget2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_PROGRESSBAR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ProgressBarWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ProgressBarWidget2D)

using namespace Candera;
using namespace Courier;

/****************************************************************************
    CONSTRUCTOR
    ****************************************************************************/
ProgressBarWidget2D::ProgressBarWidget2D() :
   _repeatTimer(),
   _stepIndex(0),
   _shouldUpdateProgressBar(false),
   _bufferValueUpdated(false),
   _stepSize_X(0.0f),
   _stepSize_Y(0.0f),
   _invalidate(false)
{
   _repeatTimer.setName("ProgressBarWidget2D", GetLegacyName());
}


/****************************************************************************
    DESTRUCTOR
    ****************************************************************************/
ProgressBarWidget2D::~ProgressBarWidget2D()
{
   _stepIndex = 0;
   _shouldUpdateProgressBar = false;
   _bufferValueUpdated = false;
   _stepSize_X = 0;
   _stepSize_Y = 0;
}


/****************************************************************************
*     Req ID      : NA
*     Function    : Init
*     Description : Initializes the widget so that all referred resource can be
*                                 resolved
*     Parameters  : Candera::AssetProvider*
*     Return      : void
****************************************************************************/
void ProgressBarWidget2D::InitWidget()
{
   Base::InitWidget();
}


/****************************************************************************
*     Req ID      : NA
*     Function    : OnParentViewRenderingEnabled
*     Description : Is called whenever enable rendering of the parent view is enabled or disabled
*     Parameters  : bool
*     Return      : void
****************************************************************************/
void ProgressBarWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   if (enable)
   {
      //initialSetup();
   }
   Base::OnParentViewRenderingEnabled(enable);
}


void ProgressBarWidget2D::OnParentViewLoad(bool enable)
{
   if (enable)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar [%s] Parent view loaded", GetLegacyName()));
      initialSetup();   // Initial setup need only when the view is loaded
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar [%s] Parent view unloaded", GetLegacyName()));
   }
}


/****************************************************************************
*     Req ID      : NA
*     Function    : Update
*     Description : Update the progress bar based on the binding source
*     Parameters  : void
*     Return      : void
****************************************************************************/
void ProgressBarWidget2D::Update()
{
   Base::Update();

   if (_shouldUpdateProgressBar)       // checks if there is any update in current/timervalue
   {
      updateProgressBar();
   }
   if (_bufferValueUpdated)
   {
      updateProgressBarBuffer();
   }
   if (_bufferValueUpdated || _shouldUpdateProgressBar || _invalidate)
   {
      Invalidate();
      _shouldUpdateProgressBar = false;
      _bufferValueUpdated = false;
      _invalidate = false;
   }
}


/****************************************************************************
*     Function    : OnMessage
*     Description : Is called when a message shall be distributed through the view
*                   tree and its views and widgets
*     Parameters  : Message object to be processed.
*     Return      : true if the message is consumed
*                   false if the message should be forwarded (default)
****************************************************************************/

bool ProgressBarWidget2D::OnMessage(const Message& msg)
{
   bool bIsMsgConsumed = Base::OnMessage(msg);

   if (bIsMsgConsumed)
   {
      return true;
   }

   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerMsg = message_cast<const TimerExpiredMsg*>(&msg);

         if (timerMsg != 0)
         {
            if ((timerMsg->GetTimer() == &_repeatTimer) && (!_repeatTimer.stopped()))
            {
               //fill one step for every repeat time value
               updatePbarBasedOnTimer();
               bIsMsgConsumed = true;
            }
         }
      }
      break;

      default:
         // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"ProgressBarWidget2D: [%s].Message not consumed.",GetLegacyName()));
         break;
   }

   return bIsMsgConsumed;
}


//----------------------------------------------------------------------------
void ProgressBarWidget2D::updateProgressBar()
{
   Candera::Float progressPosition_X = 0;
   Candera::Float progressPosition_Y = 0;

   if (0 != GetProgressFillNode())
   {
      if (GetTotalValue() > 0)
      {
         if (GetCurrentValue() > GetTotalValue())
         {
            SetCurrentValue(GetTotalValue());
         }
         progressPosition_X = (GetCurrentValue() * _fillNodeBoundingRectangle.GetWidth()) / GetTotalValue();
         progressPosition_Y = (GetCurrentValue() * _fillNodeBoundingRectangle.GetHeight()) / GetTotalValue();
         updateMaskNodePosition(GetMaskNode(), _maskNodeStaticPosition, progressPosition_X, progressPosition_Y);
      }
      else
      {
         ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "ProgressBar TotalValue is zero for [%s]", GetLegacyName()));
      }
   }
}


//----------------------------------------------------------------------------
void ProgressBarWidget2D::updateProgressBarBuffer()
{
   Candera::Float progressPosition_X = 0;
   Candera::Float progressPosition_Y = 0;

   if (0 != GetProgressBufferNode())
   {
      if (GetTotalValue() > 0)
      {
         if (GetBufferValue() > GetTotalValue())
         {
            SetBufferValue(GetTotalValue());
         }
         progressPosition_X = (GetBufferValue() * _bufferNodeBoundingRectangle.GetWidth()) / GetTotalValue();
         progressPosition_Y = (GetBufferValue() * _bufferNodeBoundingRectangle.GetHeight()) / GetTotalValue();
         updateMaskNodePosition(GetBufferMaskNode(), _bufferMaskNodeStaticPosition, progressPosition_X, progressPosition_Y);
      }
      else
      {
         ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "ProgressBar TotalValue is zero for [%s]", GetLegacyName()));
      }
   }
}


/****************************************************************************
*     Function    : updatePbarBasedOnTimer()
*     Description : Function to update the basic PB filling state based on timeout value
*     Parameters  : Candera::Float
*     Return      : void
****************************************************************************/
void ProgressBarWidget2D::updatePbarBasedOnTimer()
{
   Candera::Float bitmapFillWidth = 0.0f;
   Candera::Float bitmapFillHeight = 0.0f;

   if ((GetTimerStepCount() > 0))
   {
      if (_stepIndex >= GetTimerStepCount())
      {
         _repeatTimer.stop();
         _stepIndex = 0;
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProgressBar::updatePbarBasedOnTimer RepeatTimer stopped for [%s] ", GetLegacyName()));
         return;
      }
      else if (_stepSize_X <= _fillNodeBoundingRectangle.GetWidth())
      {
         triggerUpdate();

         _stepIndex++;
         bitmapFillWidth = _stepSize_X * static_cast<Candera::Float>(_stepIndex);
         bitmapFillHeight = _stepSize_Y * static_cast<Candera::Float>(_stepIndex);
         updateMaskNodePosition(GetMaskNode(), _maskNodeStaticPosition, bitmapFillWidth, bitmapFillHeight); // update the fill node

         //invalidate_IfWakeUpRenderMechanismIsEnabled();

         _invalidate = true;
      }
      else
      {
      }
   }
}


//----------------------------------------------------------------------------
Candera::Vector2 ProgressBarWidget2D::arrangeMaskNodeInitialPosition(Candera::Node2D* node) const
{
   Candera::Vector2 pos;
   if (0 != node)
   {
      if (0 != GetProgressFillNode())
      {
         Candera::Rectangle rec = GetProgressFillNode()->GetBoundingRectangle();
         pos = GetProgressFillNode()->GetPosition();
         switch (GetProgressBarOrientationType())
         {
            case Candera::eHorizontalLeft:
               pos.SetX(pos.GetX() - rec.GetWidth());
               break;
            case Candera::eHorizontalRight:
               pos.SetX(pos.GetX() + rec.GetWidth());
               break;
            case Candera::eVerticalTop:
               pos.SetY(pos.GetY() - rec.GetHeight());
               break;
            case Candera::eVerticalBottom:
               pos.SetY(pos.GetY() + rec.GetHeight());
               break;
            default:
               break;
         }
         node->SetPosition(pos);
      }
   }
   return pos;
}


//----------------------------------------------------------------------------
void ProgressBarWidget2D::updateMaskNodePosition(Candera::Node2D* maskNode, Candera::Vector2 maskNodeStaticPos, Candera::Float progressPosition_X, Candera::Float progressPosition_Y) const
{
   if (0 != maskNode)
   {
      Candera::Vector2 maskNodeNewPos;
      maskNodeNewPos = GetProgressFillNode()->GetPosition(); // the position of the FillNode is taken as reference point. As the mask node is placed on top of the fill node.

      switch (GetProgressBarOrientationType())
      {
         case eHorizontalLeft:
         {
            maskNodeNewPos.SetX(maskNodeStaticPos.GetX() + progressPosition_X);
            maskNode->SetPosition(maskNodeNewPos);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::updateMaskNodePosition [%f,%f] orient[eHorizontalLeft]", maskNodeNewPos.GetX(), maskNodeNewPos.GetY()));
         }
         break;

         case eHorizontalRight:
         {
            maskNodeNewPos.SetX(maskNodeStaticPos.GetX() - progressPosition_X);
            maskNode->SetPosition(maskNodeNewPos);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::updateMaskNodePosition [%f,%f] orient[eHorizontalRight]", maskNodeNewPos.GetX(), maskNodeNewPos.GetY()));
         }
         break;

         case eVerticalTop:
         {
            maskNodeNewPos.SetY(maskNodeStaticPos.GetY() + progressPosition_Y);
            maskNode->SetPosition(maskNodeNewPos);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::updateMaskNodePosition [%f,%f] orient[eVerticalTop]", maskNodeNewPos.GetX(), maskNodeNewPos.GetY()));
         }
         break;

         case eVerticalBottom:
         {
            maskNodeNewPos.SetY(maskNodeStaticPos.GetY() - progressPosition_Y);
            maskNode->SetPosition(maskNodeNewPos);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::updateMaskNodePosition[%f,%f] orient[eVerticalBottom]", maskNodeNewPos.GetX(), maskNodeNewPos.GetY()));
         }
         break;

         default:
            break;
      }
   }
}


//----------------------------------------------------------------------------
void ProgressBarWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case CurrentValuePropertyId:
         _shouldUpdateProgressBar = true;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::OnChanged CurrentValue = %f for [%s]", GetCurrentValue(), GetLegacyName()));
         break;

      case BufferValuePropertyId:
         _bufferValueUpdated = true;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::OnChanged BufferValue = %f for [%s]", GetBufferValue(), GetLegacyName()));
         break;

      case StartTimerPropertyId:
         GetStartTimer() ? _repeatTimer.start() : _repeatTimer.stop();
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::OnChanged StartTimerStatus = %d for [%s]", GetStartTimer(), GetLegacyName()));
         break;

      case RestartTimerPropertyId:
         if (GetRestartTimer())
         {
            _stepIndex = 0;
            _shouldUpdateProgressBar = true;
            _repeatTimer.start();
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::OnChanged Timer Restarted for [%s]", GetLegacyName()));
         }
         break;

      default:
         break;
   }
}


bool ProgressBarWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ProgressBarWidget2D* original = CLONEABLE_WIDGET_CAST<const ProgressBarWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetProgressFillNode(controlTemplateMap.ResolveNodeClone(original->GetProgressFillNode()));
      SetProgressBufferNode(controlTemplateMap.ResolveNodeClone(original->GetProgressBufferNode()));
      SetMaskNode(controlTemplateMap.ResolveNodeClone(original->GetMaskNode()));
      SetBufferMaskNode(controlTemplateMap.ResolveNodeClone(original->GetBufferMaskNode()));
      SetProgressBarOrientationType(original->GetProgressBarOrientationType());
      SetTotalValue(original->GetTotalValue());
      ProgressBarWidget2DBase::SetCurrentValue(original->GetCurrentValue());
      SetBufferValue(original->GetBufferValue());
      SetTimerValue(original->GetTimerValue());
      SetTimerStepCount(original->GetTimerStepCount());
      SetStartTimer(original->GetStartTimer());

      // copy the values calculated from the original widget
      _maskNodeStaticPosition = original->_maskNodeStaticPosition;	// copy the mask node initial position
      _bufferMaskNodeStaticPosition = original->_bufferMaskNodeStaticPosition;
      _fillNodeBoundingRectangle = original->_fillNodeBoundingRectangle;
      _bufferNodeBoundingRectangle = original->_bufferNodeBoundingRectangle;

      cloned = true;
   }
   return cloned;
}


void ProgressBarWidget2D::initialSetup()
{
   if (0 != GetNode())
   {
      GetNode()->SetRenderingEnabled(IsVisible());
   }

   if (NULL != GetProgressFillNode())
   {
      _fillNodeBoundingRectangle = GetProgressFillNode()->GetBoundingRectangle();
   }
   if (0 != GetProgressBufferNode())
   {
      _bufferNodeBoundingRectangle = GetProgressBufferNode()->GetBoundingRectangle();
   }
   if (0 != GetMaskNode())
   {
      _maskNodeStaticPosition = arrangeMaskNodeInitialPosition(GetMaskNode());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::initialSetup MaskNode static position[%f,%f] for [%s]", _maskNodeStaticPosition.GetX(), _maskNodeStaticPosition.GetY(), GetLegacyName()));
   }
   if (0 != GetBufferMaskNode())
   {
      _bufferMaskNodeStaticPosition = arrangeMaskNodeInitialPosition(GetBufferMaskNode());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::initialSetup Buffer MaskNode static position[%f,%f] for [%s]", _bufferMaskNodeStaticPosition.GetX(), _bufferMaskNodeStaticPosition.GetY(), GetLegacyName()));
   }

   if (GetProgressFillType() == Candera::eTimerBased && GetTimerStepCount() > 0)
   {
      _stepSize_X = (_fillNodeBoundingRectangle.GetWidth() / GetTimerStepCount());
      _stepSize_Y = (_fillNodeBoundingRectangle.GetHeight() / GetTimerStepCount());
      _repeatTimer.setTimeoutWithRepeat((GetTimerValue() / GetTimerStepCount()), (GetTimerValue() / GetTimerStepCount()));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProgressBar::initialSetup internal Timer set for %d mseconds for [%s]", (GetTimerValue() / GetTimerStepCount()), GetLegacyName()));
   }
}
