/* ***************************************************************************************
* FILE:          PanelSlideWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PanelSlideWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "PanelSlideWidget2D.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "Widgets/widget_etg_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PanelSlideWidget2D.cpp.trc.h"
#endif

#include "Widgets/2D/WidgetGestureConfig.h"
#include "Widgets/2D/WidgetFinder2D.h"
#include "Widgets/2D/Animation/AnimationWidget2D.h"
#include "Widgets/2D/Button/ButtonWidget2D.h"

CGI_WIDGET_RTTI_DEFINITION(PanelSlideWidget2D);

PanelSlideWidget2D::PanelSlideWidget2D() :
   _previousTouchPos(Candera::Vector2(0.0F, 0.0F)),
   _offset(0.0F),
   _nodeStartPosition(Candera::Vector2(0.0F, 0.0F)),
   _nodeEndPosition(Candera::Vector2(0.0F, 0.0F)),
   _isOpened(false),
   _isTouched(false),
   _invalid(false)
{
   SetDrag(true);
   SetSwipe(true);
}


/************************************************************************************************/
PanelSlideWidget2D::~PanelSlideWidget2D()
{
}


/************************************************************************************************/
WidgetGestureConfig PanelSlideWidget2D::getDefaultGestureConfig() const
{
   return WidgetGestureConfig::getDefaults();
   //return WidgetGestureConfig(WidgetGestureConfig::Drag(true, 0, 0, 3/*minDistance*/), WidgetGestureConfig::Swipe(true, 0, 5/*minDistance*/));
}


/************************************************************************************************/
void PanelSlideWidget2D::InitWidget()
{
   calStartAndEndPosition();
   if (GetPanelNode())
   {
      GetPanelNode()->SetRenderingEnabled(false);
   }
   else
   {
      ETG_TRACE_ERR(("PanelSlideWidget2D::Panel Node is NULL: [%s]", this->GetLegacyName()));
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::Update()
{
   if ((GetNode() != NULL) && (IsInvalid()))
   {
      _isOpened = GetPanelOpen();
      if (_isOpened)
      {
         onPanelOpenedOrClosed();
         startAnimation(_nodeEndPosition);
      }
      else
      {
         startAnimation(_nodeStartPosition);
      }
      SetInvalid(false);
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::calStartAndEndPosition()
{
   if (GetNode())
   {
      _nodeStartPosition = Candera::Vector2(GetNode()->GetPosition().GetX(), GetNode()->GetPosition().GetY());
      switch (GetDirection())
      {
         case Candera::enHorizontalLeft:
         {
            _nodeEndPosition = Candera::Vector2(GetNode()->GetPosition().GetX() + GetPanelSize(), \
                                                GetNode()->GetPosition().GetY());
         }
         break;
         case Candera::enHorizontalRight:
         {
            _nodeEndPosition = Candera::Vector2(GetNode()->GetPosition().GetX() - GetPanelSize(), \
                                                GetNode()->GetPosition().GetY());
         }
         break;
         case Candera::enVerticalTop:
         {
            _nodeEndPosition = Candera::Vector2(GetNode()->GetPosition().GetX(), \
                                                GetNode()->GetPosition().GetY() + GetPanelSize());
         }
         break;
         case Candera::enVerticalBottom:
         {
            _nodeEndPosition = Candera::Vector2(GetNode()->GetPosition().GetX(), \
                                                GetNode()->GetPosition().GetY() - GetPanelSize());
         }
         break;
         default:
            break;
      }
   }
}


/************************************************************************************************/
bool PanelSlideWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const PanelSlideWidget2D* original = CLONEABLE_WIDGET_CAST<const PanelSlideWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetPanelButton(controlTemplateMap.ResolveWidgetClone(original->GetPanelButton()));
      SetPanelNode(controlTemplateMap.ResolveNodeClone(original->GetPanelNode()));
      SetDragThreshold(original->GetDragThreshold());
      SetDirection(original->GetDirection());
      SetPanelSize(original->GetPanelSize());
      SetPanelOpen(original->GetPanelOpen());
      cloned = true;
   }
   return cloned;
}


/************************************************************************************************/
bool PanelSlideWidget2D::OnMessage(const Courier::Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }
   bool isConsumed = false;
   if (GetParentView() != NULL)
   {
      switch (msg.GetId())
      {
         case PanelSlideReqMsg::ID:
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PanelSlideWidget2D::Received PanelSlideReqMsg :[%s]", GetLegacyName()));
            const PanelSlideReqMsg* panelSlideReqMsg = Courier::message_cast<const PanelSlideReqMsg*>(&msg);
            if ((panelSlideReqMsg != NULL) &&
                  (panelSlideReqMsg->GetView() == GetParentView()->GetId()) &&
                  (panelSlideReqMsg->GetSender() == Courier::Identifier(GetLegacyName())))
            {
               onPanelRegMsg(*panelSlideReqMsg);
               isConsumed = true;
            }
         }
         break;
         case AnimationWidgetIndMsg::ID:
         {
            const AnimationWidgetIndMsg* animationIndMsg = Courier::message_cast<const AnimationWidgetIndMsg*>(&msg);
            AnimationWidget2D* animationWidget = WidgetFinder::FindAncestorWidget<AnimationWidget2D>(this);
            if ((animationIndMsg != NULL) && (animationWidget != NULL) && (!_isOpened) &&
                  (animationIndMsg->GetView() == GetParentView()->GetId()) &&
                  (animationIndMsg->GetSender() == Courier::Identifier(animationWidget->GetLegacyName())))
            {
               onPanelOpenedOrClosed();
               isConsumed = true;
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PanelSlideWidget2D::Consumed AnimationWidgetIndMsg:[%s]", GetLegacyName()));
            }
         }
         break;
         case ButtonReactionMsg::ID:
         {
            ButtonWidget2D* markerButton = NULL;
            const ButtonReactionMsg* buttonReactionMsg = Courier::message_cast<const ButtonReactionMsg*>(&msg);
            if (GetPanelButton())
            {
               markerButton = dynamic_cast<ButtonWidget2D*>(GetPanelButton());
            }
            if ((buttonReactionMsg != NULL) && (markerButton != NULL) &&
                  (buttonReactionMsg->GetView() == GetParentView()->GetId()) &&
                  (buttonReactionMsg->GetSender() == Courier::Identifier(markerButton->GetLegacyName())))
            {
               onBtnReactionMsg(*buttonReactionMsg);
               isConsumed = true;
            }
         }
         break;

         default:
            break;
      }
   }
   return isConsumed;
}


/************************************************************************************************/
void PanelSlideWidget2D::onPanelRegMsg(const PanelSlideReqMsg& panelSlideReqMsg)
{
   switch (panelSlideReqMsg.GetPanelAction())
   {
      case enClose:
      {
         if (_isOpened)
         {
            _isOpened = false;
            startAnimation(_nodeStartPosition);
         }
      }
      break;
      case enOpen:
      {
         if (!_isOpened)
         {
            _isOpened = true;
            onPanelOpenedOrClosed();
            startAnimation(_nodeEndPosition);
         }
      }
      break;
      case enToggle:
      {
         _isOpened = !_isOpened;
         if (_isOpened)
         {
            startAnimation(_nodeEndPosition);
            onPanelOpenedOrClosed();
         }
         else
         {
            startAnimation(_nodeStartPosition);
         }
      }
      break;
      default:
         break;
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onBtnReactionMsg(const ButtonReactionMsg& buttonReactionMsg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PanelSlideWidget2D::Consumed ButtonReactionMsg:[%s]", GetLegacyName()));
   switch (buttonReactionMsg.GetEnReaction())
   {
      case enPress:
      {
         _isTouched = true;
      }
      break;
      case enRelease:
      {
         onTap();
      }
      break;
      default:
         break;
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case PanelOpenPropertyId:
         SetInvalid(true);
         break;
      default:
         break;
   }
}


/************************************************************************************************/
bool PanelSlideWidget2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool isConsumed = false;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         onSwipeEnd(gestureData);
         isConsumed = true;
      }
      break;
      default:
         break;
   }
   return isConsumed;
}


/************************************************************************************************/
void PanelSlideWidget2D::onSwipeEnd(const hmibase::input::gesture::GestureEvent& gestureData)
{
   switch (GetDirection())
   {
      case Candera::enHorizontalLeft:
      {
         if (gestureData._velocity1.x > 0)
         {
            _isOpened = true;
            onPanelOpenedOrClosed();
            startAnimation(_nodeEndPosition);
         }
         else
         {
            startAnimation(_nodeStartPosition);
            _isOpened = false;
         }
      }
      break;
      case Candera::enHorizontalRight:
      {
         if (gestureData._velocity1.x < 0)
         {
            _isOpened = true;
            onPanelOpenedOrClosed();
            startAnimation(_nodeEndPosition);
         }
         else
         {
            startAnimation(_nodeStartPosition);
            _isOpened = false;
         }
      }
      break;
      case Candera::enVerticalTop:
      {
         if (gestureData._velocity1.y > 0)
         {
            _isOpened = true;
            onPanelOpenedOrClosed();
            startAnimation(_nodeEndPosition);
         }
         else
         {
            startAnimation(_nodeStartPosition);
            _isOpened = false;
         }
      }
      break;
      case Candera::enVerticalBottom:
      {
         if (gestureData._velocity1.y < 0)
         {
            _isOpened = true;
            onPanelOpenedOrClosed();
            startAnimation(_nodeEndPosition);
         }
         else
         {
            startAnimation(_nodeStartPosition);
            _isOpened = false;
         }
      }
      break;
      default:
         break;
   }
}


/************************************************************************************************/
bool PanelSlideWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool isConsumed = false;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         if (GetPanelNode())
         {
            GetPanelNode()->SetRenderingEnabled(true);
            Invalidate();
         }
         _previousTouchPos = Candera::Vector2(static_cast<Candera::Float>(gestureData._pt1.x), \
                                              static_cast<Candera::Float>(gestureData._pt1.x));
         isConsumed = true;
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
      {
         onDragMove(gestureData);
         isConsumed = true;
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         onDragEnd();
         isConsumed = true;
      }
      break;
      default:
         break;
   }
   return isConsumed;
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragMove(const hmibase::input::gesture::GestureEvent& gestureData)
{
   if (GetDirection() == Candera::enHorizontalLeft || GetDirection() == Candera::enHorizontalRight)
   {
      _offset = static_cast<FeatStd::Float>(gestureData._pt1.x) - _previousTouchPos.GetX();
   }
   else if (GetDirection() == Candera::enVerticalTop || GetDirection() == Candera::enVerticalBottom)
   {
      _offset = static_cast<FeatStd::Float>(gestureData._pt1.y) - _previousTouchPos.GetY();
   }
   if (_offset != 0)
   {
      switch (GetDirection())
      {
         case Candera::enHorizontalLeft:
         {
            onDragLeft();
         }
         break;
         case Candera::enHorizontalRight:
         {
            onDragRight();
         }
         break;
         case Candera::enVerticalTop:
         {
            onDragTop();
         }
         break;
         case Candera::enVerticalBottom:
         {
            onDragBottom();
         }
         break;
         default:
            break;
      }
      Invalidate();
   }
   _previousTouchPos = Candera::Vector2(static_cast<Candera::Float>(gestureData._pt1.x), \
                                        static_cast<Candera::Float>(gestureData._pt1.y));
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragLeft()
{
   if (GetNode())
   {
      if (_offset > 0 && (GetNode()->GetPosition().GetX() + _offset) <= _nodeEndPosition.GetX())
      {
         GetNode()->SetPosition((GetNode()->GetPosition().GetX() + _offset), GetNode()->GetPosition().GetY());
      }
      else if (_offset < 0 && (GetNode()->GetPosition().GetX() + _offset) >= _nodeStartPosition.GetX())
      {
         GetNode()->SetPosition((GetNode()->GetPosition().GetX() + _offset), GetNode()->GetPosition().GetY());
      }
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragRight()
{
   if (GetNode())
   {
      if (_offset < 0 && (GetNode()->GetPosition().GetX() + _offset) >= _nodeEndPosition.GetX())
      {
         GetNode()->SetPosition((GetNode()->GetPosition().GetX() + _offset), GetNode()->GetPosition().GetY());
      }
      else if (_offset > 0 && (GetNode()->GetPosition().GetX() + _offset) <= (_nodeStartPosition.GetX()))
      {
         GetNode()->SetPosition((GetNode()->GetPosition().GetX() + _offset), GetNode()->GetPosition().GetY());
      }
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragTop()
{
   if (GetNode())
   {
      if (_offset > 0 && (GetNode()->GetPosition().GetY() + _offset) <= _nodeEndPosition.GetY())
      {
         GetNode()->SetPosition(GetNode()->GetPosition().GetX(), (GetNode()->GetPosition().GetY() + _offset));
      }
      else if (_offset < 0 && (GetNode()->GetPosition().GetY() + _offset) >= (_nodeStartPosition.GetY()))
      {
         GetNode()->SetPosition(GetNode()->GetPosition().GetX(), (GetNode()->GetPosition().GetY() + _offset));
      }
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragBottom()
{
   if (GetNode())
   {
      if (_offset < 0 && (GetNode()->GetPosition().GetY() + _offset) >= _nodeEndPosition.GetY())
      {
         GetNode()->SetPosition(GetNode()->GetPosition().GetX(), (GetNode()->GetPosition().GetY() + _offset));
      }
      else if (_offset > 0 && (GetNode()->GetPosition().GetY() + _offset) <= (_nodeStartPosition.GetY()))
      {
         GetNode()->SetPosition(GetNode()->GetPosition().GetX(), (GetNode()->GetPosition().GetY() + _offset));
      }
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onTap()
{
   if (_isTouched && !_isOpened)
   {
      _isOpened = true;
      onPanelOpenedOrClosed();
      startAnimation(_nodeEndPosition);
   }
   else if (_isTouched && _isOpened)
   {
      _isOpened = false;
      startAnimation(_nodeStartPosition);
   }
   _isTouched = false;
}


/************************************************************************************************/
void PanelSlideWidget2D::onDragEnd()
{
   Candera::Float totalOffset = 0.0f;
   if (GetNode() && !_isOpened)
   {
      if (GetDirection() == Candera::enHorizontalLeft || GetDirection() == Candera::enHorizontalRight)
      {
         totalOffset = Candera::Math::Absolute(_nodeStartPosition.GetX() - GetNode()->GetPosition().GetX());
      }
      else
      {
         totalOffset = Candera::Math::Absolute(_nodeStartPosition.GetY() - GetNode()->GetPosition().GetY());
      }
      onFarwardDrag(totalOffset);
   }
   else if (GetNode() && _isOpened)
   {
      if (GetDirection() == Candera::enHorizontalLeft || GetDirection() == Candera::enHorizontalRight)
      {
         totalOffset = Candera::Math::Absolute(_nodeEndPosition.GetX() - GetNode()->GetPosition().GetX());
      }
      else
      {
         totalOffset = Candera::Math::Absolute(_nodeEndPosition.GetY() - GetNode()->GetPosition().GetY());
      }
      onBackwardDrag(totalOffset);
   }
   _offset = 0;
   _previousTouchPos = Candera::Vector2(0.0f, 0.0f);
}


/************************************************************************************************/
void PanelSlideWidget2D::onFarwardDrag(Candera::Float totalOffset)
{
   Candera::Float dragThreshold = 0.0f;
   dragThreshold = (GetPanelSize() * GetDragThreshold());
   if (totalOffset <= (dragThreshold))
   {
      startAnimation(_nodeStartPosition);
   }
   else
   {
      startAnimation(_nodeEndPosition);
      _isOpened = true;
      postPanelStateMsg();
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onBackwardDrag(Candera::Float totalOffset)
{
   Candera::Float dragThreshold = 0.0f;
   dragThreshold = (GetPanelSize() * GetDragThreshold());
   if (totalOffset <= (dragThreshold))
   {
      startAnimation(_nodeEndPosition);
   }
   else
   {
      startAnimation(_nodeStartPosition);
      _isOpened = false;
      postPanelStateMsg();
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::startAnimation(Candera::Vector2 endPos)
{
   Candera::Vector2 startPos;
   char buffer[22];
   AnimationWidget2D* animationWidget = WidgetFinder::FindAncestorWidget<AnimationWidget2D>(this);
   //set the position info to animation widget
   if (animationWidget && GetNode() && animationWidget->GetParentView())
   {
      startPos = GetNode()->GetPosition();
      sprintf(buffer, "%d;%d", static_cast<Candera::Int>(startPos.GetX()), static_cast<Candera::Int>(endPos.GetX()));
      animationWidget->SetChannel1(buffer);

      sprintf(buffer, "%d;%d", static_cast<Candera::Int>(startPos.GetY()), static_cast<Candera::Int>(endPos.GetY()));
      animationWidget->SetChannel2(buffer);
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(AnimationWidgetReqMsg)
                        (Courier::AnimationAction::Start, animationWidget->GetParentView()->GetId(), \
                         Courier::Identifier(animationWidget->GetLegacyName()))));
      Invalidate();
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::postPanelStateMsg()
{
   if (GetParentView() != NULL)
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PanelStateMsg)
                        (GetParentView()->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), _isOpened)));
   }
}


/************************************************************************************************/
void PanelSlideWidget2D::onPanelOpenedOrClosed()
{
   if (GetPanelNode())
   {
      GetPanelNode()->SetRenderingEnabled(_isOpened);
      Invalidate();
   }
   postPanelStateMsg();
}


/************************************************************************************************/
