/* ***************************************************************************************
* FILE:          PageEditManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditManager.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "PageEditController2D.h"
#include "PageEditManager.h"
#include "PageEditStrategy.h"
#include "PageEditWidget2D.h"

#include <FeatStd/Platform/Memory.h>
#include <Trace/ToString.h>
#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <View/CGI/CgiExtensions/TouchInput.h>
#include <Widgets/2D/Gizmo/GizmoWidget2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_PAGEEDIT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageEditManager.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace pageedit {

#define AS_VECTOR2(x, y) Candera::Vector2(static_cast<FeatStd::Float>(x), static_cast<FeatStd::Float>(y))
#define AS_MARGIN(left, top, right, bottom) Candera::Margin(static_cast<FeatStd::Int16>(left), static_cast<FeatStd::Int16>(top), static_cast<FeatStd::Int16>(right), static_cast<FeatStd::Int16>(bottom))
#define AS_RECTANGLE(x, y, width, height) Candera::Rectangle(static_cast<FeatStd::Float>(x), static_cast<FeatStd::Float>(y), static_cast<FeatStd::Float>(width), static_cast<FeatStd::Float>(height))


/*****************************************************************************/
PageEditManager::PageEditManager() : _session(NULL), _movementTimer(NULL)
{
}


/*****************************************************************************/
PageEditManager::~PageEditManager()
{
   hmibase::widget::gizmo::DefaultGizmoController2D::GetEventSource().RemoveEventListener(this);
   AppViewHandler::getInstance().unregisterMessageHandler(*this);

   if (_movementTimer != NULL)
   {
      FEATSTD_DELETE(_movementTimer);
      _movementTimer = NULL;
   }

   if (_session != NULL)
   {
      FEATSTD_DELETE(_session);
      _session = NULL;
   }
}


/*****************************************************************************/
void PageEditManager::Initialize()
{
   static bool _initialized = false;
   if (!_initialized)
   {
      _initialized = true;

      hmibase::widget::gizmo::DefaultGizmoController2D::GetEventSource().AddEventListener(this);
      AppViewHandler::getInstance().registerMessageHandler(*this);
   }
}


/*****************************************************************************/
Album* PageEditManager::GetOrCreateAlbum(const AlbumId& albumId)
{
   Initialize();

   Album* album = GetAlbum(albumId);
   if (album == NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetOrCreateAlbum create album=%u",
                          albumId));

      album = FEATSTD_NEW(Album)(albumId);
      _albums[albumId] = album;
   }
   return album;
}


/*****************************************************************************/
Album* PageEditManager::GetAlbum(const AlbumId& albumId)
{
   AlbumsType::const_iterator it = _albums.find(albumId);
   if (it != _albums.end())
   {
      return it->second;
   }

   return NULL;
}


/*****************************************************************************/
Session* PageEditManager::GetSession() const
{
   return _session;
}


/*****************************************************************************/
Session* PageEditManager::BeginSession(Album& album)
{
   if (_session != NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BeginSession session already exists!"));
      return NULL;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BeginSession album=%u",
                       album.GetId()));

   _session = FEATSTD_NEW(Session)(album);
   return _session;
}


/*****************************************************************************/
void PageEditManager::EndSession()
{
   if (_session != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EndSession"));

      FEATSTD_DELETE(_session);
      _session = NULL;
   }
}


/*****************************************************************************/
bool PageEditManager::BeginEditing(const ItemId& itemId)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BeginEditing itemId=%s",
                       HMIBASE_TO_STRING(itemId)));

   for (AlbumsType::const_iterator albumIt = _albums.begin(); albumIt != _albums.end(); ++albumIt)
   {
      Album* album = albumIt->second;
      if (album != NULL)
      {
         for (Album::PagesType::const_iterator pageIt = album->GetPages().begin(); pageIt != album->GetPages().end(); ++pageIt)
         {
            Page::SharedPointer page = pageIt->second;
            if (!page.PointsToNull())
            {
               ItemIndexType itemIndex = page->GetItemIndex(itemId);
               if (itemIndex != -1)
               {
                  return BeginEditing(*album, page, itemIndex);
               }
            }
         }
      }
   }

   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BeginEditing item %s not found!",
                      HMIBASE_TO_STRING(itemId)));
   return false;
}


/*****************************************************************************/
bool PageEditManager::BeginEditing(Album& album, Page::SharedPointer page, ItemIndexType itemIndex)
{
   if (page.PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BeginEditing Null page!"));
      return false;
   }

   PageEditWidget2D* pageEdit = page->WidgetAccessor.getObjectSafely();
   if (pageEdit == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BeginEditing page edit widget is not valid for page %s!",
                         HMIBASE_TO_STRING(page->Id)));
      return false;
   }

   if (!pageEdit->GetAutoArrange())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BeginEditing auto arrange disabled for page %s!",
                          HMIBASE_TO_STRING(page->Id)));
      return false;
   }

   Item* item = page->GetItem(itemIndex);
   if (item == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BeginEditing Index %d invalid for page %s!",
                         itemIndex,
                         HMIBASE_TO_STRING(page)));
      return false;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BeginEditing page=%50s itemIndex=%d item=%s",
                       HMIBASE_TO_STRING(page->Id),
                       itemIndex,
                       HMIBASE_TO_STRING(item->Id)));

   Session* session = BeginSession(album);
   if (session == NULL)
   {
      return false;
   }

   for (Album::PagesType::const_iterator pageIt = session->_Album.GetPages().begin(); pageIt != session->_Album.GetPages().end(); ++pageIt)
   {
      Page::SharedPointer page = pageIt->second;
      if (!page.PointsToNull())
      {
         page->ResetEditingInfo();
         page->InitItemEditingCells();

         //ensure items are still valid
         if (!page->AreItemAccessorsValid())
         {
            PageEditWidget2D* pageEdit = page->WidgetAccessor.getObjectSafely();

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnItemEditStart Items are not valid anymore for page %s",
                                HMIBASE_TO_STRING_VW(pageEdit)));

            PageEditController2D* controller = (pageEdit != NULL) ? Candera::Dynamic_Cast<PageEditController2D*>(pageEdit->GetController()) : NULL;
            if ((pageEdit != NULL) && (controller != NULL))
            {
               controller->CollectItems(*pageEdit, page.GetSharedInstance());
            }
            else
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnItemEditStart Null widget '%20s' or controller '%p' for page %s",
                                   HMIBASE_TO_STRING_VW(pageEdit),
                                   controller,
                                   HMIBASE_TO_STRING(page->Id)));
            }
         }
      }
   }

   session->SourcePage = page;

   //todo: remove this
   session->SourcePage->IsEditing = true;

   session->SourceItemId = item->Id;
   session->SourceItemSpan = item->Cells.Span;
   session->SourceItemTouchedCell = PositionToCell(*item, page->CellSize, GetCurrentTouchCoordinate());

   PostUpdateMessage(enEditStatus::Begin);

   return true;
}


/*****************************************************************************/
void PageEditManager::EndEditing()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EndEditing"));

   StopTimer();

   if (RearrangeItems())
   {
      PostUpdateMessage(enEditStatus::End);
   }

   Session* session = GetSession();
   if (session != NULL)
   {
      for (Album::PagesType::const_iterator pageIt = session->_Album.GetPages().begin(); pageIt != session->_Album.GetPages().end(); ++pageIt)
      {
         Page::SharedPointer page = pageIt->second;
         if (!page.PointsToNull())
         {
            page->ResetEditingInfo();

            for (Page::ItemsType::iterator itemIt = page->Items.begin(); itemIt != page->Items.end(); ++itemIt)
            {
               Item& item = *itemIt;

               if (item.Cells != item.EditingCells)
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "EndEditing page=%20s index=%d oldCells=%13s newCells=%13s item=%s",
                                      HMIBASE_TO_STRING(page->Id),
                                      page->GetItemIndex(item.WidgetAccessor.getObjectSafely()),
                                      HMIBASE_TO_STRING(item.Cells),
                                      HMIBASE_TO_STRING(item.EditingCells),
                                      HMIBASE_TO_STRING_W(item.WidgetAccessor.getObjectSafely())));
                  item.Cells = item.EditingCells;
               }

               item.EditingCells = CellRect();
               item.ExternalId = ItemId();
            }

            PageEditWidget2D* pageEdit = page->WidgetAccessor.getObjectSafely();
            PageEditController2D* controller = (pageEdit != NULL) ? Candera::Dynamic_Cast<PageEditController2D*>(pageEdit->GetController()) : NULL;
            if ((pageEdit != NULL) && (controller != NULL))
            {
               controller->SetItemsBounds(*pageEdit, page.GetSharedInstance(), false);
            }
         }
      }
   }

   EndSession();
}


/*****************************************************************************/
void PageEditManager::AbortEditing()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AbortEditing"));

   StopTimer();

   PostUpdateMessage(enEditStatus::Abort);

   Session* session = GetSession();
   if (session != NULL)
   {
      for (Album::PagesType::const_iterator pageIt = session->_Album.GetPages().begin(); pageIt != session->_Album.GetPages().end(); ++pageIt)
      {
         Page::SharedPointer page = pageIt->second;
         if (!page.PointsToNull())
         {
            page->ResetEditingInfo();

            for (Page::ItemsType::iterator itemIt = page->Items.begin(); itemIt != page->Items.end(); ++itemIt)
            {
               Item& item = *itemIt;

               item.EditingCells = CellRect();
               item.ExternalId = ItemId();
            }

            PageEditWidget2D* pageEdit = page->WidgetAccessor.getObjectSafely();
            PageEditController2D* controller = (pageEdit != NULL) ? Candera::Dynamic_Cast<PageEditController2D*>(pageEdit->GetController()) : NULL;
            if ((pageEdit != NULL) && (controller != NULL))
            {
               controller->SetItemsBounds(*pageEdit, page.GetSharedInstance(), false);
            }
         }
      }
   }

   EndSession();
}


/*****************************************************************************/
void PageEditManager::DetermineDestination()
{
   Session* session = GetSession();

   if (session == NULL)
   {
      return;
   }

   session->DestinationPageNormalizedTouchedCoordinates = Candera::Vector2();
   session->DestinationPageTouchedCell = Cell();
   session->DestinationPageEditingCells = CellRect();
   session->DestinationItemId = ItemId();
   session->DestinationItemRegion = CellRegion();

   //reset editing flag for previous destination
   if (!session->DestinationPage.PointsToNull() && (session->DestinationPage != session->SourcePage))
   {
      session->DestinationPage->IsEditing = false;
   }
   session->DestinationPage.Release();

   if (session->SourcePage.PointsToNull() || !session->SourcePage->WidgetAccessor.isValid())
   {
      return;
   }

   session->DestinationPage = FindCell(GetCurrentTouchCoordinate(), session->DestinationPageNormalizedTouchedCoordinates, session->DestinationPageTouchedCell);

   //touch coordinate is outside any page
   if (session->DestinationPage.PointsToNull())
   {
      return;
   }

   session->DestinationPage->IsEditing = true;
   session->DestinationPageEditingCells.TopLeft = session->DestinationPageTouchedCell;
   session->DestinationPageEditingCells.Span = session->SourceItemSpan;

   //adjust editing cells based on the touch cell in the source item
   if (session->SourceItemTouchedCell != Cell())
   {
      session->DestinationPageEditingCells.TopLeft -= session->SourceItemTouchedCell;
      if (session->DestinationPageEditingCells.TopLeft.Column < 0)
      {
         session->DestinationPageEditingCells.TopLeft.Column = 0;
      }
      if (session->DestinationPageEditingCells.TopLeft.Row < 0)
      {
         session->DestinationPageEditingCells.TopLeft.Row = 0;
      }
   }

   Item* destItem = session->DestinationPage->GetItem(session->DestinationPage->GetItemIndex(session->DestinationPageTouchedCell));

   //dest item can not be the same as source
   if ((destItem != NULL) && (destItem->Id == session->SourceItemId))
   {
      destItem = NULL;
   }

   //drag ended on top of an item => get the touched region from that item
   if (destItem != NULL)
   {
      session->DestinationItemId = destItem->Id;
      session->DestinationItemRegion = GetCellRegion(destItem->Cells, session->DestinationPageNormalizedTouchedCoordinates);
   }
   //drag ended in an empty cell => find an item in the destination rectangle from which to get the region
   else
   {
      CellRect destEditingCells(session->DestinationPageEditingCells);

      for (CellIndexType column = destEditingCells.TopLeft.Column; (destItem == NULL) && (column <= destEditingCells.GetBottomRight().Column); ++column)
      {
         for (CellIndexType row = destEditingCells.TopLeft.Row; (destItem == NULL) && (row <= destEditingCells.GetBottomRight().Row); ++row)
         {
            destItem = session->DestinationPage->GetItem(session->DestinationPage->GetItemIndex(Cell(column, row)));

            //dest item can not be the same as source
            if ((destItem != NULL) && (destItem->Id == session->SourceItemId))
            {
               destItem = NULL;
            }
            if (destItem != NULL)
            {
               session->DestinationItemRegion = GetCellRegion(destItem->Cells, Cell(column, row));
            }
         }
      }
   }
}


/*****************************************************************************/
bool PageEditManager::InitArrangementParams()
{
   Session* session = GetSession();

   if ((session == NULL) || session->SourcePage.PointsToNull())
   {
      return false;
   }

   Album& album = session->_Album;
   Page& srcPage = session->SourcePage.GetSharedInstance();
   Item* srcItem = srcPage.GetItem(srcPage.GetItemIndex(session->SourceItemId));
   if (srcItem == NULL)
   {
      return false;
   }

   //init item editing cells
   for (size_t pageIndex = 0; pageIndex < album.GetPageCount(); ++pageIndex)
   {
      Page::SharedPointer page = album.GetPage(pageIndex);
      if (!page.PointsToNull())
      {
         page->InitItemEditingCells();
      }
   }

   album.InitAlbumGrid(session->AlbumGrid);

   session->DestinationPage.Release();
   session->DestinationItemId = ItemId();

   DetermineDestination();

   if (session->DestinationPage.PointsToNull())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "RearrangeItems touch coordinate %s is not inside a page",
                          HMIBASE_TO_STRING(GetCurrentTouchCoordinate())));
      return false;
   }

   return true;
}


/*****************************************************************************/
bool PageEditManager::RearrangeItems()
{
   Session* session = GetSession();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "RearrangeItems session=%s", HMIBASE_TO_STRING(session)));

   if ((session == NULL) || session->SourcePage.PointsToNull() || !session->SourcePage->IsEditing)
   {
      return false;
   }

   Album& album = session->_Album;
   Page& srcPage = session->SourcePage.GetSharedInstance();
   Item* srcItem = srcPage.GetItem(srcPage.GetItemIndex(session->SourceItemId));
   if (srcItem == NULL)
   {
      return false;
   }

   if (!InitArrangementParams())
   {
      return false;
   }

   Page::SharedPointer destPage = session->DestinationPage;
   if (destPage.PointsToNull())
   {
      return false;
   }

   DefaultPageEditStrategy strategy;
   CellRectMap newCellRectMap;
   bool result = strategy.RearrangeItems(*session, newCellRectMap)
                 && strategy.ValidateNewBounds(*session, newCellRectMap)
                 && strategy.SaveNewBounds(*session, newCellRectMap);

   if (result)
   {
      //persist new bounds in the widgets
      for (Album::PagesType::const_iterator pageIt = album.GetPages().begin(); pageIt != album.GetPages().end(); ++pageIt)
      {
         Page::SharedPointer page = pageIt->second;
         if (!page.PointsToNull())
         {
            PageEditWidget2D* pageEdit = page->WidgetAccessor.getObjectSafely();
            PageEditController2D* controller = (pageEdit != NULL) ? Candera::Dynamic_Cast<PageEditController2D*>(pageEdit->GetController()) : NULL;
            if ((pageEdit != NULL) && (controller != NULL))
            {
               controller->SetItemsBounds(*pageEdit, page.GetSharedInstance(), true);
            }
         }
      }
   }

   return true;
}


/*****************************************************************************/
FeatStd::EventResult::Enum PageEditManager::OnEvent(const FeatStd::Event& e)
{
   /*bool consumed = false;*/

   if (_albums.size() > 0)
   {
      const hmibase::widget::gizmo::EditUpdEvent* editUpdEvent = Candera::Dynamic_Cast<const hmibase::widget::gizmo::EditUpdEvent*>(&e);
      if (editUpdEvent != NULL)
      {
         /*consumed = */OnItemEditUpdEvent(*editUpdEvent);
      }
   }

   return /* consumed ? FeatStd::EventResult::Stop : */ FeatStd::EventResult::Proceed;
}


/*****************************************************************************/
bool PageEditManager::OnItemEditUpdEvent(const hmibase::widget::gizmo::EditUpdEvent& e)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnItemEditUpdEvent action=%u pos=%12s %s",
                       e.GetAction(),
                       HMIBASE_TO_STRING(e.GetPosition()),
                       HMIBASE_TO_STRING_VW(e.GetGizmo())));

   bool consumed = false;

   if (e.GetGizmo() != NULL)
   {
      switch (e.GetAction())
      {
         case gizmo::EditUpdEvent::Start:
            consumed = OnItemEditStart(e);
            break;
         case gizmo::EditUpdEvent::Move:
            consumed = OnItemEditMove(e);
            break;
         case gizmo::EditUpdEvent::End:
            consumed = OnItemEditEnd(e);
            break;
         default:
            break;
      }
   }

   return consumed;
}


/*****************************************************************************/
bool PageEditManager::OnItemEditStart(const hmibase::widget::gizmo::EditUpdEvent& e)
{
   if (e.GetGizmo() == NULL)
   {
      return false;
   }

   ItemId itemId(e.GetGizmo()->GetLegacyName());
   bool result = BeginEditing(itemId);

   if (result && (e.GetGizmo()->GetNode() != NULL))
   {
      //e.GetGizmo()->GetNode()->SetRenderOrderRank(100);//todo: make this configurable
   }

   return result;
}


/*****************************************************************************/
bool PageEditManager::OnItemEditMove(const hmibase::widget::gizmo::EditUpdEvent& /*e*/)
{
   bool result = false;
   Session* session = GetSession();
   if ((session != NULL)
         && (!session->SourcePage.PointsToNull())
         && (session->SourcePage->WidgetAccessor.isValid()))
   {
      DetermineDestination();
      RestartTimer();
      result = true;
   }

   return result;
}


/*****************************************************************************/
bool PageEditManager::OnItemEditEnd(const hmibase::widget::gizmo::EditUpdEvent& e)
{
   bool result = (GetSession() != NULL);

   if (result && (e.GetGizmo() != NULL) && (e.GetGizmo()->GetNode() != NULL))
   {
      e.GetGizmo()->GetNode()->SetRenderOrderRank(0);
   }

   EndEditing();
   return result;
}


/*****************************************************************************/
Candera::Vector2 PageEditManager::GetCurrentTouchCoordinate() const
{
   return (AppViewHandler::getInstance().GetTouchSession() == NULL) ? Candera::Vector2()
          : AS_VECTOR2(AppViewHandler::getInstance().GetTouchSession()->GetTouchInfo().mX,
                       AppViewHandler::getInstance().GetTouchSession()->GetTouchInfo().mY);
}


/*****************************************************************************/
static Candera::Camera2D* GetTouchCamera(Courier::View* view)
{
   if (view != NULL)
   {
      Courier::ViewScene2D* viewScene2D = view->ToViewScene2D();
      if (viewScene2D != NULL)
      {
         const Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
         for (Courier::ViewScene2D::CameraPtrVector::ConstIterator it = cameras.ConstBegin(); it != cameras.ConstEnd(); ++it)
         {
            Candera::Camera2D* camera2D = *it;
            if ((camera2D != NULL) && camera2D->IsRenderingEnabled() && hmibase::input::ShouldUseCameraForTouch(camera2D))
            {
               return camera2D;
            }
         }
      }
   }
   return NULL;
}


/*****************************************************************************/
Cell PageEditManager::PositionToCell(Item& item, const Candera::Vector2& cellSize, const Candera::Vector2& position) const
{
   Cell cell;

   ItemWidget* itemWidget = item.WidgetAccessor.getObjectSafely();
   if ((itemWidget != NULL) && (itemWidget->GetNode() != NULL))
   {
      Candera::Camera2D* camera = GetTouchCamera(itemWidget->GetParentView());
      if (camera != NULL)
      {
         Candera::Vector2 pointInWorldSpace(Candera::Math2D::TransformViewportToScene(*camera, Candera::Math2D::TransformRenderTargetToViewport(*camera, position)));
         Candera::Matrix3x2 itemWorldTransform(itemWidget->GetNode()->GetWorldTransform());
         itemWorldTransform.Inverse();

         Candera::Vector2 adjustedPosition(itemWorldTransform.Multiply(pointInWorldSpace));
         Candera::Vector2 normalizedPosition(adjustedPosition / cellSize);

         CellIndexType column = static_cast<CellIndexType>(normalizedPosition.GetX());
         CellIndexType row = static_cast<CellIndexType>(normalizedPosition.GetY());

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "PositionToCell pos=%12s adjustedPos=%12s normalizedPos=%12s item=%s",
                             HMIBASE_TO_STRING(position),
                             HMIBASE_TO_STRING(adjustedPosition),
                             HMIBASE_TO_STRING(normalizedPosition),
                             HMIBASE_TO_STRING(item.Id)));

         if ((0 <= column) && (column < item.Cells.Span.Column) && (0 <= row) && (row < item.Cells.Span.Row))
         {
            cell.Column = column;
            cell.Row = row;
         }
      }
   }

   return cell;
}


/*****************************************************************************/
Page::SharedPointer PageEditManager::FindCell(const Candera::Vector2& position, Candera::Vector2& normalizedPosition, Cell& cell) const
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "FindCell pos=%12s", HMIBASE_TO_STRING(position)));

   Session* session = GetSession();
   if (session != NULL)
   {
      for (Album::PagesType::const_iterator pageIt = session->_Album.GetPages().begin(); pageIt != session->_Album.GetPages().end(); ++pageIt)
      {
         Page::SharedPointer page = pageIt->second;
         PageEditWidget2D* pageEdit = !page.PointsToNull() ? page->WidgetAccessor.getObjectSafely() : NULL;
         if (!page.PointsToNull() && (pageEdit != NULL))
         {
            Candera::Node2D* itemsNode = (pageEdit->GetItemsNode() != NULL) ? pageEdit->GetItemsNode() : pageEdit->GetNode();
            Candera::Camera2D* camera = GetTouchCamera(pageEdit->GetParentView());

            if ((itemsNode != NULL) && itemsNode->IsEffectiveRenderingEnabled() && (camera != NULL))
            {
               Candera::Vector2 pointInWorldSpace(Candera::Math2D::TransformViewportToScene(*camera, Candera::Math2D::TransformRenderTargetToViewport(*camera, position)));
               Candera::Matrix3x2 pageWorldTransform(itemsNode->GetWorldTransform());
               pageWorldTransform.Inverse();

               Candera::Vector2 adjustedPosition(pageWorldTransform.Multiply(pointInWorldSpace));
               normalizedPosition = adjustedPosition / page->CellSize;

               cell.Column = static_cast<CellIndexType>(normalizedPosition.GetX());
               cell.Row = static_cast<CellIndexType>(normalizedPosition.GetY());

               Candera::Rectangle pageBounds;
               pageBounds.SetSize(page->CellSize * AS_VECTOR2(page->CellCount.Column, page->CellCount.Row));

               if (pageBounds.Contains(adjustedPosition) && (0 <= cell.Column) && (cell.Column < page->CellCount.Column) && (0 <= cell.Row) && (cell.Row < page->CellCount.Row))
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "FindCell adjustedPos=%12s normalizedPos=%12s cell=%6s page=%s",
                                      HMIBASE_TO_STRING(adjustedPosition),
                                      HMIBASE_TO_STRING(normalizedPosition),
                                      HMIBASE_TO_STRING(cell),
                                      HMIBASE_TO_STRING(page->Id)));
                  return page;
               }
            }
         }
      }
   }

   return Page::SharedPointer();
}


/*****************************************************************************/
CellRegion PageEditManager::GetCellRegion(const CellRect& cellRect, const Candera::Vector2& normalizedPosition) const
{
   CellRegion region;

   //todo: make this factor configurable
   static const float RegionSizeFactor = 0.40f; /* 0.0f < factor < 0.5f  */

   Candera::Rectangle destRect(AS_RECTANGLE(cellRect.TopLeft.Column, cellRect.TopLeft.Row, cellRect.Span.Column, cellRect.Span.Row));
   if (destRect.Contains(normalizedPosition))
   {
      if ((normalizedPosition.GetY() - destRect.GetTop()) < (RegionSizeFactor * destRect.GetHeight()))
      {
         region.Vertical = CellRegion::VTop;
      }
      else if ((destRect.GetTop() + destRect.GetHeight() - normalizedPosition.GetY()) < (RegionSizeFactor  * destRect.GetHeight()))
      {
         region.Vertical = CellRegion::VBottom;
      }
      else
      {
         region.Vertical = CellRegion::VCenter;
      }

      if ((normalizedPosition.GetX() - destRect.GetLeft()) < (RegionSizeFactor * destRect.GetWidth()))
      {
         region.Horizontal = CellRegion::HLeft;
      }
      else if ((destRect.GetLeft() + destRect.GetWidth() - normalizedPosition.GetX()) < (RegionSizeFactor * destRect.GetWidth()))
      {
         region.Horizontal = CellRegion::HRight;
      }
      else
      {
         region.Horizontal = CellRegion::HCenter;
      }
   }

   return region;
}


/*****************************************************************************/
CellRegion PageEditManager::GetCellRegion(const CellRect& cellRect, const Cell& cell) const
{
   CellRegion region;

   //todo: make this factor configurable
   static const float RegionSizeFactor = 0.45f; /* 0.0f < factor < 0.5f  */

   if (cellRect.Contains(cell))
   {
      if ((float)(cell.Column - cellRect.TopLeft.Column) < (RegionSizeFactor * (float)(cellRect.Span.Column)))
      {
         region.Horizontal = CellRegion::HLeft;
      }
      else if ((float)(cellRect.GetBottomRight().Column - cell.Column) < (RegionSizeFactor * (float)(cellRect.Span.Column)))
      {
         region.Horizontal = CellRegion::HRight;
      }
      else
      {
         region.Horizontal = CellRegion::HCenter;
      }

      if ((float)(cell.Row - cellRect.TopLeft.Row) < (RegionSizeFactor * (float)(cellRect.Span.Row)))
      {
         region.Vertical = CellRegion::VTop;
      }
      else if ((float)(cellRect.GetBottomRight().Row - cell.Row) < (RegionSizeFactor * (float)(cellRect.Span.Row)))
      {
         region.Vertical = CellRegion::VBottom;
      }
      else
      {
         region.Vertical = CellRegion::VCenter;
      }
   }

   return region;
}


/*****************************************************************************/
bool PageEditManager::onMessage(const Courier::Message& msg)
{
   bool consumed = false;

   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if (timerMsg != NULL)
         {
            consumed = OnTimerExpiredMsg(*timerMsg);
         }
         break;
      }

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
bool PageEditManager::OnTimerExpiredMsg(const TimerExpiredMsg& msg)
{
   bool consumed = false;

   if ((_movementTimer != NULL) && (msg.GetTimer() == _movementTimer))
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTimerExpiredMsg"));

      consumed = true;
      _movementTimer->stop();

      if (RearrangeItems())
      {
         PostUpdateMessage(enEditStatus::Arrange);
      }
   }

   return consumed;
}


/*****************************************************************************/
void PageEditManager::RestartTimer()
{
   Session* session = GetSession();
   if ((session == NULL) || session->SourcePage.PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RestartTimer Session or start page is null!"));
      return;
   }

   PageEditWidget2D* pageEdit = Candera::Dynamic_Cast<PageEditWidget2D*>(session->SourcePage->WidgetAccessor.getObjectSafely());
   PageEditController2D* controller = (pageEdit != NULL) ? Candera::Dynamic_Cast<PageEditController2D*>(pageEdit->GetController()) : NULL;
   if ((pageEdit == NULL) || (controller == NULL))
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RestartTimer Page edit widget or controller is null!"));
      return;
   }

   if (_movementTimer == NULL)
   {
      _movementTimer = FEATSTD_NEW(Util::Timer);
   }

   if (_movementTimer != NULL)
   {
      unsigned int timeout = controller->GetMovementTimerTimeout(*pageEdit);
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "RestartTimer timeout=%u %s",
                          timeout, HMIBASE_TO_STRING_VW(pageEdit)));

      if (timeout == 0)
      {
         timeout = 1000;
      }

      //configure timer and start it
      _movementTimer->setName("PageEditMovement", "");
      _movementTimer->setTimeout(0, timeout);
      _movementTimer->start();
   }
}


/*****************************************************************************/
void PageEditManager::StopTimer()
{
   if (_movementTimer != NULL)
   {
      _movementTimer->stop();
   }
}


/*****************************************************************************/
Cell PageEditManager::PositionToCell(const Candera::Vector2& position, const Candera::Vector2& cellSize) const
{
   if ((cellSize.GetX() > 0) && (cellSize.GetY() > 0))
   {
      Candera::Vector2 floatCell(position / cellSize);

      return Cell(static_cast<CellIndexType>(floatCell.GetX()), static_cast<CellIndexType>(floatCell.GetY()));
   }
   return Cell();
}


/*****************************************************************************/
CellRect PageEditManager::BoundsToCellRect(const Candera::Vector2& position, const Candera::Vector2& size, const Candera::Vector2& cellSize) const
{
   if ((cellSize.GetX() > 1.0f) && (cellSize.GetY() > 1.0f) && (size.GetX() >= 0.0f) && (size.GetY() >= 0.0f))
   {
      Candera::Vector2 adjustedTopLeft;
      if (position.GetX() > 0.0f)
      {
         adjustedTopLeft.SetX(position.GetX() + cellSize.GetX() / 2.0f);
      }
      else
      {
         adjustedTopLeft.SetX(position.GetX() - cellSize.GetX() / 2.0f);
      }
      if (position.GetY() > 0.0f)
      {
         adjustedTopLeft.SetY(position.GetY() + cellSize.GetY() / 2.0f);
      }
      else
      {
         adjustedTopLeft.SetY(position.GetY() - cellSize.GetY() / 2.0f);
      }

      Candera::Vector2 floatTopLeftCell(adjustedTopLeft / cellSize);
      Cell topLeftCell(static_cast<CellIndexType>(floatTopLeftCell.GetX()), static_cast<CellIndexType>(floatTopLeftCell.GetY()));

      Candera::Vector2 adjustedSize(size + cellSize / 2.0f);
      Candera::Vector2 floatCellSpan(adjustedSize / cellSize);
      Cell cellSpan(static_cast<CellIndexType>(floatCellSpan.GetX()), static_cast<CellIndexType>(floatCellSpan.GetY()));

      return CellRect(topLeftCell, cellSpan);
   }
   return CellRect();
}


/*****************************************************************************/
void PageEditManager::PostUpdateMessage(enEditStatus::Enum status)
{
   Session* session = GetSession();
   PageEditData::SharedPointer data(FEATSTD_NEW(PageEditData));
   if (!data.PointsToNull() && (session != NULL) && !session->SourcePage.PointsToNull() && !session->DestinationPage.PointsToNull())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "\n\n"));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostUpdateMessage album=%u status=%u session=%s",
                          session->_Album.GetId(), status, HMIBASE_TO_STRING(*session)));

      data->AlbumId = session->_Album.GetId();

      data->SourceItemId = session->SourceItemId;
      if (!session->SourcePage.PointsToNull())
      {
         data->SourcePageId = session->SourcePage->Id;
      }

      data->DestinationItemId = session->DestinationItemId;
      if (!session->DestinationPage.PointsToNull())
      {
         data->DestinationPageId = session->DestinationPage->Id;
      }

      data->Pages.Reserve(session->_Album.GetPages().size());
      for (size_t i = 0; i < session->_Album.GetPages().size(); ++i)
      {
         data->Pages.Add(PageEditPage());
         Page::SharedPointer page = session->_Album.GetPage(i);
         if (!page.PointsToNull())
         {
            PageEditPage& pageOut = data->Pages[i];

            pageOut.Id = page->Id;

            pageOut.Items.Reserve(page->Items.size());
            for (size_t j = 0; j < page->Items.size(); ++j)
            {
               Item& item = page->Items[j];

               PageEditItem itemOut;
               itemOut.Id = item.Id;
               itemOut.ExternalId = item.ExternalId;
               itemOut.Cells = AS_RECTANGLE(item.EditingCells.TopLeft.Column, item.EditingCells.TopLeft.Row, item.EditingCells.Span.Column, item.EditingCells.Span.Row);

               pageOut.Items.Add(itemOut);
            }

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostUpdateMessage index=%u page=%s", i, HMIBASE_TO_STRING(pageOut.Id)));
            if (IsTraceEnabled())
            {
               PrintBigText(HMIBASE_TO_STRING(pageOut));
            }
         }
      }

      PageEditUpdMsg* msg = COURIER_MESSAGE_NEW(PageEditUpdMsg)(session->_Album.GetId(), data, status);
      if (msg != NULL)
      {
         msg->Post();
      }
   }
}


/*****************************************************************************/
bool PageEditManager::IsTraceEnabled() const
{
   return etg_bIsTraceActiveDouble(((0xFFFFu & static_cast<etg_tU16>(TR_CLASS_HMI_WIDGET_PAGEEDIT)) << 16) | static_cast<etg_tU16>(ETG_LEVEL_USER_4), static_cast<etg_tU16>(APP_TRACECLASS_ID()));
}


/*****************************************************************************/
void PageEditManager::PrintBigText(char const* text) const
{
   if (text != NULL)
   {
#ifdef _MSC_VER
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%s", text));
#else

#if !defined(ETG_C_TRACE_MAX)
#define ETG_C_TRACE_MAX 200
#endif

      const size_t maxLineLength = ((ETG_C_TRACE_MAX) > 20) ? (ETG_C_TRACE_MAX) - 20/*overhead*/ : 100;

      ::std::istringstream iss(text);
      ::std::ostringstream oss;

      char c;
      while (iss.get(c))
      {
         if ((c == '\n') || (oss.tellp() >= (unsigned int)maxLineLength))
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%s", oss.str().c_str()));

            //reset the output stream
            oss.str(::std::string());
         }

         if (c != '\n')
         {
            oss << c;
         }
      }

      if (oss.tellp() > 0)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%s", oss.str().c_str()));
      }

#endif
   }
}


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const PageEditItem& value)
{
   out << "\nid=" << value.Id;

   if (value.ExternalId != ItemId())
   {
      out << " extId=" << value.ExternalId;
   }

   out << " cells=" << value.Cells;

   return out;
}


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const PageEditPage& value)
{
   out << "id=" << value.Id;
   out << " items=" << value.Items;

   return out;
}


}
}


}
