/* ***************************************************************************************
* FILE:          PageEditData.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditData.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "PageEditData.h"

#include <FeatStd/Platform/Memory.h>
#include <Trace/ToString.h>
#include <Widgets/2D/Gizmo/GizmoWidget2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_PAGEEDIT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageEditData.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace pageedit {


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const Item& value)
{
   out << "\n";
   out << "name=" << value.Id;

   if (value.ExternalId != ItemId())
   {
      out << " external=" << value.ExternalId;
   }

   out << " cells=" << value.Cells;

   if (value.EditingCells != CellRect())
   {
      out << " editingCells=" << value.EditingCells;
   }

   return out;
}


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const Page& value)
{
   //out << "\n";
   out << "name=" << value.Id;

   if (value.CellCount != Cell())
   {
      out << " cellCount=" << value.CellCount;
   }

   if (value.CellSize != Candera::Vector2())
   {
      out << " cellSize=" << value.CellSize;
   }

   if (value.PageSize != Candera::Vector2())
   {
      out << " pageSize=" << value.PageSize;
   }

   if (value.Items.size() > 0)
   {
      out << " items=" << value.Items;
   }

   return out;
}


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const Album& value)
{
   //out << "\n";
   out << "id=" << value.GetId();
   out << " pages=" << value.GetPages();

   return out;
}


/*****************************************************************************/
::std::ostream& operator<<(::std::ostream& out, const Session& value)
{
   out << "album=" << value._Album.GetId();

   out << " source=[";
   out << "item=" << value.SourceItemId;
   out << " touchCell=[" << value.SourceItemTouchedCell << "]";
   if (!value.SourcePage.PointsToNull())
   {
      out << " page=" << value.SourcePage->Id;
   }
   out << "]";

   out << " destination=[";
   out << "item=" << value.DestinationItemId;
   if (!value.DestinationPage.PointsToNull())
   {
      out << " page=" << value.DestinationPage->Id;
   }
   out << "]";

   return out;
}


/*****************************************************************************/
Page::Page(const PageId& id) : Id(id), ItemsValid(false), CellConfigValid(false), IsEditing(false)
{
}


/*****************************************************************************/
Page::SharedPointer Page::Create(const PageId& id)
{
   return SharedPointer(FEATSTD_NEW(Page)(id));
}


/*****************************************************************************/
ItemIndexType Page::GetItemIndex(const ItemId& itemId) const
{
   for (size_t index = 0; index < Items.size(); ++index)
   {
      if (itemId == Items[index].Id)
      {
         return static_cast<ItemIndexType>(index);
      }
   }
   return -1;
}


/*****************************************************************************/
ItemIndexType Page::GetItemIndex(const ItemWidget* widget) const
{
   for (size_t index = 0; index < Items.size(); ++index)
   {
      if (widget == Items[index].WidgetAccessor.getObjectSafely())
      {
         return static_cast<ItemIndexType>(index);
      }
   }
   return -1;
}


/*****************************************************************************/
ItemIndexType Page::GetItemIndex(const Cell& cell) const
{
   for (size_t index = 0; index < Items.size(); ++index)
   {
      if (Items[index].EditingCells.Contains(cell))
      {
         return static_cast<ItemIndexType>(index);
      }
   }
   return -1;
}


/*****************************************************************************/
Item* Page::GetItem(ItemIndexType index)
{
   return ((index >= 0) && (index < static_cast<ItemIndexType>(Items.size()))) ? &Items[static_cast<size_t>(index)] : NULL;
}


/*****************************************************************************/
const Item* Page::GetItem(ItemIndexType index) const
{
   return ((index >= 0) && (index < static_cast<ItemIndexType>(Items.size()))) ? &Items[static_cast<size_t>(index)] : NULL;
}


/*****************************************************************************/
bool Page::AreItemAccessorsValid() const
{
   for (ItemsType::const_iterator it = Items.begin(); it != Items.end(); ++it)
   {
      const Item& item = (*it);
      if ((item.WidgetAccessor.operator->() != NULL) && !item.WidgetAccessor.isValid())
      {
         return false;
      }
   }
   return true;
}


/*****************************************************************************/
void Page::ResetEditingInfo()
{
   IsEditing = false;
}


/*****************************************************************************/
void Page::InitItemEditingCells()
{
   //initialize editing cells
   for (ItemsType::iterator itemIt = Items.begin(); itemIt != Items.end(); ++itemIt)
   {
      Item& item = *itemIt;

      item.EditingCells = item.Cells;
      item.ExternalId = ItemId();
   }
}


/*****************************************************************************/
Album::Album(const AlbumId& id) : _id(id)
{
}


/*****************************************************************************/
const AlbumId& Album::GetId() const
{
   return _id;
}


/*****************************************************************************/
const Album::PagesType& Album::GetPages() const
{
   return _pages;
}


/*****************************************************************************/
Album::PagesType& Album::GetPages()
{
   return _pages;
}


/*****************************************************************************/
size_t Album::GetPageCount() const
{
   return _pages.size();
}


/*****************************************************************************/
Page::SharedPointer Album::GetPage(size_t index) const
{
   PagesType::const_iterator it = _pages.begin();
   std::advance(it, index);
   return it->second;
}


/*****************************************************************************/
Page::SharedPointer Album::GetPage(const PageId& pageId) const
{
   if (pageId != PageId())
   {
      PagesType::const_iterator it = _pages.find(pageId);
      if (it != _pages.end())
      {
         return it->second;
      }
   }

   return Page::SharedPointer();
}


/*****************************************************************************/
Page::SharedPointer Album::AddNewPage(const PageId& pageId)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AddPage pageId=%s", HMIBASE_TO_STRING(pageId)));

   if (pageId == PageId())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AddPage page id '%s' is invalid!", HMIBASE_TO_STRING(pageId)));
      return Page::SharedPointer();
   }

   if (!GetPage(pageId).PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AddPage page with id '%s' already exists!", HMIBASE_TO_STRING(pageId)));
      return Page::SharedPointer();
   }

   Page::SharedPointer page = Page::Create(pageId);
   _pages[pageId] = page;
   return page;
}


/*****************************************************************************/
void Album::RemovePage(const PageId& pageId)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "RemovePage pageId=%s", HMIBASE_TO_STRING(pageId)));

   PagesType::iterator it = _pages.find(pageId);
   if (it != _pages.end())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "RemovePage found page"));
      _pages.erase(it);
   }
}


/*****************************************************************************/
Page::SharedPointer Album::GetPage(const Cell& albumCell) const
{
   CellIndexType albumColumnCount = 0;
   for (Album::PagesType::const_iterator pageIt = GetPages().begin(); pageIt != GetPages().end(); ++pageIt)
   {
      Page::SharedPointer page = pageIt->second;
      if (!page.PointsToNull())
      {
         CellRect pageCellRect(Cell(albumColumnCount, 0), page->CellCount);
         if (pageCellRect.Contains(albumCell))
         {
            return page;
         }

         albumColumnCount += page->CellCount.Column;
      }
   }

   return Page::SharedPointer();
}


/*****************************************************************************/
Cell Album::GetCellCount() const
{
   Cell albumCellCount;
   for (Album::PagesType::const_iterator pageIt = GetPages().begin(); pageIt != GetPages().end(); ++pageIt)
   {
      Page::SharedPointer page = pageIt->second;
      if (!page.PointsToNull())
      {
         albumCellCount.Column += page->CellCount.Column;
         if (albumCellCount.Row < page->CellCount.Row)
         {
            albumCellCount.Row = page->CellCount.Row;
         }
      }
   }

   return albumCellCount;
}


/*****************************************************************************/
Cell Album::GetPageOffset(Page::SharedPointer pageToCheck) const
{
   CellIndexType albumColumnCount = 0;
   for (Album::PagesType::const_iterator pageIt = GetPages().begin(); pageIt != GetPages().end(); ++pageIt)
   {
      Page::SharedPointer page = pageIt->second;
      if (!page.PointsToNull())
      {
         if (page == pageToCheck)
         {
            return Cell(albumColumnCount, 0);
         }

         albumColumnCount += page->CellCount.Column;
      }
   }

   return Cell();
}


/*****************************************************************************/
PageIndexType Album::GetPageIndex(Page::SharedPointer pageToCheck) const
{
   for (size_t pageIndex = 0; pageIndex < GetPages().size(); ++pageIndex)
   {
      Page::SharedPointer page = GetPage(pageIndex);
      if (!page.PointsToNull() && page == pageToCheck)
      {
         return static_cast<PageIndexType>(pageIndex);
      }
   }

   return -1;
}


/*****************************************************************************/
void Album::InitAlbumGrid(CellGrid& albumGrid) const
{
   Cell albumCellCount(GetCellCount());
   albumGrid.Init(albumCellCount.Column, albumCellCount.Row);

   for (size_t pageIndex = 0; pageIndex < GetPages().size(); ++pageIndex)
   {
      Page::SharedPointer page = GetPage(pageIndex);
      if (!page.PointsToNull())
      {
         Cell pageOffset = GetPageOffset(page);

         for (size_t itemIndex = 0; itemIndex < page->Items.size(); ++itemIndex)
         {
            CellRect itemCells(page->Items[itemIndex].Cells);
            itemCells.TopLeft += pageOffset;

            albumGrid.SetCellData(itemCells, PageItemIndex(static_cast<PageIndexType>(pageIndex), static_cast<ItemIndexType>(itemIndex)));
         }
      }
   }
}


/*****************************************************************************/
Session::Session(Album& album) : _Album(album)
{
}


}
}


}
