/* ***************************************************************************************
* FILE:          PageEditController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "PageEditController2D.h"
#include "PageEditWidget2D.h"
#include "PageEditStrategy.h"
#include "PageEditManager.h"

#include <FeatStd/Platform/Memory.h>
#include <Trace/ToString.h>
#include <Widgets/2D/Gizmo/GizmoWidget2D.h>
#include <Widgets/2D/Gizmo/GizmoController2D.h>
#include <Widgets/utils/WidgetFunctors.h>
#include <Widgets/utils/WidgetTraverser.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_PAGEEDIT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageEditController2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace pageedit {

using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(PageEditController2DData)
CANDERA_RTTI_DEFINITION(PageEditController2D)
CANDERA_RTTI_DEFINITION(DefaultPageEditController2DData)
CANDERA_RTTI_DEFINITION(DefaultPageEditController2D)
CANDERA_RTTI_DEFINITION(ExternalItemEditUpdEvent)


#define AS_VECTOR2(x, y) Candera::Vector2(static_cast<FeatStd::Float>(x), static_cast<FeatStd::Float>(y))
#define AS_MARGIN(left, top, right, bottom) Candera::Margin(static_cast<FeatStd::Int16>(left), static_cast<FeatStd::Int16>(top), static_cast<FeatStd::Int16>(right), static_cast<FeatStd::Int16>(bottom))
#define AS_RECTANGLE(x, y, width, height) Candera::Rectangle(static_cast<FeatStd::Float>(x), static_cast<FeatStd::Float>(y), static_cast<FeatStd::Float>(width), static_cast<FeatStd::Float>(height))


/*****************************************************************************/
DefaultPageEditController2D::DefaultPageEditController2D() : _movementTimerTimeout(250)
{
}


/*****************************************************************************/
DefaultPageEditController2D::~DefaultPageEditController2D()
{
}


/*****************************************************************************/
unsigned int DefaultPageEditController2D::GetMovementTimerTimeout(PageEditWidget2D& /*pageEdit*/) const
{
   return _movementTimerTimeout;
}


/*****************************************************************************/
void DefaultPageEditController2D::SetMovementTimerTimeout(unsigned int timeout)
{
   _movementTimerTimeout = timeout;
}


/*****************************************************************************/
hmibase::widget::WidgetControllerData* DefaultPageEditController2D::CreateData(DelegateWidget&)
{
   return CANDERA_NEW(DefaultPageEditController2DData);
}


/*****************************************************************************/
void DefaultPageEditController2D::Attach(DelegateWidget& delegateWidget)
{
   Base::Attach(delegateWidget);

   //PageEditWidget2D* pageEdit = Candera::Dynamic_Cast<PageEditWidget2D*>(&delegateWidget);
   //DefaultPageEditController2DData* pageEditData = Candera::Dynamic_Cast<DefaultPageEditController2DData*>(delegateWidget.GetControllerData());
   //if ((pageEdit != NULL) && (pageEditData != NULL))
   //{
   //   pageEditData->PageData = PageEditManager::GetInstance().GetPage(Courier::Identifier(pageEdit->GetLegacyName()));
   //   if (pageEditData->PageData.PointsToNull())
   //   {
   //      pageEditData->PageData = PageEditManager::GetInstance().AddNewPage(Courier::Identifier(pageEdit->GetLegacyName()));
   //   }
   //
   //   if (!pageEditData->PageData.PointsToNull())
   //   {
   //      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Attach page=%p %s",
   //                          pageEditData->PageData.GetPointerToSharedInstance(),
   //                          HMIBASE_TO_STRING_VW(pageEdit)));
   //
   //      pageEditData->PageData->Name = pageEdit->GetLegacyName();
   //      pageEditData->PageData->WidgetAccessor = PageWidgetAccessor(pageEdit);
   //   }
   //}
}


/*****************************************************************************/
void DefaultPageEditController2D::Detach(DelegateWidget& delegateWidget)
{
   UnregisterPage(delegateWidget);

   Base::Detach(delegateWidget);
}


/*****************************************************************************/
void DefaultPageEditController2D::UnregisterPage(DelegateWidget& delegateWidget)
{
   PageEditWidget2D* pageEdit = Candera::Dynamic_Cast<PageEditWidget2D*>(&delegateWidget);
   DefaultPageEditController2DData* pageEditData = Candera::Dynamic_Cast<DefaultPageEditController2DData*>(delegateWidget.GetControllerData());
   if ((pageEdit != NULL) && (pageEdit->GetNode() != NULL) && (pageEditData != NULL))
   {
      Album* album = PageEditManager::GetInstance().GetAlbum(pageEdit->GetAlbumId());
      if (album != NULL)
      {
         ItemId pageId(pageEdit->GetLegacyName());
         album->RemovePage(pageId);
      }

      pageEditData->PageData->WidgetAccessor.release();
   }
}


/*****************************************************************************/
void DefaultPageEditController2D::ValidateCellSize(PageEditWidget2D& pageEdit, Page& page)
{
   Candera::Vector2 cellSize(pageEdit.GetCellSize());
   Cell cellCount(static_cast<CellIndexType>(pageEdit.GetColumnCount()), static_cast<CellIndexType>(pageEdit.GetRowCount()));

   if ((cellSize.GetX() < 1.0f) || (cellSize.GetY() < 1.0f) || (cellCount.Column < 1) || (cellCount.Row < 1))
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ValidateCellSize RowCount=[%d], ColumnCount=[%d], CellSize=%12s. They must be equal or greater than 1! %s",
                         pageEdit.GetRowCount(),
                         pageEdit.GetColumnCount(),
                         HMIBASE_TO_STRING(pageEdit.GetCellSize()),
                         HMIBASE_TO_STRING_VW(&pageEdit)));

      if (cellCount.Column < 1)
      {
         cellCount.Column = 1;
      }
      if (cellCount.Row < 1)
      {
         cellCount.Row = 1;
      }
      if (cellSize.GetX() < 1.0f)
      {
         cellSize.SetX(1.0f);
      }
      if (cellSize.GetY() < 1.0f)
      {
         cellSize.SetY(1.0f);
      }
   }

   Candera::Vector2 pageSize(cellSize * AS_VECTOR2(cellCount.Column, cellCount.Row));
   if ((page.CellSize != cellSize) || (page.CellCount != cellCount) || (page.PageSize != pageSize))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ValidateCellSize columnsXrows=[%d,%d] pageSize=%12s cellSize=%12s %s",
                          cellCount.Column,
                          cellCount.Row,
                          HMIBASE_TO_STRING(pageSize),
                          HMIBASE_TO_STRING(cellSize),
                          HMIBASE_TO_STRING_VW(&pageEdit)));

      page.CellSize = cellSize;
      page.CellCount = cellCount;
      page.PageSize = pageSize;
      page.ItemsValid = false;
   }

   Candera::Node2D* itemsNode = (pageEdit.GetItemsNode() != NULL) ? pageEdit.GetItemsNode() : pageEdit.GetNode();
   if ((itemsNode != NULL) && (Candera::Layouter::GetSize(*itemsNode) != page.PageSize))
   {
      Candera::Layouter::SetSize(*itemsNode, page.PageSize);
      Candera::Layouter::InvalidateLayout(itemsNode);
      page.ItemsValid = false;
   }
}


/*****************************************************************************/
void DefaultPageEditController2D::ProcessItemCellsConfig(PageEditWidget2D& pageEdit, Page& page)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProcessItemCellsConfig %50s itemCells=%s",
                       HMIBASE_TO_STRING_VW(&pageEdit),
                       HMIBASE_TO_STRING(pageEdit.GetItemCells())));

   page.Items.reserve(pageEdit.GetItemCells().GetCount());

   size_t index = 0;
   for (; index < static_cast<size_t>(pageEdit.GetItemCells().GetCount()); ++index)
   {
      if (page.Items.size() <= index)
      {
         page.Items.push_back(Item());
      }

      Candera::Rectangle rect(pageEdit.GetItemCells().Get(static_cast<FeatStd::UInt>(index)));
      CellRect cellRect(static_cast<CellIndexType>(rect.GetLeft()), static_cast<CellIndexType>(rect.GetTop()), static_cast<CellIndexType>(rect.GetWidth()), static_cast<CellIndexType>(rect.GetHeight()));
      if ((cellRect.Span.Column <= 0) || (cellRect.Span.Row <= 0))
      {
         cellRect.Span = Cell();
      }

      page.Items[index].Cells = cellRect;

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "ProcessItemCellsConfig index=%u cells=%s",
                          index,
                          HMIBASE_TO_STRING(page.Items[index].Cells)));
   }

   for (; index < page.Items.size(); ++index)
   {
      page.Items[index].Cells = CellRect();
   }
}


/*****************************************************************************/
void DefaultPageEditController2D::CollectItems(PageEditWidget2D& pageEdit, Page& page)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "CollectItems %s",
                       HMIBASE_TO_STRING_VW(&pageEdit)));

   std::vector<ItemWidget*> widgets;
   SearchItems(pageEdit, widgets);

   //reset the owner of the old items if they are missing or got replaced
   for (size_t index = 0; index < page.Items.size(); ++index)
   {
      Item& item = page.Items[index];
      ItemWidget* oldWidget = item.WidgetAccessor.getObjectSafely();
      if ((oldWidget != NULL) && ((index >= widgets.size()) || (widgets[index] != oldWidget)))
      {
         item.Id = ItemId();
         item.WidgetAccessor = ItemWidgetAccessor();
      }
   }

   page.Items.reserve(widgets.size());
   for (size_t index = 0; index < widgets.size(); ++index)
   {
      if (page.Items.size() <= index)
      {
         page.Items.push_back(Item());
      }

      Item& item = page.Items[index];

      ItemWidget* widget = widgets[index];
      ItemWidget* oldWidget = item.WidgetAccessor.getObjectSafely();
      if (widget != oldWidget)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "CollectItems index=%u item=%s",
                             index,
                             HMIBASE_TO_STRING_W(widget)));

         if (widget != NULL)
         {
            item.Id = widget->GetLegacyName();
            item.WidgetAccessor = ItemWidgetAccessor(widget);
            widget->SetOwnedByPageEdit(true);
         }
         else
         {
            item.Id = ItemId();
            item.WidgetAccessor = ItemWidgetAccessor();
         }
      }
   }

   //remove extra entries which are not needed anymore
   if (page.Items.size() != widgets.size())
   {
      page.Items.resize(widgets.size());
   }
}


/*****************************************************************************/
void DefaultPageEditController2D::SetItemsBounds(PageEditWidget2D& pageEdit, Page& page, bool useEditingCells)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetItemBounds useEditingCells=%u %s",
                       useEditingCells,
                       HMIBASE_TO_STRING_VW(&pageEdit)));

   Session* session = PageEditManager::GetInstance().GetSession();
   ItemId srcItemId = (session != NULL) ? session->SourceItemId : ItemId();

   for (size_t index = 0; index < page.Items.size(); ++index)
   {
      Item& item = page.Items[index];
      if (!useEditingCells || (item.Id != srcItemId))
      {
         SetItemBounds(pageEdit, page, item, useEditingCells ? item.EditingCells : item.Cells);
      }
   }
   pageEdit.Invalidate();
}


/*****************************************************************************/
void DefaultPageEditController2D::SetItemBounds(PageEditWidget2D& /*pageEdit*/, Page& page, Item& item, const CellRect& cells)
{
   ItemWidget* widget = item.WidgetAccessor.getObjectSafely();
   if (widget != NULL)
   {
      if ((page.CellSize == Candera::Vector2()) || (cells.Span.Row == 0) || (cells.Span.Column == 0))
      {
         widget->SetVisible(false);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetItemBounds invisible item=%s",
                             HMIBASE_TO_STRING_W(widget)));
      }
      else
      {
         widget->SetVisible(true);

         Candera::Vector2 position(AS_VECTOR2(cells.TopLeft.Column, cells.TopLeft.Row) * page.CellSize);
         widget->SetPosition(position);

         Candera::Vector2 size(AS_VECTOR2(cells.Span.Column, cells.Span.Row) * page.CellSize);
         widget->SetSize(size);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetItemBounds visible rect=%24s cells=%13s item=%s",
                             HMIBASE_TO_STRING(Candera::Rectangle(position, size)),
                             HMIBASE_TO_STRING(cells),
                             HMIBASE_TO_STRING_W(widget)));
      }
   }
}


/*****************************************************************************/
static bool IsPageUsedByOtherWidget(const PageEditWidget2D& pageEdit, const Page::SharedPointer& page)
{
   return !page.PointsToNull()
          && (page->WidgetAccessor.getObjectSafely() != NULL)
          && (page->WidgetAccessor.getObjectSafely() != &pageEdit);
}


/*****************************************************************************/
void DefaultPageEditController2D::Update(DelegateWidget& delegateWidget)
{
   Base::Update(delegateWidget);

   //don't register page for list item templates
   if (delegateWidget.IsItemTemplate())
   {
      UnregisterPage(delegateWidget);
      return;
   }

   PageEditWidget2D* pageEdit = Candera::Dynamic_Cast<PageEditWidget2D*>(&delegateWidget);
   DefaultPageEditController2DData* pageEditData = Candera::Dynamic_Cast<DefaultPageEditController2DData*>(delegateWidget.GetControllerData());
   if ((pageEdit != NULL) && (pageEdit->GetNode() != NULL) && (pageEditData != NULL))
   {
      //validate existing page
      if (IsPageUsedByOtherWidget(*pageEdit, pageEditData->PageData))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update Detected other widget %50s using the existing page %50s! %s",
                            HMIBASE_TO_STRING_W(pageEditData->PageData->WidgetAccessor.getObjectSafely()),
                            HMIBASE_TO_STRING(pageEditData->PageData.GetPointerToSharedInstance()),
                            HMIBASE_TO_STRING_W(pageEdit)));
         pageEditData->PageData.Release();
      }

      ItemId pageId(pageEdit->GetLegacyName());

      //get or create the page
      if (pageEditData->PageData.PointsToNull())
      {
         Album* album = PageEditManager::GetInstance().GetOrCreateAlbum(pageEdit->GetAlbumId());
         if (album != NULL)
         {
            pageEditData->PageData = album->GetPage(pageId);
            if (pageEditData->PageData.PointsToNull())
            {
               pageEditData->PageData = album->AddNewPage(pageId);
            }
            else
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Update Reuse page %100s %s",
                                   HMIBASE_TO_STRING(pageEditData->PageData.GetPointerToSharedInstance()),
                                   HMIBASE_TO_STRING_W(pageEdit)));
            }
         }

         //validate new page
         if (IsPageUsedByOtherWidget(*pageEdit, pageEditData->PageData))
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update Detected other widget %50s using the new page %50s! %s",
                               HMIBASE_TO_STRING_W(pageEditData->PageData->WidgetAccessor.getObjectSafely()),
                               HMIBASE_TO_STRING(pageEditData->PageData.GetPointerToSharedInstance()),
                               HMIBASE_TO_STRING_W(pageEdit)));
            //pageEditData->PageData.Release();
         }
      }

      if (!pageEditData->PageData.PointsToNull())
      {
         Page& page = pageEditData->PageData.GetSharedInstance();

         page.Id = pageId;
         if (page.WidgetAccessor.getObjectSafely() != pageEdit)
         {
            page.WidgetAccessor = PageWidgetAccessor(pageEdit);
         }

         ValidateCellSize(*pageEdit, page);
         SetEditingCellsNodeBounds(*pageEdit, page);

         if (!page.IsEditing)
         {
            if (!page.CellConfigValid || !page.ItemsValid || !page.AreItemAccessorsValid())
            {
               page.CellConfigValid = true;
               page.ItemsValid = true;

               CollectItems(*pageEdit, page);

               if (pageEdit->GetArrangeItems())
               {
                  ProcessItemCellsConfig(*pageEdit, page);
                  SetItemsBounds(*pageEdit, page, false);
               }
            }
            else
            {
               if (pageEdit->GetArrangeItems())
               {
                  //todo: check why correction of visibility is required
                  for (Page::ItemsType::iterator it = page.Items.begin(); it != page.Items.end(); ++it)
                  {
                     Item& item = *it;
                     ItemWidget* itemWidget = item.WidgetAccessor.getObjectSafely();
                     bool visible = (item.Cells.Span != Cell());
                     if ((itemWidget != NULL) && (itemWidget->GetNode2D() != NULL) && (itemWidget->GetNode2D()->IsRenderingEnabled() != visible))
                     {
                        itemWidget->GetNode2D()->SetRenderingEnabled(visible);
                     }
                  }
               }
            }
         }

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "\n\n"));
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update %s", HMIBASE_TO_STRING_VW(pageEdit)));
         //ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update page=%s", HMIBASE_TO_STRING(page)));

         if (PageEditManager::GetInstance().IsTraceEnabled())
         {
            PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(page));
         }
      }
   }
}


/*****************************************************************************/
class PageEditContentFinderCallback : public WidgetCheckCallback
{
   public:
      PageEditContentFinderCallback(Candera::Node2D* groupNode, std::vector<ItemWidget*>& items) : _groupNode(groupNode), _items(items) {}
      ~PageEditContentFinderCallback()
      {
         _groupNode = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         ItemWidget* item = Candera::Dynamic_Cast<ItemWidget*>(widget);
         if ((item != NULL) && (item->GetNode() != NULL)
               && hmibase::widget::utils::CompareUtils::areAncestorAndDescendent(_groupNode, item->GetNode())
               && (_set.find(item) == _set.end()))
         {
            _items.push_back(item);
            _set.insert(item);
         }

         //continue searching through all the widgets to find the entire content
         return false;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(PageEditContentFinderCallback);

      Candera::Node2D* _groupNode;
      std::set<ItemWidget*> _set;//to determine quickly if was found already (when used in lists it can be checked twice)
      std::vector<ItemWidget*>& _items;//to have the widgets in the proper order
};


/*****************************************************************************/
bool DefaultPageEditController2D::SearchItems(PageEditWidget2D& pageEdit, std::vector<ItemWidget*>& items)
{
   Candera::Node2D* itemsNode = (pageEdit.GetItemsNode() != NULL) ? pageEdit.GetItemsNode() : pageEdit.GetNode();
   if (itemsNode != NULL)
   {
      PageEditContentFinderCallback finderCallback(itemsNode, items);
      //hmibase::widget::utils::MessageUtils::distribute(hmibase::widget::utils::MessageUtils::getSceneContext(&buttonGroup), WidgetCheckReqMsg(&finderCallback));
      hmibase::widget::utils::MessageUtils::distributeToDescendants(hmibase::widget::utils::MessageUtils::getSceneContext(&pageEdit), *itemsNode, WidgetCheckReqMsg(&finderCallback));
      return true;
   }
   return false;
}


/*****************************************************************************/
void DefaultPageEditController2D::OnChanged(DelegateWidget& delegateWidget, FeatStd::UInt32 propertyId)
{
   Base::OnChanged(delegateWidget, propertyId);

   switch (propertyId)
   {
      case PageEditWidget2D::ArrangeItemsPropertyId:
      case PageEditWidget2D::CellSizePropertyId:
      case PageEditWidget2D::ColumnCountPropertyId:
      case PageEditWidget2D::ItemCellsPropertyId:
      case PageEditWidget2D::ItemsNodePropertyId:
      case PageEditWidget2D::RowCountPropertyId:
      {
         DefaultPageEditController2DData* pageEditData = Candera::Dynamic_Cast<DefaultPageEditController2DData*>(delegateWidget.GetControllerData());
         if ((pageEditData != NULL) && !pageEditData->PageData.PointsToNull())
         {
            pageEditData->PageData->CellConfigValid = false;
         }
         break;
      }
      default:
         break;
   }
}


/*****************************************************************************/
void DefaultPageEditController2D::SetEditingCellsNodeBounds(PageEditWidget2D& pageEdit, Page& page)
{
   if ((pageEdit.GetEditingCellsNode() != NULL) && (pageEdit.GetParentView() != NULL)/*IsNotRunningInSceneComposer*/)
   {
      Session* session = PageEditManager::GetInstance().GetSession();

      Candera::Rectangle editingCellBounds;
      if ((session != NULL) && (session->DestinationPage.GetPointerToSharedInstance() == &page))
      {
         editingCellBounds = Candera::Rectangle(AS_VECTOR2(session->DestinationPageEditingCells.TopLeft.Column, session->DestinationPageEditingCells.TopLeft.Row) * page.CellSize,
                                                AS_VECTOR2(session->DestinationPageEditingCells.Span.Column, session->DestinationPageEditingCells.Span.Row) * page.CellSize);
      }

      if (page.IsEditing && (editingCellBounds.GetWidth() > 0.0f) && (editingCellBounds.GetHeight() > 0.0f))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetEditingCellsNodeBounds rect=%24s",
                             HMIBASE_TO_STRING(editingCellBounds)));

         pageEdit.GetEditingCellsNode()->SetRenderingEnabled(true);

         pageEdit.GetEditingCellsNode()->SetPosition(editingCellBounds.GetPosition());

         Candera::Margin margin(static_cast<FeatStd::Int16>(editingCellBounds.GetLeft()),
                                static_cast<FeatStd::Int16>(editingCellBounds.GetTop()),
                                static_cast<FeatStd::Int16>(page.PageSize.GetX() - editingCellBounds.GetLeft() - editingCellBounds.GetWidth()),
                                static_cast<FeatStd::Int16>(page.PageSize.GetY() - editingCellBounds.GetTop() - editingCellBounds.GetHeight()));
         Candera::Layouter::SetMargin(*(pageEdit.GetEditingCellsNode()), margin);

         Candera::Layouter::SetSize(*(pageEdit.GetEditingCellsNode()), editingCellBounds.GetSize());

         Candera::Layouter::InvalidateLayout(pageEdit.GetEditingCellsNode());
      }
      else
      {
         pageEdit.GetEditingCellsNode()->SetRenderingEnabled(false);
      }
   }
}


/*****************************************************************************/
bool DefaultPageEditController2D::OnEvent(DelegateWidget& delegateWidget, const hmibase::widget::WidgetControllerEvent& e)
{
   PageEditWidget2D* pageEdit = Candera::Dynamic_Cast<PageEditWidget2D*>(&delegateWidget);
   DefaultPageEditController2DData* pageEditData = Candera::Dynamic_Cast<DefaultPageEditController2DData*>(delegateWidget.GetControllerData());

   if ((pageEdit != NULL) && (pageEditData != NULL))
   {
      //const hmibase::widget::gizmo::EditUpdEvent* itemEditUpd = Candera::Dynamic_Cast<const hmibase::widget::gizmo::EditUpdEvent*>(&e);
      //if (itemEditUpd != NULL)
      //{
      //   return OnItemEditUpdEvent(*pageEdit, *pageEditData, *itemEditUpd);
      //}

      //const ExternalItemEditUpdEvent* externalItemEditUpd = Candera::Dynamic_Cast<const ExternalItemEditUpdEvent*>(&e);
      //if (externalItemEditUpd != NULL)
      //{
      //   return OnExternalItemEditUpdEvent(*pageEdit, *pageEditData, *externalItemEditUpd);
      //}
   }

   return Base::OnEvent(delegateWidget, e);
}


}
}


}
