/* ***************************************************************************************
* FILE:          PageEditCell.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditCell.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#pragma once

#include <Trace/IOSExtensions.h>

namespace hmibase {
namespace widget {
namespace pageedit {


typedef std::string PageIdType;


/*****************************************************************************/
struct CellRegion
{
   enum ENUM { Undef, HLeft, HCenter, HRight, VTop, VCenter, VBottom };

   ENUM Horizontal;
   ENUM Vertical;

   CellRegion(ENUM h = Undef, ENUM v = Undef) : Horizontal(h), Vertical(v)
   {
   }
};


/*****************************************************************************/
typedef int CellIndexType;
typedef int ItemIndexType;
typedef int PageIndexType;


/*****************************************************************************/
struct Cell
{
   CellIndexType Column;
   CellIndexType Row;

   Cell() : Column(0), Row(0)
   {
   }

   Cell(CellIndexType column, CellIndexType row) : Column(column), Row(row)
   {
   }

   bool operator ==(const Cell& other) const
   {
      return (Column == other.Column) && (Row == other.Row);
   }

   bool operator !=(const Cell& other) const
   {
      return !operator==(other);
   }

   Cell operator +(const Cell& other) const
   {
      return Cell(Column + other.Column, Row + other.Row);
   }

   Cell& operator +=(const Cell& other)
   {
      Column += other.Column;
      Row += other.Row;
      return *this;
   }

   Cell operator -(const Cell& other) const
   {
      return Cell(Column - other.Column, Row - other.Row);
   }

   Cell& operator -=(const Cell& other)
   {
      Column -= other.Column;
      Row -= other.Row;
      return *this;
   }

   bool IsZero() const
   {
      return (Column == 0) && (Row == 0);
   }
};


HMIBASE_STREAM_INSERTION_OP(Cell)


/*****************************************************************************/
struct CellRect
{
   Cell TopLeft;//position
   Cell Span;//size

   CellRect()
   {
   }

   CellRect(const Cell& topLeft, const Cell& span = Cell()) : TopLeft(topLeft), Span(span)
   {
   }

   CellRect(CellIndexType column, CellIndexType row, CellIndexType columnSpan, CellIndexType rowSpan) : TopLeft(column, row), Span(columnSpan, rowSpan)
   {
   }

   bool operator ==(const CellRect& other) const
   {
      return (TopLeft == other.TopLeft) && (Span == other.Span);
   }

   bool operator !=(const CellRect& other) const
   {
      return !operator==(other);
   }

   bool Intersects(const CellRect& other) const
   {
      return (((TopLeft.Column <= other.TopLeft.Column) && (other.TopLeft.Column < TopLeft.Column + Span.Column))
              || ((other.TopLeft.Column <= TopLeft.Column) && (TopLeft.Column < other.TopLeft.Column + other.Span.Column)))
             && (((TopLeft.Row <= other.TopLeft.Row) && (other.TopLeft.Row < TopLeft.Row + Span.Row))
                 || ((other.TopLeft.Row <= TopLeft.Row) && (TopLeft.Row < other.TopLeft.Row + other.Span.Row)));
   }

   bool Contains(const CellRect& other) const
   {
      return Contains(other.TopLeft) && Contains(other.GetBottomRight());
   }

   bool Contains(const Cell& cell) const
   {
      return (TopLeft.Column <= cell.Column) && (cell.Column < TopLeft.Column + Span.Column)
             && (TopLeft.Row <= cell.Row) && (cell.Row < TopLeft.Row + Span.Row);
   }

   Cell GetBottomRight() const
   {
      return TopLeft + Span - Cell(1, 1);
   }
};


HMIBASE_STREAM_INSERTION_OP(CellRect)


/*****************************************************************************/
struct PageItemIndex
{
   explicit PageItemIndex(PageIndexType pageId = -1, ItemIndexType itemIndex = -1) : PageIndex(pageId), ItemIndex(itemIndex)
   {
   }

   bool operator ==(const PageItemIndex& other) const
   {
      return (PageIndex == other.PageIndex) && (ItemIndex == other.ItemIndex);
   }

   bool operator !=(const PageItemIndex& other) const
   {
      return !operator==(other);
   }

   bool operator <(const PageItemIndex& other) const
   {
      return (PageIndex < other.PageIndex) || ((PageIndex == other.PageIndex) && (ItemIndex < other.ItemIndex));
   }

   PageIndexType PageIndex;
   ItemIndexType ItemIndex;

   static const PageItemIndex Default;
};


HMIBASE_STREAM_INSERTION_OP(PageItemIndex)


/*****************************************************************************/
typedef std::vector<CellIndexType> CellIndexVector;
typedef std::vector<Cell> CellVector;
//typedef std::map<ItemIndexType, Cell> CellMap;
typedef std::vector<CellRect> CellRectVector;
typedef std::map<PageItemIndex, CellRect> CellRectMap;


/*****************************************************************************/
template <typename TCellData, const TCellData& DefaultCellData = TCellData() >
class SimpleGrid
{
   public:
      typedef TCellData CellDataType;

      /*****************************************************************************/
      SimpleGrid(CellIndexType columnCount = 0, CellIndexType rowCount = 0) : _columnCount(0), _rowCount(0), _cellData(NULL)
      {
         Init(columnCount, rowCount);
      }

      /*****************************************************************************/
      ~SimpleGrid()
      {
         if (_cellData != NULL)
         {
            FEATSTD_DELETE_ARRAY(_cellData);
            _cellData = NULL;
         }
      }

      /*****************************************************************************/
      SimpleGrid(const SimpleGrid& other) : _columnCount(0), _rowCount(0), _cellData(NULL)
      {
         Init(other.GetColumnCount(), other.GetRowCount(), &other);
      }

      /*****************************************************************************/
      SimpleGrid& operator =(const SimpleGrid& other)
      {
         if (this != &other)
         {
            Init(other.GetColumnCount(), other.GetRowCount(), &other);
         }
         return *this;
      }

      /*****************************************************************************/
      void Init(CellIndexType columnCount, CellIndexType rowCount)
      {
         Init(columnCount, rowCount, NULL);
      }

      /*****************************************************************************/
      CellIndexType GetColumnCount() const
      {
         return _columnCount;
      }

      /*****************************************************************************/
      CellIndexType GetRowCount() const
      {
         return _rowCount;
      }

      /*****************************************************************************/
      Cell GetCount() const
      {
         return Cell(_columnCount, _rowCount);
      }

      /*****************************************************************************/
      TCellData GetCellData(CellIndexType column, CellIndexType row) const
      {
         return GetCellRef(column, row);
      }

      /*****************************************************************************/
      TCellData GetCellData(const Cell& cell) const
      {
         return GetCellData(cell.Column, cell.Row);
      }

      /*****************************************************************************/
      Cell FindTopLeftCell(const TCellData& cellData) const
      {
         if (cellData != DefaultCellData)
         {
            for (CellIndexType column = 0; column < _columnCount; ++column)
            {
               for (CellIndexType row = 0; row < _rowCount; ++row)
               {
                  if (GetCellData(column, row) == cellData)
                  {
                     return Cell(column, row);
                  }
               }
            }
         }
         return Cell(-1, -1);
      }

      /*****************************************************************************/
      Cell GetCellSpan(const Cell& topLeftCell) const
      {
         Cell cellSpan(0, 0);
         TCellData cellData = GetCellData(topLeftCell);
         if (cellData != DefaultCellData)
         {
            CellIndexType column;
            for (column = topLeftCell.Column + 1; column < _columnCount; ++column)
            {
               if (GetCellData(column, topLeftCell.Row) != cellData)
               {
                  break;
               }
            }
            cellSpan.Column = column - topLeftCell.Column;

            CellIndexType row;
            for (row = topLeftCell.Row + 1; row < _rowCount; ++row)
            {
               if (GetCellData(topLeftCell.Column, row) != cellData)
               {
                  break;
               }
            }
            cellSpan.Row = row - topLeftCell.Row;
         }
         return cellSpan;
      }

      /*****************************************************************************/
      CellRect GetCellRect(const TCellData& cellData) const
      {
         CellRect itemRect;
         itemRect.TopLeft = FindTopLeftCell(cellData);
         itemRect.Span = GetCellSpan(itemRect.TopLeft);
         return itemRect;
      }

      /*****************************************************************************/
      void SetCellData(CellIndexType column, CellIndexType row, const TCellData& cellData)
      {
         if (ContainsCell(column, row))
         {
            GetCellRef(column, row) = cellData;
         }
      }

      /*****************************************************************************/
      void SetCellData(const Cell& cell, const TCellData& cellData)
      {
         SetCellData(cell.Column, cell.Row, cellData);
      }

      /*****************************************************************************/
      void SetCellData(const CellRect& cellRect, const TCellData& cellData)
      {
         for (CellIndexType column = 0; column < cellRect.Span.Column; ++column)
         {
            for (CellIndexType row = 0; row < cellRect.Span.Row; ++row)
            {
               SetCellData(cellRect.TopLeft.Column + column, cellRect.TopLeft.Row + row, cellData);
            }
         }
      }

      /*****************************************************************************/
      void ResetCellData(CellIndexType column, CellIndexType row)
      {
         SetCellData(column, row, DefaultCellData);
      }

      /*****************************************************************************/
      void ResetCellData(const Cell& cell)
      {
         SetCellData(cell.Column, cell.Row, DefaultCellData);
      }

      /*****************************************************************************/
      void ResetCellData(const CellRect& cellRect)
      {
         SetCellData(cellRect, DefaultCellData);
      }

      /*****************************************************************************/
      bool ContainsCell(CellIndexType column, CellIndexType row) const
      {
         return (_cellData != NULL) && (column >= 0) && (column < _columnCount) && (row >= 0) && (row < _rowCount);
      }

      /*****************************************************************************/
      bool ContainsCell(const Cell& cell) const
      {
         return ContainsCell(cell.Column, cell.Row);
      }

      /*****************************************************************************/
      bool ContainsCellRect(const CellRect& cellRect) const
      {
         return ContainsCell(cellRect.TopLeft) && ContainsCell(cellRect.GetBottomRight());
      }

      /*****************************************************************************/
      bool IsCellEmpty(CellIndexType column, CellIndexType row) const
      {
         return ContainsCell(column, row) && (GetCellRef(column, row) == DefaultCellData);
      }

      /*****************************************************************************/
      bool IsCellEmpty(const Cell& cell) const
      {
         return IsCellEmpty(cell.Column, cell.Row);
      }

      /*****************************************************************************/
      bool IsCellRectEmpty(const CellRect& cellRect) const
      {
         if (!ContainsCellRect(cellRect))
         {
            return false;
         }
         for (CellIndexType column = 0; column < cellRect.Span.Column; ++column)
         {
            for (CellIndexType row = 0; row < cellRect.Span.Row; ++row)
            {
               if (!IsCellEmpty(cellRect.TopLeft.Column + column, cellRect.TopLeft.Row + row))
               {
                  return false;
               }
            }
         }
         return true;
      }

      /*****************************************************************************/
      void SwapCells(const Cell& cell1, const Cell& cell2)
      {
         if ((cell1 != cell2) && ContainsCell(cell1) && ContainsCell(cell2))
         {
            TCellData temp = GetCellData(cell1);
            SetCellData(cell1, GetCellData(cell2));
            SetCellData(cell2, temp);
         }
      }

   private:
      /*****************************************************************************/
      void Init(CellIndexType columnCount, CellIndexType rowCount, const SimpleGrid* other)
      {
         if (columnCount < 1)
         {
            columnCount = 1;
         }
         if (rowCount < 1)
         {
            rowCount = 1;
         }

         if ((_columnCount != columnCount) || (_rowCount != rowCount))
         {
            _columnCount = columnCount;
            _rowCount = rowCount;

            if (_cellData != NULL)
            {
               FEATSTD_DELETE_ARRAY(_cellData);
               _cellData = NULL;
            }
         }
         if (_cellData == NULL)
         {
            _cellData = FEATSTD_NEW_ARRAY(TCellData, _columnCount * _rowCount);
         }

         for (CellIndexType column = 0; column < _columnCount; ++column)
         {
            for (CellIndexType row = 0; row < _rowCount; ++row)
            {
               SetCellData(column, row, (other != NULL) ? other->GetCellData(column, row) : DefaultCellData);
            }
         }
      }

      /*****************************************************************************/
      const TCellData& GetCellRef(CellIndexType column, CellIndexType row) const
      {
         static TCellData _invalid;
         return ContainsCell(column, row) && (_cellData != NULL) ? _cellData[GetCellIndex(column, row)] : (_invalid = DefaultCellData);
      }

      /*****************************************************************************/
      TCellData& GetCellRef(CellIndexType column, CellIndexType row)
      {
         static TCellData _invalid;
         return ContainsCell(column, row) && (_cellData != NULL) ? _cellData[GetCellIndex(column, row)] : (_invalid = DefaultCellData);
      }

      /*****************************************************************************/
      size_t GetCellIndex(CellIndexType column, CellIndexType row) const
      {
         return static_cast<size_t>(row * _columnCount + column);
      }

      /*****************************************************************************/
      CellIndexType _columnCount;
      CellIndexType _rowCount;
      TCellData* _cellData;
};


typedef SimpleGrid < PageItemIndex, PageItemIndex::Default > CellGrid;
HMIBASE_STREAM_INSERTION_OP(CellGrid)
}


}
}
