/* ***************************************************************************************
* FILE:          OffscreenTouchProxyWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OffscreenTouchProxyWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2017 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "OffscreenTouchProxyWidget2D.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include "Courier/Visualization/ViewScene2D.h"
#include <Courier/Visualization/IViewHandler.h>
#include "BaseContract/generated/BaseTouchSessionMsgs.h"

#include "View/CGI/CgiExtensions/ViewScene2D.h"

#include "hmibase/util/ItemRegistry.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TOUCHABLE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OffscreenTouchProxyWidget2D.cpp.trc.h"
#endif

CGI_WIDGET_RTTI_DEFINITION(OffscreenTouchProxyWidget2D)

OffscreenTouchProxyWidget2D::OffscreenTouchProxyWidget2D() : _viewScene(0)
{
}


OffscreenTouchProxyWidget2D::~OffscreenTouchProxyWidget2D()
{
   const Candera::Node2D* offScreenNode = GetOffscreenScene();
   if (offScreenNode)
   {
      Candera::Scene2D* offScreenScene = offScreenNode->GetScene();
      if (offScreenScene != 0)
      {
         Courier::ViewId viewId(offScreenScene->GetName());

         if (OffscreenTouchOffset::isItemRegistered(viewId))
         {
            OffscreenTouchOffset::unregisterItem(viewId);
         }
      }
   }
}


bool OffscreenTouchProxyWidget2D::OnMessage(const ::Courier::Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case hmibase::input::TouchSessionStartEvent::ID:
      {
         const hmibase::input::TouchSessionStartEvent* pStartEvent = Courier::message_cast<const hmibase::input::TouchSessionStartEvent*>(&msg);

         if (pStartEvent != 0)
         {
            const Courier::TouchInfo& touchInfo = pStartEvent->GetTouchInfo();

            if (IsInsideBoundingRect(touchInfo) && IsTouchable() && IsEnabled())
            {
               OnTouchSessionStartEvent(*pStartEvent);
            }
         }
      }
      break;

      case hmibase::input::TouchSessionStopEvent::ID:
      {
         const hmibase::input::TouchSessionStopEvent* pStopEvent = Courier::message_cast<const hmibase::input::TouchSessionStopEvent*>(&msg);
         if (pStopEvent != 0)
         {
            OnTouchSessionStopEvent(*pStopEvent);
         }
      }
      break;

      case Courier::TouchMsg::ID:
      {
         const Courier::TouchMsg* touchMsg = Courier::message_cast<const Courier::TouchMsg*>(&msg);
         if (touchMsg != NULL)
         {
            consumed = OnTouchMessage(*touchMsg);
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


bool OffscreenTouchProxyWidget2D::OnTouchSessionStartEvent(const hmibase::input::TouchSessionStartEvent& msg)
{
   const Candera::Node2D* node = GetNode();
   const Candera::Node2D* offScreenNode = GetOffscreenScene();
   const Candera::Camera2D* camera = GetCamera();

   if ((node != 0) && (offScreenNode != 0) && (camera != 0))
   {
      const Candera::RenderTarget2D* renderTarget = camera->GetRenderTarget();
      Candera::Scene2D* offScreenScene = offScreenNode->GetScene();
      if ((offScreenScene != 0) && (renderTarget != 0))
      {
         Courier::ViewId viewId(offScreenScene->GetName());

         Courier::View* parentView = GetParentView();
         if (parentView != 0)
         {
            Courier::IViewHandler* viewHandler = parentView->GetViewHandler();
            if (viewHandler != 0)
            {
               Courier::View* view = viewHandler->FindView(viewId);
               if (view != 0)
               {
                  _viewScene = dynamic_cast<hmibase::view::ViewScene2D*>(view->ToViewScene2D());

                  if (_viewScene)
                  {
                     // don't process TouchSessionStart/Stop events in this offscreen view
                     // otherwise it could happen that there are several button in different offscreens pressed at the same time
                     _viewScene->ProcessTouchSessionEventsInOffscreen(false);
                  }

                  Candera::Vector2 viewportPos = Candera::Math2D::TransformSceneToViewport(*camera, node->GetWorldPosition());
                  Candera::Vector2 rendertargetPos = Candera::Math2D::TransformViewportToRenderTarget(*camera, viewportPos);
                  Candera::Vector2 screenPos = Candera::Math2D::TransformRenderTargetToScreen(*renderTarget, rendertargetPos);
                  _touchOffset = -screenPos;

                  OffscreenTouchOffset::registerItem(viewId, _touchOffset);

                  ForwardTouchSessionEvent(msg);
               }
            }
         }
      }
   }

   return false;
}


bool OffscreenTouchProxyWidget2D::OnTouchSessionStopEvent(const hmibase::input::TouchSessionStopEvent& msg)
{
   const Candera::Node2D* offScreenNode = GetOffscreenScene();
   if (offScreenNode)
   {
      Candera::Scene2D* offScreenScene = offScreenNode->GetScene();
      if (offScreenScene != 0)
      {
         Courier::ViewId viewId(offScreenScene->GetName());

         if (OffscreenTouchOffset::isItemRegistered(viewId))
         {
            OffscreenTouchOffset::unregisterItem(viewId);
         }
      }
   }

   ForwardTouchSessionEvent(msg);
   return false;
}


bool OffscreenTouchProxyWidget2D::OnTouchMsg(const Courier::TouchMsg& msg)
{
   bool consumed = false;

   if (_viewScene)
   {
      FeatStd::SizeType posX = msg.GetXPos();
      FeatStd::SizeType posY = msg.GetYPos();
      CalcOffscreenTouchPos(posX, posY);

      Courier::TouchMsg proxyMsg(msg.GetState(), static_cast<Courier::XYDIM>(posX), static_cast<Courier::XYDIM>(posY), msg.GetTimeStamp(), msg.GetPointerId(), msg.GetSourceId());
      consumed = _viewScene->DistributeMessage(proxyMsg);
   }

   return consumed;
}


template <typename T>
void OffscreenTouchProxyWidget2D::ForwardTouchSessionEvent(const T& msg)
{
   if (_viewScene != 0)
   {
      Courier::TouchInfo touchInfo = msg.GetTouchInfo();
      FeatStd::SizeType posX = touchInfo.mX;
      FeatStd::SizeType posY = touchInfo.mY;
      CalcOffscreenTouchPos(posX, posY);

      hmibase::input::TouchSession& rTouchSession = const_cast<hmibase::input::TouchSession&>(msg.GetTouchSession().Get());
      const ::hmibase::input::TouchSession::Ref touchSessionRef(rTouchSession);

      touchInfo.mX = static_cast<FeatStd::Int32>(posX);
      touchInfo.mY = static_cast<FeatStd::Int32>(posY);
      _viewScene->DistributeMessageDirect(T(touchSessionRef, touchInfo));
   }
}


void OffscreenTouchProxyWidget2D::CalcOffscreenTouchPos(FeatStd::SizeType& posX, FeatStd::SizeType& posY) const
{
   posX = FeatStd::SizeType(posX + static_cast<FeatStd::SizeType>(_touchOffset.GetX()));
   posY = FeatStd::SizeType(posY + static_cast<FeatStd::SizeType>(_touchOffset.GetY()));
}
